package jaxx.runtime.swing.editor;

/*
 * #%L
 * JAXX :: Widgets
 * %%
 * Copyright (C) 2008 - 2014 Code Lutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import jaxx.runtime.spi.UIHandler;
import jaxx.runtime.swing.editor.gis.DmdCoordinate;
import jaxx.runtime.swing.editor.gis.DmdCoordinateConverter;
import jaxx.runtime.swing.editor.gis.MaskFormatterFromConverter;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.beans.BeanUtil;

import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JFormattedTextField;
import javax.swing.SwingUtilities;
import javax.swing.text.BadLocationException;
import javax.swing.text.DefaultFormatterFactory;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.lang.reflect.Method;
import java.text.ParseException;

/**
 * Created on 2/11/14.
 *
 * @author Tony Chemit - chemit@codelutin.com
 * @since 2.8.1
 */
public class NumberEditor2Handler implements UIHandler<NumberEditor2> {

    /** Logger. */
    private static final Log log = LogFactory.getLog(NumberEditor2Handler.class);

    protected NumberEditor2 ui;

    protected JFormattedTextField editor;

    protected DmdCoordinateConverter signedConverter;

    protected DmdCoordinateConverter unsignedConverter;

    protected DefaultFormatterFactory signedFormatterFactory;

    protected DefaultFormatterFactory unsignedFormatterFactory;

    /** the mutator method on the property of boxed bean in the editor */
    protected Method mutator;

    /** the getter method on the property */
    protected Method getter;

    /** a flag to known if mutator accept null value */
    protected Boolean acceptNull;

    protected NumberEditor2Model model;

    protected String lastValidText;

    @Override
    public void beforeInit(NumberEditor2 ui) {
        this.ui = ui;
    }

    @Override
    public void afterInit(NumberEditor2 ui) {
        this.model = ui.getModel();
        this.editor = ui.getEditor();

    }

    public void init() {

        Preconditions.checkNotNull(model.getBean(), "could not find bean in " + model);
        Preconditions.checkNotNull(model.getBeanProperty(), "could not find beanProperty in " + model);
        Preconditions.checkNotNull(model.getNumberType(), "could not find numberType in " + model);

        Object bean = model.getBean();
        mutator = BeanUtil.getMutator(bean, model.getBeanProperty());
        Preconditions.checkNotNull(mutator, "could not find mutator for " + model.getBeanProperty());

        {
            // prepare unsigned formatter factory
            String pattern = getMaskFormatterPattern(false);
            MaskFormatterFromConverter<DmdCoordinate> maskFormatter;
            try {
                maskFormatter = MaskFormatterFromConverter.newFormatter(
                        DmdCoordinate.class,
                        pattern, unsignedConverter);
                maskFormatter.setValidCharacters(" 01234567890.");
                maskFormatter.setCommitsOnValidEdit(true);
                unsignedFormatterFactory = new DefaultFormatterFactory(maskFormatter);
            } catch (ParseException e) {
                // can't happen here
                throw new RuntimeException(e);
            }
        }
        {
            // prepare signed formatter factory
            String pattern = getMaskFormatterPattern(true);
            MaskFormatterFromConverter<DmdCoordinate> maskFormatter;
            try {
                maskFormatter = MaskFormatterFromConverter.newFormatter(
                        DmdCoordinate.class,
                        pattern, signedConverter);
                maskFormatter.setValidCharacters(" 01234567890.");
                maskFormatter.setCommitsOnValidEdit(true);
                signedFormatterFactory = new DefaultFormatterFactory(maskFormatter);
            } catch (ParseException e) {
                // can't happen here
                throw new RuntimeException(e);
            }
        }

//        JFormattedTextField editor = ui.getEditor();
//        editor.setFormatterFactory(model.isSign() ?
//                                   signedFormatterFactory :
//                                   unsignedFormatterFactory);
//        editor.setFocusLostBehavior(JFormattedTextField.COMMIT);
//
//        // When editor changes his value, propagate it to model
//        editor.addPropertyChangeListener("value", new PropertyChangeListener() {
//            @Override
//            public void propertyChange(PropertyChangeEvent evt) {
//                DmdCoordinate newValue = (DmdCoordinate) evt.getNewValue();
//                if (log.isDebugEnabled()) {
//                    log.debug("Value has changed: " + newValue);
//                }
//                model.setValue(newValue);
//            }
//        });
    }

    public void setModel(Object model) {

    }

    /**
     * Ajoute le caractère donné à l'endroit où est le curseur dans la zone de
     * saisie et met à jour le modèle.
     *
     * @param s le caractère à ajouter.
     */
    void addChar(String s) {
        char c = s.charAt(0);
        try {
            editor.getDocument().insertString(editor.getCaretPosition(), c + "", null);
//            setModel(editor.getText());
            //setModel(editor.getModelText() + c);

        } catch (BadLocationException e) {
            log.warn(e);
        }
    }

    /**
     * Supprime le caractère juste avant le curseur du modèle (textuel) et
     * met à jour la zone de saisie.
     */
    public void removeChar() {
//        String s = editor.getModelText();
//        int position = editor.getCaretPosition();
//        if (position < 1 || s.isEmpty()) {
//            if (log.isDebugEnabled()) {
//                log.debug("cannot remove when caret on first position or text empty");
//            }
//            // on est au debut du doc, donc rien a faire
//            return;
//        }
//        try {
//            editor.getDocument().remove(position - 1, 1);
//        } catch (BadLocationException ex) {
//            // ne devrait jamais arrive vu qu'on a fait le controle...
//            log.debug(ex);
//            return;
//        }
//        String newText = editor.getText();
//        if (log.isDebugEnabled()) {
//            log.debug("text updated : " + newText);
//        }
//        position--;
//        editor.setCaretPosition(position);
//        setModel(newText);
    }

    /**
     * Permute le signe dans la zone de saisie et
     * dans le modèle.
     */
    public void toggleSign() {
//        // change sign
//        String newValue = editor.getModelText();
//
//        if (newValue.startsWith("-")) {
//            setModel(newValue.substring(1));
//        } else {
//            setModel("-" + newValue);
//        }
    }

    void addChar(ActionEvent event) {
        String text = ((JButton) event.getSource()).getText();
        addChar(text);
    }

    /**
     * Affiche ou cache la popup.
     *
     * @param newValue la nouvelle valeur de visibilité de la popup.
     */
    public void setPopupVisible(Boolean newValue) {

        if (log.isTraceEnabled()) {
            log.trace(newValue);
        }

        if (newValue == null || !newValue) {
            ui.getPopup().setVisible(false);
            return;
        }
        SwingUtilities.invokeLater(new Runnable() {

            @Override
            public void run() {
                JComponent invoker =
                        model.isShowPopup() ?
                        ui.getShowPopUpButton() :
                        ui;
                Dimension dim = ui.getPopup().getPreferredSize();
                int x = (int) (invoker.getPreferredSize().getWidth() - dim.getWidth());
                ui.getPopup().show(invoker, x, invoker.getHeight());
                ui.getEditor().requestFocus();
            }
        });
    }

    protected String getMaskFormatterPattern(boolean useSign) {
        String pattern = "############";
        if (useSign) {
            pattern = "-" + pattern;
        }
        return pattern;
    }

    public void validate() {
        //TODO
    }

    public void resetEditor() {
        //TODO
    }

    public void onKeyReleased(KeyEvent event) {
        //TODO
    }

    public void setValue(Number value, boolean b) {
        //TODO
    }

    protected class PopupListener extends MouseAdapter {

        @Override
        public void mousePressed(MouseEvent e) {
            maybeShowPopup(e);
        }

        @Override
        public void mouseReleased(MouseEvent e) {
            maybeShowPopup(e);
        }

        protected void maybeShowPopup(MouseEvent e) {
            if (!e.isPopupTrigger()) {
                return;
            }
            if (model.isAutoPopup()) {
                if (model.isPopupVisible()) {
                    if (!ui.getPopup().isVisible()) {
                        setPopupVisible(true);
                    }
                    // popup already visible

                } else {
                    // set popup auto
                    model.setPopupVisible(true);

                }
            } else {
                if (model.isPopupVisible()) {
                    setPopupVisible(true);
                }
            }
        }

    }
}
