/*
 * Copyright MapStruct Authors.
 *
 * Licensed under the Apache License version 2.0, available at http://www.apache.org/licenses/LICENSE-2.0
 */
package org.mapstruct.ap.internal.util;

import java.util.HashSet;
import java.util.Set;

import javax.lang.model.type.TypeMirror;

import org.mapstruct.ap.spi.AstModifyingAnnotationProcessor;

/**
 * Keeps contextual data in the scope of one annotation processing round.
 *
 * @author Gunnar Morling
 */
public class RoundContext {

    private final AnnotationProcessorContext annotationProcessorContext;
    private final Set<TypeMirror> clearedTypes;

    public RoundContext(AnnotationProcessorContext annotationProcessorContext) {
        this.annotationProcessorContext = annotationProcessorContext;
        this.clearedTypes = new HashSet<>();
    }

    public AnnotationProcessorContext getAnnotationProcessorContext() {
        return annotationProcessorContext;
    }

    /**
     * Marks the given type as being ready for further processing.
     * @param type the type that is ready for further processing by MapStruct
     */
    public void addTypeReadyForProcessing(TypeMirror type) {
        clearedTypes.add( type );
    }

    /**
     * Whether the given type has been found to be ready for further processing or not. This is the case if the type's
     * hierarchy is complete (no super-types need to be generated by other processors) and no processors have signaled
     * the intention to amend the given type.
     *
     * @param type the typed to be checked for its readiness
     * @return true when the type is ready to be processed by MapStruct
     *
     * @see AstModifyingAnnotationProcessor
     */
    public boolean isReadyForProcessing(TypeMirror type) {
        return clearedTypes.contains( type );
    }
}
