/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.authentication.authenticators.client;

import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.MultivaluedMap;
import jakarta.ws.rs.core.Response;
import org.jboss.logging.Logger;
import org.keycloak.authentication.AuthenticationFlowError;
import org.keycloak.authentication.ClientAuthenticationFlowContext;
import org.keycloak.authentication.authenticators.client.ClientAuthUtil;
import org.keycloak.common.util.Time;
import org.keycloak.http.HttpRequest;
import org.keycloak.jose.jws.JWSInput;
import org.keycloak.jose.jws.JWSInputException;
import org.keycloak.models.ClientModel;
import org.keycloak.models.RealmModel;
import org.keycloak.models.SingleUseObjectProvider;
import org.keycloak.protocol.oidc.OIDCAdvancedConfigWrapper;
import org.keycloak.representations.JsonWebToken;

public class JWTClientValidator {
    private static final Logger logger = Logger.getLogger(JWTClientValidator.class);
    private final ClientAuthenticationFlowContext context;
    private final RealmModel realm;
    private final int currentTime;
    private final String clientAuthenticatorProviderId;
    private MultivaluedMap<String, String> params;
    private String clientAssertion;
    private JWSInput jws;
    private JsonWebToken token;
    private ClientModel client;
    private static final int ALLOWED_CLOCK_SKEW = 15;

    public JWTClientValidator(ClientAuthenticationFlowContext context, String clientAuthenticatorProviderId) {
        this.context = context;
        this.realm = context.getRealm();
        this.currentTime = Time.currentTime();
        this.clientAuthenticatorProviderId = clientAuthenticatorProviderId;
    }

    public boolean clientAssertionParametersValidation() {
        if (!this.isFormDataRequest(this.context.getHttpRequest())) {
            Response challengeResponse = ClientAuthUtil.errorResponse(Response.Status.BAD_REQUEST.getStatusCode(), "invalid_client", "Parameter client_assertion_type is missing");
            this.context.challenge(challengeResponse);
            return false;
        }
        this.params = this.context.getHttpRequest().getDecodedFormParameters();
        String clientAssertionType = (String)this.params.getFirst((Object)"client_assertion_type");
        this.clientAssertion = (String)this.params.getFirst((Object)"client_assertion");
        if (clientAssertionType == null) {
            Response challengeResponse = ClientAuthUtil.errorResponse(Response.Status.BAD_REQUEST.getStatusCode(), "invalid_client", "Parameter client_assertion_type is missing");
            this.context.challenge(challengeResponse);
            return false;
        }
        if (!clientAssertionType.equals("urn:ietf:params:oauth:client-assertion-type:jwt-bearer")) {
            Response challengeResponse = ClientAuthUtil.errorResponse(Response.Status.BAD_REQUEST.getStatusCode(), "invalid_client", "Parameter client_assertion_type has value '" + clientAssertionType + "' but expected is 'urn:ietf:params:oauth:client-assertion-type:jwt-bearer'");
            this.context.challenge(challengeResponse);
            return false;
        }
        if (this.clientAssertion == null) {
            Response challengeResponse = ClientAuthUtil.errorResponse(Response.Status.BAD_REQUEST.getStatusCode(), "invalid_client", "client_assertion parameter missing");
            this.context.failure(AuthenticationFlowError.INVALID_CLIENT_CREDENTIALS, challengeResponse);
            return false;
        }
        return true;
    }

    public void readJws() throws JWSInputException {
        if (this.clientAssertion == null) {
            throw new IllegalStateException("Incorrect usage. Variable 'clientAssertion' is null. Need to validate clientAssertion first before read JWS");
        }
        this.jws = new JWSInput(this.clientAssertion);
        this.token = (JsonWebToken)this.jws.readJsonContent(JsonWebToken.class);
    }

    public boolean validateClient() {
        if (this.token == null) {
            throw new IllegalStateException("Incorrect usage. Variable 'token' is null. Need to read JWS first before validateClient");
        }
        String clientId = this.token.getSubject();
        if (clientId == null) {
            throw new RuntimeException("Can't identify client. Subject missing on JWT token");
        }
        if (!clientId.equals(this.token.getIssuer())) {
            throw new RuntimeException("Issuer mismatch. The issuer should match the subject");
        }
        String clientIdParam = (String)this.params.getFirst((Object)"client_id");
        if (clientIdParam != null && !clientIdParam.equals(clientId)) {
            throw new RuntimeException("client_id parameter not matching with client from JWT token");
        }
        this.context.getEvent().client(clientId);
        this.client = this.realm.getClientByClientId(clientId);
        if (this.client == null) {
            this.context.failure(AuthenticationFlowError.CLIENT_NOT_FOUND, null);
            return false;
        }
        this.context.setClient(this.client);
        if (!this.client.isEnabled()) {
            this.context.failure(AuthenticationFlowError.CLIENT_DISABLED, null);
            return false;
        }
        if (!this.clientAuthenticatorProviderId.equals(this.client.getClientAuthenticatorType())) {
            this.context.failure(AuthenticationFlowError.INVALID_CLIENT_CREDENTIALS, null);
            return false;
        }
        return true;
    }

    public boolean validateSignatureAlgorithm() {
        if (this.jws == null) {
            throw new IllegalStateException("Incorrect usage. Variable 'jws' is null. Need to read token first before validate signature algorithm");
        }
        if (this.client == null) {
            throw new IllegalStateException("Incorrect usage. Variable 'client' is null. Need to validate client first before validate signature algorithm");
        }
        String expectedSignatureAlg = OIDCAdvancedConfigWrapper.fromClientModel(this.client).getTokenEndpointAuthSigningAlg();
        if (this.jws.getHeader().getAlgorithm() == null || this.jws.getHeader().getAlgorithm().name() == null) {
            Response challengeResponse = ClientAuthUtil.errorResponse(Response.Status.BAD_REQUEST.getStatusCode(), "invalid_client", "invalid signature algorithm");
            this.context.challenge(challengeResponse);
            return false;
        }
        String actualSignatureAlg = this.jws.getHeader().getAlgorithm().name();
        if (expectedSignatureAlg != null && !expectedSignatureAlg.equals(actualSignatureAlg)) {
            Response challengeResponse = ClientAuthUtil.errorResponse(Response.Status.BAD_REQUEST.getStatusCode(), "invalid_client", "invalid signature algorithm");
            this.context.challenge(challengeResponse);
            return false;
        }
        return true;
    }

    public void validateToken() {
        if (this.token == null) {
            throw new IllegalStateException("Incorrect usage. Variable 'token' is null. Need to read token first before validateToken");
        }
        if (!this.token.isActive(15)) {
            throw new RuntimeException("Token is not active");
        }
        if (!(this.token.getExp() != null && this.token.getExp() > 0L || this.token.getIat() != null && this.token.getIat() > 0L)) {
            throw new RuntimeException("Token cannot be validated. Neither the exp nor the iat claim are present.");
        }
        if (this.token.getExp() == null || this.token.getExp() <= 0L) {
            if (this.token.getIat() + 15L + 10L < (long)this.currentTime) {
                throw new RuntimeException("Token is not active");
            }
        } else if (this.token.getIat() != null && this.token.getIat() > 0L && this.token.getIat() - 15L > (long)this.currentTime) {
            throw new RuntimeException("Token was issued in the future");
        }
        if (this.token.getId() == null) {
            throw new RuntimeException("Missing ID on the token");
        }
    }

    private long calculateLifespanInSeconds() {
        if (!(this.token.getExp() != null && this.token.getExp() > 0L || this.token.getIat() != null && this.token.getIat() > 0L)) {
            throw new RuntimeException("Token cannot be validated. Neither the exp nor the iat claim are present.");
        }
        if (this.token.getExp() == null || this.token.getExp() <= 0L) {
            long lifespan = this.token.getIat() + 15L + 10L - (long)this.currentTime;
            if (lifespan <= 0L) {
                throw new RuntimeException("Token is not active");
            }
            return lifespan;
        }
        if (this.token.getIat() == null || this.token.getIat() <= 0L) {
            int maxExp = OIDCAdvancedConfigWrapper.fromClientModel(this.client).getTokenEndpointAuthSigningMaxExp();
            long lifespan = this.token.getExp() - (long)this.currentTime;
            if (lifespan > (long)maxExp) {
                throw new RuntimeException("Token expiration is too far in the future and iat claim not present in token");
            }
            return lifespan;
        }
        if (this.token.getIat() - 15L > (long)this.currentTime) {
            throw new RuntimeException("Token was issued in the future");
        }
        int maxExp = OIDCAdvancedConfigWrapper.fromClientModel(this.client).getTokenEndpointAuthSigningMaxExp();
        long lifespan = Math.min(this.token.getExp() - (long)this.currentTime, (long)maxExp);
        if (lifespan <= 0L) {
            throw new RuntimeException("Token is not active");
        }
        if ((long)this.currentTime > this.token.getIat() + (long)maxExp) {
            throw new RuntimeException("Token was issued too far in the past to be used now");
        }
        return lifespan;
    }

    public void validateTokenReuse() {
        if (this.token == null) {
            throw new IllegalStateException("Incorrect usage. Variable 'token' is null. Need to read token first before validateToken reuse");
        }
        if (this.client == null) {
            throw new IllegalStateException("Incorrect usage. Variable 'client' is null. Need to validate client first before validateToken reuse");
        }
        SingleUseObjectProvider singleUseCache = this.context.getSession().singleUseObjects();
        long lifespanInSecs = this.calculateLifespanInSeconds();
        if (!singleUseCache.putIfAbsent(this.token.getId(), lifespanInSecs)) {
            logger.warnf("Token '%s' already used when authenticating client '%s'.", (Object)this.token.getId(), (Object)this.client.getClientId());
            throw new RuntimeException("Token reuse detected");
        }
        logger.tracef("Added token '%s' to single-use cache. Lifespan: %d seconds, client: %s", (Object)this.token.getId(), (Object)lifespanInSecs, (Object)this.client.getClientId());
    }

    public ClientAuthenticationFlowContext getContext() {
        return this.context;
    }

    public RealmModel getRealm() {
        return this.realm;
    }

    public MultivaluedMap<String, String> getParams() {
        return this.params;
    }

    public String getClientAssertion() {
        return this.clientAssertion;
    }

    public JWSInput getJws() {
        return this.jws;
    }

    public JsonWebToken getToken() {
        return this.token;
    }

    public ClientModel getClient() {
        return this.client;
    }

    private boolean isFormDataRequest(HttpRequest request) {
        MediaType mediaType = request.getHttpHeaders().getMediaType();
        return mediaType != null && mediaType.isCompatible(MediaType.APPLICATION_FORM_URLENCODED_TYPE);
    }
}

