/*
 * Decompiled with CFR 0.152.
 */
package org.jboss.logging.processor.validation;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import org.jboss.logging.annotations.Pos;
import org.jboss.logging.annotations.Transform;
import org.jboss.logging.processor.model.MessageInterface;
import org.jboss.logging.processor.model.MessageMethod;
import org.jboss.logging.processor.model.Parameter;
import org.jboss.logging.processor.model.ReturnType;
import org.jboss.logging.processor.model.ThrowableType;
import org.jboss.logging.processor.validation.FormatValidator;
import org.jboss.logging.processor.validation.FormatValidatorFactory;
import org.jboss.logging.processor.validation.IdLengthValidator;
import org.jboss.logging.processor.validation.IdRangeValidator;
import org.jboss.logging.processor.validation.MessageIdValidator;
import org.jboss.logging.processor.validation.ValidationMessage;
import org.jboss.logging.processor.validation.ValidationMessageFactory;

public final class Validator {
    private final MessageIdValidator messageIdValidator = new MessageIdValidator();
    private final IdLengthValidator idLengthValidator = new IdLengthValidator();
    private final IdRangeValidator idRangeValidator = new IdRangeValidator();

    public final Collection<ValidationMessage> validate(MessageInterface messageInterface) {
        ArrayList<ValidationMessage> messages = new ArrayList<ValidationMessage>();
        switch (messageInterface.getAnnotatedType()) {
            case MESSAGE_BUNDLE: {
                Set<MessageMethod> messageMethods = this.getAllMethods(messageInterface);
                messages.addAll(this.validateCommon(messageInterface, messageMethods));
                messages.addAll(this.validateBundle(messageMethods));
                break;
            }
            case MESSAGE_LOGGER: {
                Set<MessageMethod> messageMethods = this.getAllMethods(messageInterface);
                messages.addAll(this.validateCommon(messageInterface, messageMethods));
                messages.addAll(this.validateLogger(messageMethods));
                break;
            }
            default: {
                messages.add(ValidationMessageFactory.createError(messageInterface, "Message interface %s is not a message bundle or message logger.", messageInterface.name()));
            }
        }
        return messages;
    }

    private Collection<ValidationMessage> validateCommon(MessageInterface messageInterface, Set<MessageMethod> messageMethods) {
        ArrayList<ValidationMessage> messages = new ArrayList<ValidationMessage>();
        HashMap<String, MessageMethod> methodNames = new HashMap<String, MessageMethod>();
        messages.addAll(this.idLengthValidator.validate(messageInterface));
        messages.addAll(this.idRangeValidator.validate(messageInterface));
        for (MessageMethod messageMethod : messageMethods) {
            FormatValidator formatValidator;
            for (ThrowableType throwableType : messageMethod.thrownTypes()) {
                if (!throwableType.isChecked()) continue;
                messages.add(ValidationMessageFactory.createError(messageMethod, "Interface message methods cannot throw checked exceptions."));
            }
            MessageMethod.Message message = messageMethod.message();
            if (message == null) {
                messages.add(ValidationMessageFactory.createError(messageMethod, "All message bundles and message logger message methods must have or inherit a message."));
                continue;
            }
            if (message.hasId()) {
                if (message.id() < 0) {
                    messages.add(ValidationMessageFactory.createError(messageMethod, "Message id %d is invalid. Must be greater than 0 or inherit another valid id.", message.id()));
                } else {
                    messages.addAll(this.messageIdValidator.validate(messageInterface, messageMethod));
                }
            }
            if ((formatValidator = FormatValidatorFactory.create(messageMethod)).isValid()) {
                int paramCount = messageMethod.formatParameterCount();
                if (messageMethod.formatParameterCount() != formatValidator.argumentCount()) {
                    messages.add(ValidationMessageFactory.createError(messageMethod, "Parameter count does not match for format '%s'. Required: %d Provided: %d", formatValidator.format(), formatValidator.argumentCount(), paramCount));
                }
                if (!messageMethod.parameters(Parameter.ParameterType.TRANSFORM).isEmpty()) {
                    Set<Parameter> parameters = messageMethod.parameters(Parameter.ParameterType.TRANSFORM);
                    for (Parameter parameter : parameters) {
                        this.validateTransform(messages, parameter, parameter.transform());
                    }
                }
                if (!messageMethod.parameters(Parameter.ParameterType.POS).isEmpty()) {
                    TreeMap<Integer, Parameter> positions = new TreeMap<Integer, Parameter>();
                    Set<Parameter> parameters = messageMethod.parameters(Parameter.ParameterType.POS);
                    for (Parameter parameter : parameters) {
                        Pos pos = parameter.pos();
                        Transform[] transforms = pos.transform();
                        if (transforms != null && transforms.length > 0) {
                            if (pos.value().length != transforms.length) {
                                messages.add(ValidationMessageFactory.createError(parameter, "Positional parameters with transforms must have an equal number of positions and transforms."));
                            } else {
                                for (Transform transform : transforms) {
                                    this.validateTransform(messages, parameter, transform);
                                }
                            }
                        }
                        HashSet<Integer> usedPositions = new HashSet<Integer>();
                        for (int position : pos.value()) {
                            if (usedPositions.contains(position)) {
                                messages.add(ValidationMessageFactory.createError(parameter, "Position '%d' already used for this parameter.", position));
                            } else {
                                usedPositions.add(position);
                            }
                            if (positions.containsKey(position)) {
                                messages.add(ValidationMessageFactory.createError(parameter, "Position '%d' already defined on parameter '%s'", position, ((Parameter)positions.get(position)).name()));
                                continue;
                            }
                            positions.put(position, parameter);
                        }
                    }
                    for (int i = 0; i < messageMethod.formatParameterCount(); ++i) {
                        int positionIndex = i + 1;
                        if (positions.containsKey(positionIndex)) continue;
                        messages.add(ValidationMessageFactory.createError(messageMethod, "Missing parameter with position '%d' defined.", positionIndex));
                    }
                }
            } else {
                messages.add(ValidationMessageFactory.createError(messageMethod, formatValidator.summaryMessage()));
            }
            if (!messageMethod.inheritsMessage()) {
                String key = messageMethod.name() + messageMethod.formatParameterCount();
                if (methodNames.containsKey(key)) {
                    MessageMethod previousMethod = (MessageMethod)methodNames.get(key);
                    messages.add(ValidationMessageFactory.createError(previousMethod, "Only one message with the same format parameters is allowed."));
                    messages.add(ValidationMessageFactory.createError(messageMethod, "Only one message with the same format parameters is allowed."));
                } else {
                    methodNames.put(key, messageMethod);
                }
            }
            messages.addAll(this.validateParameters(messageMethod));
        }
        return messages;
    }

    private void validateTransform(List<ValidationMessage> messages, Parameter parameter, Transform transform) {
        List<Transform.TransformType> transformTypes = Arrays.asList(transform.value());
        if (parameter.isPrimitive()) {
            messages.add(ValidationMessageFactory.createError(parameter, "Parameters annotated with @Transform cannot be primitives."));
        } else if (transformTypes.contains(Transform.TransformType.GET_CLASS) && transformTypes.contains(Transform.TransformType.SIZE)) {
            messages.add(ValidationMessageFactory.createError(parameter, "Transform type '%s' not allowed with type '%s'", Transform.TransformType.GET_CLASS, Transform.TransformType.SIZE));
        } else if (transformTypes.contains(Transform.TransformType.HASH_CODE) && transformTypes.contains(Transform.TransformType.SIZE)) {
            messages.add(ValidationMessageFactory.createError(parameter, "Transform type '%s' not allowed with type '%s'", Transform.TransformType.HASH_CODE, Transform.TransformType.SIZE));
        } else if (transformTypes.contains(Transform.TransformType.IDENTITY_HASH_CODE) && transformTypes.contains(Transform.TransformType.SIZE)) {
            messages.add(ValidationMessageFactory.createError(parameter, "Transform type '%s' not allowed with type '%s'", Transform.TransformType.IDENTITY_HASH_CODE, Transform.TransformType.SIZE));
        } else if (transformTypes.contains(Transform.TransformType.IDENTITY_HASH_CODE) && transformTypes.contains(Transform.TransformType.HASH_CODE)) {
            messages.add(ValidationMessageFactory.createError(parameter, "Transform type '%s' not allowed with type '%s'", Transform.TransformType.IDENTITY_HASH_CODE, Transform.TransformType.HASH_CODE));
        } else if (!(!transformTypes.contains(Transform.TransformType.SIZE) || parameter.isArray() || parameter.isVarArgs() || parameter.isSubtypeOf(Map.class) || parameter.isSubtypeOf(Collection.class) || parameter.isSubtypeOf(CharSequence.class))) {
            messages.add(ValidationMessageFactory.createError(parameter, "Invalid type (%s) for %s. Must be an array, %s, %s or %s.", parameter.type(), Transform.TransformType.SIZE, Collection.class.getName(), Map.class.getName(), CharSequence.class.getName()));
        }
    }

    private Collection<ValidationMessage> validateParameters(MessageMethod messageMethod) {
        ArrayList<ValidationMessage> messages = new ArrayList<ValidationMessage>();
        boolean foundCause = false;
        ReturnType returnType = messageMethod.returnType();
        for (Parameter parameter : messageMethod.parameters(Parameter.ParameterType.ANY)) {
            switch (parameter.parameterType()) {
                case CAUSE: {
                    if (foundCause) {
                        messages.add(ValidationMessageFactory.createError(messageMethod, "Only one cause parameter is allowed."));
                        break;
                    }
                    foundCause = true;
                    break;
                }
                case FQCN: {
                    if (parameter.type().equals(Class.class.getName())) break;
                    messages.add(ValidationMessageFactory.createError(parameter, "Parameter %s annotated with @LoggingClass on method %s must be of type %s.", parameter.name(), messageMethod.name(), Class.class.getName()));
                    break;
                }
                case FIELD: {
                    if (returnType.hasFieldFor(parameter)) break;
                    messages.add(ValidationMessageFactory.createError(parameter, "No target field found in %s with name %s with type %s.", returnType.type(), parameter.targetName(), parameter.type()));
                    break;
                }
                case PROPERTY: {
                    if (returnType.hasMethodFor(parameter)) break;
                    messages.add(ValidationMessageFactory.createError(parameter, "No method found in %s with signature %s(%s).", returnType.type(), parameter.targetName(), parameter.type()));
                }
            }
        }
        return messages;
    }

    private Collection<ValidationMessage> validateBundle(Set<MessageMethod> messageMethods) {
        ArrayList<ValidationMessage> messages = new ArrayList<ValidationMessage>();
        for (MessageMethod messageMethod : messageMethods) {
            messages.addAll(this.validateBundleMethod(messageMethod));
        }
        return messages;
    }

    private Collection<ValidationMessage> validateBundleMethod(MessageMethod messageMethod) {
        ArrayList<ValidationMessage> messages = new ArrayList<ValidationMessage>();
        ReturnType returnType = messageMethod.returnType();
        if (returnType.equals(ReturnType.VOID) || returnType.isPrimitive()) {
            messages.add(ValidationMessageFactory.createError(messageMethod, "Message bundle messageMethod %s has an invalid return type. Cannot be void or a primitive.", messageMethod.name()));
        } else if (returnType.isThrowable()) {
            ThrowableType throwableReturnType = returnType.throwableReturnType();
            if (!throwableReturnType.useConstructionParameters()) {
                if (!throwableReturnType.useConstructionParameters() && !messageMethod.parameters(Parameter.ParameterType.CONSTRUCTION).isEmpty()) {
                    messages.add(ValidationMessageFactory.createError(messageMethod, "MessageMethod does not have an usable constructor for the return type %s.", returnType.name()));
                } else {
                    boolean usableConstructor;
                    boolean hasMessageConstructor = throwableReturnType.hasStringAndThrowableConstructor() || throwableReturnType.hasThrowableAndStringConstructor() || throwableReturnType.hasStringConstructor();
                    boolean bl = usableConstructor = throwableReturnType.hasDefaultConstructor() || throwableReturnType.hasStringAndThrowableConstructor() || throwableReturnType.hasStringConstructor() || throwableReturnType.hasThrowableAndStringConstructor() || throwableReturnType.hasThrowableConstructor();
                    if (!usableConstructor) {
                        messages.add(ValidationMessageFactory.createError(messageMethod, "MessageMethod does not have an usable constructor for the return type %s.", returnType.name()));
                    } else if (!hasMessageConstructor) {
                        messages.add(ValidationMessageFactory.createWarning(messageMethod, "The message cannot be set via the throwable constructor and will be ignored."));
                    }
                }
            }
        } else if (!returnType.isAssignableFrom(String.class)) {
            messages.add(ValidationMessageFactory.createError(messageMethod, "Message bundle method (%s) has an invalid return type of %s.", messageMethod.name(), returnType.name()));
        }
        return messages;
    }

    private Collection<ValidationMessage> validateLogger(Set<MessageMethod> messageMethods) {
        ArrayList<ValidationMessage> messages = new ArrayList<ValidationMessage>();
        for (MessageMethod messageMethod : messageMethods) {
            if (messageMethod.isLoggerMethod()) {
                messages.addAll(this.validateLoggerMethod(messageMethod));
                continue;
            }
            messages.addAll(this.validateBundleMethod(messageMethod));
        }
        return messages;
    }

    private Collection<ValidationMessage> validateLoggerMethod(MessageMethod messageMethod) {
        ArrayList<ValidationMessage> messages = new ArrayList<ValidationMessage>();
        if (!ReturnType.VOID.equals(messageMethod.returnType())) {
            messages.add(ValidationMessageFactory.createError(messageMethod, "Message logger methods can only have a void return type."));
        }
        return messages;
    }

    private Set<MessageMethod> getAllMethods(MessageInterface messageInterface) {
        if (messageInterface.getAnnotatedType() == MessageInterface.AnnotatedType.NONE) {
            return Collections.emptySet();
        }
        LinkedHashSet<MessageMethod> messageMethods = new LinkedHashSet<MessageMethod>();
        for (MessageMethod messageMethod : messageInterface.methods()) {
            messageMethods.add(messageMethod);
        }
        for (MessageInterface msgInterface : messageInterface.extendedInterfaces()) {
            messageMethods.addAll(this.getAllMethods(msgInterface));
        }
        return messageMethods;
    }
}

