/**********************************************************************
Copyright (c) 2002 Mike Martin (TJDO) and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2003 Andy Jefferson - coding standards
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;

import java.util.HashMap;

import org.datanucleus.metadata.JoinMetaData;
import org.datanucleus.store.mapped.DatastoreClass;
import org.datanucleus.store.mapped.DatastoreContainerObject;
import org.datanucleus.store.mapped.DatastoreField;
import org.datanucleus.store.mapped.DatastoreIdentifier;
import org.datanucleus.store.mapped.IdentifierFactory;
import org.datanucleus.store.mapped.SecondaryDatastoreClass;

/**
 * A SQL table expression that joins superclass tables by joining them directly
 * to the surrounding QueryStatement.
 *
 * @see QueryExpression
 * @version $Revision: 1.20 $ 
 */
public class TableExprAsJoins extends LogicSetExpression
{
    protected final HashMap aliasesByTable = new HashMap();

    /**
     * Constructor for a table expressed as a series of joins.
     * @param qs the QueryExpression
     * @param mainTable the main table in the query
     * @param alias The alias/identifier of the table in the query
     */
    public TableExprAsJoins(QueryExpression qs, DatastoreContainerObject mainTable, DatastoreIdentifier alias)
    {
        super(qs, mainTable, alias);

        aliasesByTable.put(mainTable, alias);
    }

    public String referenceColumn(DatastoreField col)
    {
        assertNotFrozen();

        DatastoreContainerObject table = col.getDatastoreContainerObject();
        DatastoreIdentifier alias = (DatastoreIdentifier)aliasesByTable.get(table);

        if (alias == null)
        {
            if (!(mainTable instanceof DatastoreClass) || !(table instanceof DatastoreClass))
            {
                throw new TableMismatchException(col, mainTable);
            }

            /*
             * Since both tables are ClassTables we assume that the column
             * is a superclass field, meaning 'table' is a supertable of
             * 'mainTable'.  We join the supertable to the query statement using
             * range variable derived from the main range variable (an
             * underscore and a counter are appended).
             *
             * So even though the caller may have been looking for column FOO
             * relative to rangevar THIS he may actually get back something like
             * THIS_1.FOO.
             */
            IdentifierFactory idFactory = qs.getStoreManager().getIdentifierFactory();
            alias = idFactory.newIdentifier(mainAlias, String.valueOf(aliasesByTable.size()));
            aliasesByTable.put(table, alias);

            /* mt... = "main table" */
            /* st... = "secondary table" */
            DatastoreClass mt = (DatastoreClass)mainTable;
            DatastoreClass st = (DatastoreClass)table;
            boolean useInnerJoin = true;
            if (st instanceof SecondaryDatastoreClass)
            {
                // Support join with any secondary table via outer join
                JoinMetaData joinmd = ((SecondaryDatastoreClass)st).getJoinMetaData();
                if (joinmd != null && joinmd.isOuter())
                {
                    useInnerJoin = false;
                }
            }

            LogicSetExpression stTblExpr = qs.newTableExpression(st, alias);

            ScalarExpression mtExpr = mt.getIDMapping().newScalarExpression(qs,this);
            ScalarExpression stExpr = st.getIDMapping().newScalarExpression(qs,stTblExpr);

            if (useInnerJoin)
            {
                qs.innerJoin(mtExpr, stExpr, stTblExpr, true, true);
            }
            else
            {
                qs.leftOuterJoin(mtExpr, stExpr, stTblExpr, true, true);
            }
        }
        String aliasString = alias.toString();
        if (aliasString.length() > 0)
        {
            return col.applySelectFunction(aliasString + "." + col.getIdentifier());
        }
        else
        {
            return col.applySelectFunction(col.getIdentifier().toString());
        }
    }

    public String toString()
    {
        if (sqlText == null)
        {
            DatastoreIdentifier mainTableName = mainTable.getIdentifier();
            StringBuffer sb = new StringBuffer(mainTable.toString());
            if (!mainAlias.equals(mainTableName))
            {
                sb.append(' ').append(mainAlias);
            }

            sqlText = sb.toString();
        }

        return sqlText;
    }
}