/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.sstable;

import java.io.Closeable;
import java.io.DataInputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import org.apache.cassandra.cache.RefCountedMemory;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.RowPosition;
import org.apache.cassandra.dht.IPartitioner;
import org.apache.cassandra.io.sstable.Downsampling;
import org.apache.cassandra.io.util.DataOutputPlus;
import org.apache.cassandra.io.util.MemoryOutputStream;
import org.apache.cassandra.utils.FBUtilities;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IndexSummary
implements Closeable {
    private static final Logger logger = LoggerFactory.getLogger(IndexSummary.class);
    public static final IndexSummarySerializer serializer = new IndexSummarySerializer();
    private final int minIndexInterval;
    private final IPartitioner partitioner;
    private final int summarySize;
    private final int sizeAtFullSampling;
    private final RefCountedMemory bytes;
    private final int samplingLevel;

    public IndexSummary(IPartitioner partitioner, RefCountedMemory memory, int summarySize, int sizeAtFullSampling, int minIndexInterval, int samplingLevel) {
        this.partitioner = partitioner;
        this.minIndexInterval = minIndexInterval;
        this.summarySize = summarySize;
        this.sizeAtFullSampling = sizeAtFullSampling;
        this.bytes = memory;
        this.samplingLevel = samplingLevel;
    }

    public int binarySearch(RowPosition key) {
        int low = 0;
        int mid = this.summarySize;
        int high = mid - 1;
        int result = -1;
        while (low <= high) {
            mid = low + high >> 1;
            result = -DecoratedKey.compareTo(this.partitioner, ByteBuffer.wrap(this.getKey(mid)), key);
            if (result > 0) {
                low = mid + 1;
                continue;
            }
            if (result == 0) {
                return mid;
            }
            high = mid - 1;
        }
        return -mid - (result < 0 ? 1 : 2);
    }

    public int getPositionInSummary(int index) {
        return this.bytes.getInt(index << 2);
    }

    public byte[] getKey(int index) {
        long start = this.getPositionInSummary(index);
        int keySize = (int)(this.calculateEnd(index) - start - 8L);
        byte[] key = new byte[keySize];
        this.bytes.getBytes(start, key, 0, keySize);
        return key;
    }

    public long getPosition(int index) {
        return this.bytes.getLong(this.calculateEnd(index) - 8L);
    }

    public byte[] getEntry(int index) {
        long start = this.getPositionInSummary(index);
        long end = this.calculateEnd(index);
        byte[] entry = new byte[(int)(end - start)];
        this.bytes.getBytes(start, entry, 0, (int)(end - start));
        return entry;
    }

    private long calculateEnd(int index) {
        return index == this.summarySize - 1 ? this.bytes.size() : (long)this.getPositionInSummary(index + 1);
    }

    public int getMinIndexInterval() {
        return this.minIndexInterval;
    }

    public double getEffectiveIndexInterval() {
        return 128.0 / (double)this.samplingLevel * (double)this.minIndexInterval;
    }

    public long getEstimatedKeyCount() {
        return ((long)this.getMaxNumberOfEntries() + 1L) * (long)this.minIndexInterval;
    }

    public int size() {
        return this.summarySize;
    }

    public int getSamplingLevel() {
        return this.samplingLevel;
    }

    public int getMaxNumberOfEntries() {
        return this.sizeAtFullSampling;
    }

    public long getOffHeapSize() {
        return this.bytes.size();
    }

    public int getEffectiveIndexIntervalAfterIndex(int index) {
        return Downsampling.getEffectiveIndexIntervalAfterIndex(index, this.samplingLevel, this.minIndexInterval);
    }

    @Override
    public void close() {
        this.bytes.unreference();
    }

    public IndexSummary readOnlyClone() {
        this.bytes.reference();
        return this;
    }

    public static class IndexSummarySerializer {
        public void serialize(IndexSummary t, DataOutputPlus out, boolean withSamplingLevel) throws IOException {
            out.writeInt(t.minIndexInterval);
            out.writeInt(t.summarySize);
            out.writeLong(t.bytes.size());
            if (withSamplingLevel) {
                out.writeInt(t.samplingLevel);
                out.writeInt(t.sizeAtFullSampling);
            }
            out.write(t.bytes);
        }

        public IndexSummary deserialize(DataInputStream in, IPartitioner partitioner, boolean haveSamplingLevel, int expectedMinIndexInterval, int maxIndexInterval) throws IOException {
            int fullSamplingSummarySize;
            int samplingLevel;
            int minIndexInterval = in.readInt();
            if (minIndexInterval != expectedMinIndexInterval) {
                throw new IOException(String.format("Cannot read index summary because min_index_interval changed from %d to %d.", minIndexInterval, expectedMinIndexInterval));
            }
            int summarySize = in.readInt();
            long offheapSize = in.readLong();
            if (haveSamplingLevel) {
                samplingLevel = in.readInt();
                fullSamplingSummarySize = in.readInt();
            } else {
                samplingLevel = 128;
                fullSamplingSummarySize = summarySize;
            }
            int effectiveIndexInterval = (int)Math.ceil(128.0 / (double)samplingLevel * (double)minIndexInterval);
            if (effectiveIndexInterval > maxIndexInterval) {
                throw new IOException(String.format("Rebuilding index summary because the effective index interval (%d) is higher than the current max index interval (%d)", effectiveIndexInterval, maxIndexInterval));
            }
            RefCountedMemory memory = new RefCountedMemory(offheapSize);
            FBUtilities.copy(in, new MemoryOutputStream(memory), offheapSize);
            return new IndexSummary(partitioner, memory, summarySize, fullSamplingSummarySize, minIndexInterval, samplingLevel);
        }
    }
}

