package com.formos.tapestry.testify.junit4;

import org.junit.After;
import org.junit.Before;

import com.formos.tapestry.testify.core.TapestryTester;

/**
 * JUnit4 base class for Tapestry tests. Note that test classes need to override
 * {@link #doSetUp()}. 
 * <p>
 * I recommend that you create your own abstract class for you application tests
 * that subclasses this. You can then create a {@link TapestryTester} in that
 * class and store it in a static variable and also override
 * {@link #setUpForAllTestMethods()} for any other common setup.
 * <p>
 * For example:
 * 
 * <pre>
 * public abstract class MyApplicationTest extends TapestryTest {
 *     private static final TapestryTester SHARED_TESTER = new TapestryTester(
 *             &quot;pac.ka.ge&quot;, MyTestInfrastructureModule.class);
 * 
 *     public MyApplicationTest() {
 *         super(SHARED_TESTER);
 *     }
 * 
 *     &#064;Override
 *     protected void setUpForAllTestClasses() throws Exception {
 *         MockitoAnnotations.initMocks(this);
 *     }
 * }
 * </pre>
 */
public abstract class TapestryTest {
    protected final TapestryTester tester;


    /**
     * Creates a new instance using the given {@link TapestryTester}; you should
     * <em>create this tester just once</em> and share it across multiple tests,
     * for example by storing it in a static variable.
     * 
     * @param tester
     *            the tester to use (not null)
     */
    public TapestryTest(TapestryTester tester) {
        assert tester != null;
        this.tester = tester;
    }


    /**
     * @deprecated Rename your method so that it overrides {@link #setUpForAllTestMethods()} 
     */
    @Deprecated
    protected void setUpForAllTestClasses() throws Exception {
        setUpForAllTestMethods();
    }

    
    /**
     * See class javadoc for example: {@link TapestryTest}
     */
    protected void setUpForAllTestMethods() throws Exception {
        // Do nothing by default
    }

    
    /**
     * Override {@link #doSetUp()} instead.
     */
    @Before
    public final void setUp() throws Exception {
        tester.injectInto(this);
        setUpForAllTestClasses();
        doSetUp();
        tester.collectForComponentsFrom(this);
    }


    /**
     * Override this instead of {@link #setUp()}
     */
    protected void doSetUp() throws Exception {
        // Do nothing by default
    }


    @After
    public final void tearDown() throws Exception {
        tester.endTest();
    }


}
