package fr.onema.sispea.struts.synthesis.action;

/*
 * #%L
 * SISPEA web application
 * %%
 * Copyright (C) 2014 - 2015 ONEMA
 * %%
 * ONEMA - Tous droits réservés
 * #L%
 */


import com.opensymphony.xwork2.Action;
import fr.onema.sispea.SispeaException;
import fr.onema.sispea.service.data.CriterionType;
import fr.onema.sispea.service.data.IndicatorDto;
import fr.onema.sispea.service.data.IndicatorService;
import fr.onema.sispea.service.data.IndicatorType;
import fr.onema.sispea.service.data.SliceDto;
import fr.onema.sispea.service.data.SliceValuesDto;
import fr.onema.sispea.struts.common.menu.MenuConstants;
import org.apache.log4j.Logger;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * This action displays the global synthesis pages.
 *
 * @author CS
 */
public class ShowGlobalSynthesisAction extends AbstractSynthesisAction {

    private static final long serialVersionUID = 1L;

    private static final Logger logger = Logger.getLogger(ShowGlobalSynthesisAction.class);

    /**
     * the list of performance indicator.
     */
    protected List<IndicatorDto> ipList;

    /**
     * the list of descriptiv indicator.
     */
    protected List<IndicatorDto> idList;

    protected List<IndicatorType> indicatorTypes;

    /**
     * the number of population slice count (without all spea slides).
     */
    protected int populationSliceCount;

    /**
     * Store only visible slice values for each indicator type and indicator (the key is indicatorType_indicatorId).
     */
    protected Map<String, SliceValuesDto> globalValuesMap;

    /**
     * Store only visible  slice values for each indicator type - indicator and slice (the key is indicatorType_indicatorId_sliceId).
     */
    protected Map<String, SliceValuesDto> sliceValuesMap;

    @Override
    protected String doExecute() throws Exception {

        //init
        String lRes = Action.SUCCESS;

        if (logger.isDebugEnabled()) {
            logger.debug(getText("fr.onema.sispea.struts.synthesis.action.showGlobalSynthesis.debug"));
        }

        // no rights to check

        // default menu
        openMenu(MenuConstants.MENU_KEY_WHOLEFRANCE);

        try {

            // check params
            checkAndGetParameters(true, true);

            // open right menu
            openMenu(getMenuKey(true));

            // get Slices :
            slices = indicatorService.findSliceByCompetence(competenceId, IndicatorService.POPULATION_SLICE);

            // filter slice to remove pounderation criteria
            for (Iterator<SliceDto> iterator = slices.iterator(); iterator.hasNext(); ) {
                SliceDto lSlice = iterator.next();
                if (lSlice.getCriterion().getType() != CriterionType.IP) {
                    iterator.remove();
                }
            }

            // get slices values :
            slicesConsolidatedValues = indicatorService.getSlicesValues(slices, chosenExerciseId);

            ipList = computeIndicatorTypedList(IndicatorType.Performance);
            idList = computeIndicatorTypedList(IndicatorType.Descriptive);

            populationSliceCount = computePopulationSliceCount(slices);

            indicatorTypes = computeIndicatorTypes();
            globalValuesMap = computeGlobalValuesMap(indicatorTypes, slicesConsolidatedValues);
            sliceValuesMap = computeSliceValuesMap(indicatorTypes, slicesConsolidatedValues);

        } catch (SispeaException lEx) {
            String lMessage = getText(lEx.getMessage());
            addActionError(lMessage);
            lRes = Action.INPUT;
        }

        // result
        return lRes;
    }

    /**
     * @return the number of population slice count (without all spea slides)
     */
    public int getPopulationSliceCount() {
        return populationSliceCount;
    }

    /**
     * @return a performance indicator slice value
     */
    public List<IndicatorType> getIndicatorTypes() {
        return indicatorTypes;
    }

    public List<IndicatorDto> getIndicatorList(IndicatorType indicatorType) {

        List<IndicatorDto> result;
        if (indicatorType.isDescriptive()) {
            result = idList;
        } else {
            result = ipList;
        }
        return result;

    }

    /**
     * @param pIndicatorId
     * @param pIndicatorType
     * @return an indicator slice value for specified indicator type
     */
    public SliceValuesDto getGlobalValues(Integer pIndicatorId, IndicatorType pIndicatorType) {
        String key = pIndicatorType.name() + "_" + pIndicatorId;
        SliceValuesDto result = globalValuesMap.get(key);
        return result;
    }

    /**
     * @param pIndicatorId
     * @param pIndicatorType
     * @return an indicator slice value for specified indicator type
     */
    public SliceValuesDto getSliceValues(Integer pIndicatorId, Integer pSliceId, IndicatorType pIndicatorType) {
        String key = pIndicatorType.name() + "_" + pIndicatorId + "_" + pSliceId;
        SliceValuesDto result = sliceValuesMap.get(key);
        return result;
    }

//    public SliceValuesDto getGlobalPerformanceSliceValue(Integer pIndicatorId) {
//        return getGlobalValues(pIndicatorId, IndicatorType.Performance);
//    }

//    public SliceValuesDto getGlobalDescriptiveSliceValue(Integer pIndicatorId) {
//        return getGlobalValues(pIndicatorId, IndicatorType.Descriptive);
//    }

    /**
     * @param slicesConsolidatedValues
     * @param pIndicatorId
     * @param pIndicatorType
     * @return an indicator slice value for specified indicator type
     */
    protected SliceValuesDto getGlobalValues(List<SliceValuesDto> slicesConsolidatedValues, Integer pIndicatorId, IndicatorType pIndicatorType) {

        // init
        SliceValuesDto lRes = null;

        // loop on slice values to get search value
        for (Iterator<SliceValuesDto> iterator = slicesConsolidatedValues.iterator(); iterator.hasNext() && lRes == null; ) {
            SliceValuesDto lSliceValues = iterator.next();
            if (lSliceValues != null
                && lSliceValues.getIndicator().getId().equals(pIndicatorId)
                && lSliceValues.getSlice().getAllSpea()
                && lSliceValues.getIndicator().getType() == pIndicatorType) {
                lRes = lSliceValues;
            }
        }

        // result
        return lRes;
    }

    protected List<IndicatorType> computeIndicatorTypes() {

        List<IndicatorType> result = new ArrayList<>();
        for (int i = 0; i < IndicatorType.values().length; i++) {
            IndicatorType lType = IndicatorType.values()[i];
            if (lType != IndicatorType.Unknown) {
                result.add(lType);
            }
        }
        return result;

    }

    protected int computePopulationSliceCount(List<SliceDto> slices) {

        int result = 0;
        if (slices != null) {
            for (SliceDto lSlice : slices) {
                if (!lSlice.getAllSpea()) {
                    result += 1;
                }
            }
        }
        return result;

    }

    protected Map<String, SliceValuesDto> computeGlobalValuesMap(List<IndicatorType> indicatorTypes, List<SliceValuesDto> slicesConsolidatedValues) {

        Map<String, SliceValuesDto> result = new HashMap<>();

        for (IndicatorType indicatorType : indicatorTypes) {

            String keyPrefix = indicatorType.name() + "_";

            List<IndicatorDto> indicatorList = getIndicatorList(indicatorType);
            for (IndicatorDto indicatorDto : indicatorList) {
                if (indicatorDto.getHasConsolidation()) {
                    Integer indicatorId = indicatorDto.getId();
                    SliceValuesDto displayValues = getGlobalValues(slicesConsolidatedValues, indicatorId, indicatorType);
                    if (displayValues != null) {
                        boolean isVisible = getVisibility(chosenExerciseId, indicatorId, Integer.valueOf(displayValues.getIdSlice()), displayValues.getNb());
                        if (isVisible) {
                            result.put(keyPrefix + indicatorId, displayValues);
                        }
                    }
                }
            }

        }
        return result;

    }

    protected Map<String, SliceValuesDto> computeSliceValuesMap(List<IndicatorType> indicatorTypes, List<SliceValuesDto> slicesConsolidatedValues) {

        Map<String, SliceValuesDto> result = new HashMap<>();

        for (IndicatorType indicatorType : indicatorTypes) {

            List<IndicatorDto> indicatorList = getIndicatorList(indicatorType);
            for (IndicatorDto indicatorDto : indicatorList) {
                if (indicatorDto.getHasConsolidation()) {

                    Integer indicatorId = indicatorDto.getId();
                    String keyPrefix = indicatorType.name() + "_" + indicatorId + "_";
                    for (SliceDto sliceDto : getSlices()) {
                        SliceValuesDto displayValues = getSliceValues(slicesConsolidatedValues, indicatorId, (int) sliceDto.getId(), indicatorType);

                        if (displayValues != null) {
                            boolean isVisible = getVisibility(chosenExerciseId, indicatorId, Integer.valueOf(displayValues.getIdSlice()), displayValues.getNb());
                            if (isVisible) {
                                result.put(keyPrefix + sliceDto.getId(), displayValues);
                            }
                        }
                    }

                }
            }

        }
        return result;

    }

    /**
     * @param pIndicatorId
     * @param pIndicatorType
     * @return an indicator slice value for specified indicator type
     */
    protected SliceValuesDto getSliceValues(List<SliceValuesDto> slicesConsolidatedValues, Integer pIndicatorId, Integer pSliceId, IndicatorType pIndicatorType) {

        // init
        SliceValuesDto lRes = null;

        // loop on slice values to get search value
        for (Iterator<SliceValuesDto> iterator = slicesConsolidatedValues.iterator(); iterator.hasNext() && lRes == null; ) {
            SliceValuesDto lSliceValues = iterator.next();
            if (lSliceValues != null
                && lSliceValues.getSlice().getId() == pSliceId.shortValue()
                && lSliceValues.getIndicator().getId().equals(pIndicatorId)
                && lSliceValues.getIndicator().getType() == pIndicatorType) {
                lRes = lSliceValues;
            }
        }

        // result
        return lRes;
    }

//    /**
//     * @param pIndicatorId
//     * @param pSliceId
//     * @return an indicator slice value for Performance Indicator Type
//     */
//    public SliceValuesDto getPerformanceSliceValues(Integer pIndicatorId, Integer pSliceId) {
//        return getSliceValues(pIndicatorId, pSliceId, IndicatorType.Performance);
//    }


//    /**
//     * @param pIndicatorId
//     * @param pSliceId
//     * @return an indicator slice value for Descriptive Indicator Type
//     */
//    public SliceValuesDto getDescriptiveSliceValues(Integer pIndicatorId, Integer pSliceId) {
//        return getSliceValues(pIndicatorId, pSliceId, IndicatorType.Descriptive);
//    }

//    /**
//     * @return a list of performance indicator
//     */
//    public List<IndicatorDto> getIpList() {
//        return ipList;
//    }
//
//    /**
//     * @return a list of descriptiv indicator
//     */
//    public List<IndicatorDto> getIdList() {
//        return idList;
//    }

    /**
     * @return a list of performance indicator
     */
    protected List<IndicatorDto> computeIndicatorTypedList(IndicatorType pType) {
        // init
        List<IndicatorDto> lRes = new ArrayList<>();

        Short year = null;
        try {
            year = exerciseService.readExercise(getChosenExerciseId()).getYear();
        } catch (SispeaException e) {
            //Do nothing
        }

        for (IndicatorDto lIndicator : indicators) {
            if (lIndicator.getType() == pType) {
                Boolean isValid = null;
                // VALIDITY : keep non obsolete indicator
                if (year != null) {
                    isValid = indicatorService.isInValidityPeriod(lIndicator.getCode(), year);
                }
                if (isValid != null && isValid) {
                    lRes.add(lIndicator);
                }
            }
        }

        // result
        return lRes;
    }


}
