package fr.onema.sispea.struts.referential.action;

/*
 * #%L
 * SISPEA web application
 * %%
 * Copyright (C) 2014 - 2015 ONEMA
 * %%
 * ONEMA - Tous droits réservés
 * #L%
 */


import fr.onema.sispea.SispeaException;
import fr.onema.sispea.service.data.ExerciseDto;
import fr.onema.sispea.service.data.SPEALotStatus;
import fr.onema.sispea.service.referential.DDTDto;
import fr.onema.sispea.service.referential.DdtRefUpdateDto;
import fr.onema.sispea.service.referential.SPEADto;
import fr.onema.sispea.service.referential.TerritoryDto;
import fr.onema.sispea.service.referential.TerritoryType;
import fr.onema.sispea.service.user.Right;
import fr.onema.sispea.service.user.UserDto;
import fr.onema.sispea.struts.SispeaSession;
import fr.onema.sispea.struts.common.menu.MenuConstants;
import org.apache.log4j.Logger;

import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * This action handles the territory visualization.
 *
 * @author CS
 */
public class ShowTerritoryAction extends AbstractTerritoryAction {

    private static final long serialVersionUID = 1L;

    private static final Logger logger = Logger.getLogger(ShowTerritoryAction.class);

    /**
     * Store for each spea the lot status associated.
     */
    protected Map<Integer, SPEALotStatus> speaLotStatusMap = new HashMap<>();

    /**
     * true if current user can view this territory's users
     */
    protected boolean canViewUsers;

    protected boolean canEditTerritory;

    protected int associatedTerritoryId;

    /**
     * update date for ddt
     */
    protected Date updateDate;

    @Override
    protected String doExecute() throws Exception {
        String lResult = SUCCESS;

        if (logger.isDebugEnabled()) {
            logger.debug(getText("fr.onema.sispea.struts.referential.action.ShowTerritoryAction.debug.doExecute"));
        }

        try {

            // check exercise
            checkExercise();

            // check territory
            checkTerritory();

            if (territory.getType().isDDT()) {
                updateDate = computeDdtUpdateDate(territory, exercise);
            }

            associatedTerritoryId = getAssociatedTerritoryId(territory);

            canViewUsers = computeCanViewUsers(getCurrentUser(), territory, exercise);
            canEditTerritory = computeCanEditTerritory(territory);

            // load speas
            loadTerritorySpeas(territory, exercise);

            Integer exerciceId = getExercise().getId();
            for (SPEADto spea : getSpeas()) {

                Integer speaId = spea.getId();

                if (logger.isDebugEnabled()) {
                    logger.debug("Loading data for spea " + speaId);
                }

                SPEALotStatus speaLotStatus = speaLotService.readSpeaLotStatus(speaId, exerciceId);
                speaLotStatusMap.put(speaId, speaLotStatus);

            }

        } catch (SispeaException lEx) {
            addActionError(lEx.getMessage());
            lResult = ERROR;
        }

        if (!ERROR.equals(lResult)) {

            // should redirect ?
            TerritoryType territoryType = territory.getType();

            if (TerritoryType.SPEA == territoryType) {
                // show SPEA
                lResult = TerritoryType.SPEA.name();
            } else if (TerritoryType.Collectivity == territoryType) {
                // show collectivity
                lResult = TerritoryType.Collectivity.name();
            } else if (TerritoryType.Municipality == territoryType) {
                // show municipality
                lResult = TerritoryType.Municipality.name();
            } else {

                // open right menu
                if (mode != null && mode.equals(DISPLAY_MODE_MY_TERRITORY)) {
                    openMenu(MenuConstants.MENU_KEY_MYTERRITORIES);
                } else {
                    openMenu(MenuConstants.MENU_KEY_SEE_TERRITORY);
                }

                // Reset the Territory name edition panel state.
                modifName = false;
                // Reset the Territory organisms edition panel state.
                modifOrganisms = false;
                // Initialize list of organisms to remove from territory.
                organismsToRemove.clear();

                // load users ?
                if (canViewUsers) {
                    loadTerritoryUsers(territory);
                }

            }

            manageOrganismPaging();
            manageUserPaging(territory);
        }

        // result
        return lResult;

    }

    public SPEALotStatus getSpeaLotStatus(Integer speaId) {
        return speaLotStatusMap.get(speaId);
    }

    public boolean isCanViewUsers() {
        return canViewUsers;
    }

    public void setCanViewUsers(boolean canViewUsers) {
        this.canViewUsers = canViewUsers;
    }

    public int getAssociatedTerritoryId() {
        return associatedTerritoryId;
    }

    public void setAssociatedTerritoryId(int associatedTerritoryId) {
        this.associatedTerritoryId = associatedTerritoryId;
    }

    public boolean isCanEditTerritory() {
        return canEditTerritory;
    }

    public Date getUpdateDate() throws SispeaException {
        return updateDate;
    }

    public void setUpdateDate(Date pUpdateDate) {
        updateDate = pUpdateDate;
    }

    /**
     * Test if currently displayed territory can be edited by current user.
     * User must be an administrator and the territory must be of type 'mixed'.
     *
     * @param territory
     * @return
     */
    protected boolean computeCanEditTerritory(TerritoryDto territory) {

        // If a territory exists and is a Mixed territory, it can be edited.
        boolean lMixedTerritory = (territory != null) && (TerritoryType.Mixed.equals(territory.getType()));
        // If user is an administrator, he can edit a territory.
        SispeaSession sispeaSession = getSispeaSession();
        UserDto user = sispeaSession.getUser();
        boolean lAdminUser = user != null && user.checkRight(Right.Manage);

        return lMixedTerritory && lAdminUser;
    }

    /**
     * @return
     * @throws SispeaException
     */
    protected Date computeDdtUpdateDate(TerritoryDto territory, ExerciseDto exercise) throws SispeaException {

        DDTDto lDdt = referentialService.readDDTFromTerritoryId(territory.getId(), exercise);

        Date lRes = null;

        Integer exerciseId = exercise.getId();
        for (DdtRefUpdateDto lDdtRefUpdate : lDdt.getDdtUpdates()) {
            if (lDdtRefUpdate != null && lDdtRefUpdate.getExercise().getId().equals(exerciseId)) {
                lRes = lDdtRefUpdate.getDateUpdate();
            }
        }

        return lRes;

    }
}
