package fr.onema.sispea.struts.referential.action;

/*
 * #%L
 * SISPEA web application
 * %%
 * Copyright (C) 2014 - 2015 ONEMA
 * %%
 * ONEMA - Tous droits réservés
 * #L%
 */


import com.google.common.collect.HashMultimap;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Multimap;
import fr.onema.sispea.SispeaException;
import fr.onema.sispea.service.ServiceMessages;
import fr.onema.sispea.service.common.MessageDto;
import fr.onema.sispea.service.common.MessageType;
import fr.onema.sispea.service.data.DataDetailValueDto;
import fr.onema.sispea.service.data.DataDto;
import fr.onema.sispea.service.data.DataFloatValueDto;
import fr.onema.sispea.service.data.DataListDto;
import fr.onema.sispea.service.data.DrinkingWaterVolumeService;
import fr.onema.sispea.service.data.ExerciseDto;
import fr.onema.sispea.service.data.ExerciseService;
import fr.onema.sispea.service.data.FormCalculateService;
import fr.onema.sispea.service.data.IndicatorDto;
import fr.onema.sispea.service.data.IndicatorService;
import fr.onema.sispea.service.data.OriginDto;
import fr.onema.sispea.service.data.OriginType;
import fr.onema.sispea.service.data.Reliability;
import fr.onema.sispea.service.data.SPEALotDto;
import fr.onema.sispea.service.data.SPEALotInputDataType;
import fr.onema.sispea.service.data.SPEALotService;
import fr.onema.sispea.service.data.SPEALotStatus;
import fr.onema.sispea.service.data.VPDto;
import fr.onema.sispea.service.data.Verification;
import fr.onema.sispea.service.exchange.ExportParameters;
import fr.onema.sispea.service.exchange.FileExchangeService;
import fr.onema.sispea.service.exchange.FileType;
import fr.onema.sispea.service.referential.CompetenceDto;
import fr.onema.sispea.service.referential.OrganismDto;
import fr.onema.sispea.service.referential.OrganismType;
import fr.onema.sispea.service.referential.ReferentialService;
import fr.onema.sispea.service.referential.SPEADto;
import fr.onema.sispea.service.referential.TerritoryDto;
import fr.onema.sispea.service.user.PasswordService;
import fr.onema.sispea.service.user.Right;
import fr.onema.sispea.service.user.UserDto;
import fr.onema.sispea.struts.AbstractSispeaAction;
import fr.onema.sispea.struts.StrutsMessages;
import fr.onema.sispea.struts.referential.bean.DrinkingWaterNetworkSchemaBean;
import fr.onema.sispea.struts.referential.bean.XmlData;
import fr.onema.sispea.struts.referential.bean.XmlDisplayGroup;
import fr.onema.sispea.struts.referential.bean.XmlGroup;
import fr.onema.sispea.struts.referential.bean.XmlGroupTab;
import fr.onema.sispea.struts.referential.bean.XmlGroupTable;
import fr.onema.sispea.struts.referential.bean.XmlGroupTableDetail;
import fr.onema.sispea.struts.referential.bean.XmlGroupTableValue;
import fr.onema.sispea.struts.referential.xml.XmlDataReader;
import fr.onema.sispea.util.constants.NumberUtils;
import org.apache.log4j.Logger;
import org.apache.struts2.ServletActionContext;
import org.springframework.beans.factory.annotation.Autowired;

import javax.servlet.http.HttpServletRequest;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.TreeMap;

/**
 * This abstract action is the basis for all actions used to manage SPEA indicators.
 *
 * @author CS
 */
public abstract class AbstractSpeaIndicatorsAction extends AbstractSispeaAction {

    private static final long serialVersionUID = 1L;

    private static final Logger logger = Logger.getLogger(AbstractSpeaIndicatorsAction.class);

//    private static final String GROUP_CODE_EFFICIENCY = "EFFICIENCY";

    @Autowired
    protected transient ReferentialService referentialService;

    @Autowired
    protected transient SPEALotService speaLotService;

    @Autowired
    protected transient ExerciseService exerciseService;

    @Autowired
    protected transient FileExchangeService fileExchangeService;

    @Autowired
    protected transient IndicatorService indicatorService;

    @Autowired
    protected transient DrinkingWaterVolumeService drinkingWaterService;

    @Autowired
    protected transient PasswordService passwordService;

    @Autowired
    protected transient FormCalculateService formCalculateService;

    /**
     * spea id to display
     */
    protected Integer id;

    /**
     * Exercise id to display
     */
    protected Integer exerciseId;

    /**
     * exercise to display
     */
    protected ExerciseDto exercise;

    /**
     * data lot to display
     */
    protected SPEALotDto lot;

    /**
     * data lot from previous exercise
     */
    protected SPEALotDto oldLot;

    /**
     * spea to display
     */
    protected SPEADto spea;

    /**
     * general comment on lot
     */
    protected String generalComment;

    /**
     * true if comment is beiing edited
     */
    protected boolean editingComment;

    /**
     * true if datatoinput is beiing edited
     */
    protected boolean editingDataToInput;

    /**
     * true if autopublish is beiing edited
     */
    protected boolean editingAutopublish;

    protected boolean unpublishable;

    /**
     * the value of upper bound
     */
    protected String upperBound = "0";

    /**
     * the value of upper bound for vp 177
     */
    protected String upperBoundVP177 = "0";

    /**
     * the value of upper bound for vp 178
     */
    protected String upperBoundVP178 = "0";

    /**
     * the value of lower bound
     */
    protected Long lowerBound = (long) 0;

    /**
     * display xmlGroup list
     */
    protected List<XmlGroupTab> xmlGroups = new ArrayList<>();

    /**
     * true if user has view right
     */
    protected boolean hasViewRight;

    /**
     * true if user has edit right
     */
    protected boolean hasEditRight;

    /**
     * true if user has check right
     */
    protected boolean hasCheckRight;

    /**
     * true if user has publish right
     */
    protected boolean hasPublishRight;

    /**
     * true if user has unpublish right
     */
    protected boolean hasUnpublishRight;

    /**
     * editing mode
     */
    protected String editingGroupId;

    /**
     * group to edit id to go back to data page
     */
    protected String groupToEditId;

    /**
     * Data context group
     */
    protected String dcCode = "D151.0";

    /**
     * Data Context xml display group
     */
    protected XmlDisplayGroup dataContextGroup;

    /**
     * Lists of IP/VP
     */
    protected List<DataListDto> dataLists = new ArrayList<>();

    /**
     * Bean holding all values for the drinking water network schema.
     */
    protected DrinkingWaterNetworkSchemaBean dwnSchema;

    /**
     * Slices to delete for VP.177, upper bounds slices values are separated with ';'
     */
    protected String slicesToDeleteVP177;

    /**
     * Slices to delete for VP.178, upper bounds slices values are separated with ';'
     */
    protected String slicesToDeleteVP178;

    /**
     * Slices to add for VP.177, upper bounds slices values are separated with ';'
     */
    protected String slicesToAddVP177;

    /**
     * Slices to add for VP.178, upper bounds slices values are separated with ';'
     */
    protected String slicesToAddVP178;

    /**
     * Alert message no data
     */
    protected String emptyData;

    protected Map<String, String> exerciseList;

    /**
     * Key: dataList id, Values : all ip and vp code of the dataList.
     */
    protected Multimap<Integer, String> ipOrVpCodesPerDataList;

    /**
     * handle the two visualisation actions
     *
     * @throws SispeaException
     */
    protected void manageViewActions(boolean pJustShowIndictors, boolean pFromSession, boolean pReloadLot) throws SispeaException {

        // init messages
        Collection<MessageDto> lMessages = new HashSet<>();

        // check spea id
        checkSpeaParameter();

        // check exercise
        checkExerciseParameter();

        // check if spea exists
        loadSpea(exercise);

        // load all exercises
        exerciseList = loadExcerciseList();

        // load users rights
        loadUserRights();

        // load lot
        if (pReloadLot) {
            loadLot();
        }

        loadOldLot();

        // Compute values for the drinking water network schema
        if (spea.getCompetence().isDrinkingWater()) {
            computeDrinkingWaterNetworkValues(lot);
        }

        //
        boolean lHasViewRight = isHasViewRight();
        if (pJustShowIndictors) {
            lHasViewRight = false;
        }

        // load data lists
        loadDataLists();

        if (!pFromSession) {
            // load data lists (put it out of the if condition, but still keep this code in comment to remember the evolution, just in case...)
//            loadDataLists();

            xmlGroups = buildXmlBean(lot, oldLot, spea.getCompetence().getId(), lHasViewRight);

            // add xml groups in sispea session (for update purpose)
            getSispeaSession().setXmlGroups(xmlGroups);
        } else {

            xmlGroups = getSispeaSession().getXmlGroups();
        }

        // check if there's tab to display
        if (xmlGroups != null) {
            boolean isTabToDisplay = false;
            for (Iterator<XmlGroupTab> iterator = xmlGroups.iterator(); iterator.hasNext() && !isTabToDisplay; ) {
                XmlGroupTab lXmlGroupTab = iterator.next();
                if (lXmlGroupTab != null && !lXmlGroupTab.isGeneralTab()) {
                    isTabToDisplay = true;
                }
            }
            if (!isTabToDisplay) {
                emptyData = getText("fr.onema.sispea.referential.jsp.showSpeaIndicators.noData");
            } else if (!pJustShowIndictors) {
                // generate messages of lot if displaying all data
                lMessages = speaLotService.generateMessages(lot);
            }
        }

        //Debut -- Mantis 4172
        if (xmlGroups != null) {
            UserDto lUser = getCurrentUser();
            // Not connected and old lot status not published
            if (lUser == null && oldLot != null && !SPEALotStatus.Published.equals(oldLot.getLotStatus())) {
                for (XmlGroupTab xmlGroup : xmlGroups) {
                    if (xmlGroup != null) {
                        List<XmlGroupTable> xmlGoupTables = xmlGroup.getGroupTables();
                        if (xmlGoupTables != null) {
                            for (XmlGroupTable xmlGoupTable : xmlGoupTables) {
                                if (xmlGoupTable != null) {
                                    List<XmlGroupTableValue> xmlGroupTableValues = xmlGoupTable.getGroupTableValues();
                                    if (xmlGroupTableValues != null) {
                                        for (XmlGroupTableValue xmlGroupTableValue : xmlGroupTableValues) {
                                            if (xmlGroupTableValue != null) {
                                                xmlGroupTableValue.setVisible(false);
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        //Fin -- Mantis 4172

        // display messages
        displayResultMessages(lMessages);

    }

    protected void loadUserRights() throws SispeaException {
        hasViewRight = computeUserViewRight();
        hasEditRight = computeUserEditRight();
        hasCheckRight = computeUserCheckRight();
        hasPublishRight = computeUserPublishRight();
        hasUnpublishRight = computeUserUnpublishRight();
        unpublishable = speaLotService.isSpeaLotUnpublishable(exerciseId, id);
    }

    /**
     * check Spea parameter
     *
     * @throws SispeaException
     */
    protected void checkSpeaParameter() throws SispeaException {
        // check spea id
        if (id == null) {
            id = getId();
        }
        if (id == null) {
            throw new SispeaException("fr.onema.sispea.referential.action.showSpeaIndicators.error.noSpeaId");
        }
    }

    /**
     * check exercise parameter
     *
     * @throws SispeaException
     */
    protected void checkExerciseParameter() throws SispeaException {

        if (exerciseId == null) {
            exerciseId = exerciseService.getCurrentExerciseId();
        }

        // check if exercise exists
        exercise = exerciseService.readExercise(exerciseId);
        if (exercise == null) {
            throw new SispeaException("fr.onema.sispea.referential.action.showSpeaIndicators.error.wrongExerciseId");
        }
    }

    /**
     * loads spea if exists, raise exception else
     *
     * @throws SispeaException
     */
    protected void loadSpea(ExerciseDto pExercise) throws SispeaException {
        long start = System.currentTimeMillis();
        spea = referentialService.readSPEA(id, pExercise, false);
        if (logger.isInfoEnabled()) {
            logger.info(String.format("Spea (%d) read in %dms", id, System.currentTimeMillis() - start));
        }
        if (spea == null) {
            throw new SispeaException("fr.onema.sispea.referential.action.showSpeaIndicators.error.wrongSpeaId");
        }
    }

    /**
     * loads spea if exists, raise exception else
     *
     * @throws SispeaException
     */
    protected void loadLot() throws SispeaException {
        lot = speaLotService.readSpeaLot(exerciseId, id, hasViewRight);
        // check lot
        if (lot == null) {
            throw new SispeaException("fr.onema.sispea.referential.action.showSpeaIndicators.error.noLot");
        }

        // save comment
        generalComment = lot.getComment();
    }

    /**
     * loads spea from previous exercise if exists
     *
     * @throws SispeaException
     */
    protected void loadOldLot() throws SispeaException {
        ExerciseDto lExercise = null;
        List<ExerciseDto> lExercises = exerciseService.readExercises();
        for (ExerciseDto ex : lExercises) {
            if (ex.getYear() == exercise.getYear() - 1) {
                lExercise = ex;
            }
        }

        if (lExercise != null) {
            oldLot = speaLotService.readSpeaLot(lExercise.getId(), id, hasViewRight);
        }
    }

    protected void computeDrinkingWaterNetworkValues(SPEALotDto pSpeaLot) throws SispeaException {

        DrinkingWaterNetworkSchemaBean bean = new DrinkingWaterNetworkSchemaBean();

        String valueNotSet = "____";

        // Produced volume: VP.059
        DataFloatValueDto vp59 = (DataFloatValueDto) pSpeaLot.getDataValueByCode("VP.059");
        if (vp59 == null || vp59.isEmpty()) {
            bean.setProduction(valueNotSet);
        } else {
            String production = vp59.getFormattedFloatValue();
            bean.setProduction(production);
        }

        // Distributed volume: use the vmd formula
        Double distributedVolume = drinkingWaterService.computeVMD(pSpeaLot);
        if (distributedVolume == null || distributedVolume.equals(Double.NaN)) {
            bean.setDistribution(valueNotSet);
        } else {
            String distributionStr = NumberUtils.doubleToString(distributedVolume, 0, false);
            bean.setDistribution(distributionStr);
        }

        // Authorized consumed volume: use the vca formula
        Double authorizedConsumedVolume = drinkingWaterService.computeVCA(pSpeaLot);
        if (authorizedConsumedVolume == null || authorizedConsumedVolume.equals(Double.NaN)) {
            bean.setAuthorized(valueNotSet);
        } else {
            String authorizedConsumedVolumeStr = NumberUtils.doubleToString(authorizedConsumedVolume, 0, false);
            bean.setAuthorized(authorizedConsumedVolumeStr);
        }

        // Counted consumption: use the cc formula
        Double countedConsumption = drinkingWaterService.computeCC(pSpeaLot);
        if (countedConsumption == null || countedConsumption.equals(Double.NaN)) {
            bean.setCounted(valueNotSet);
        } else {
            String countedConsumptionStr = NumberUtils.doubleToString(countedConsumption, 0, false);
            bean.setCounted(countedConsumptionStr);
        }

        // Imported volume: VP.060
        DataFloatValueDto vp60 = (DataFloatValueDto) pSpeaLot.getDataValueByCode("VP.060");
        if (vp60 == null || vp60.isEmpty()) {
            bean.setImported(valueNotSet);
        } else {
            String imported = vp60.getFormattedFloatValue();
            bean.setImported(imported);
        }

        // Exported volume: VP.061
        DataFloatValueDto vp61 = (DataFloatValueDto) pSpeaLot.getDataValueByCode("VP.061");
        if (vp61 == null || vp61.isEmpty()) {
            bean.setExported(valueNotSet);
        } else {
            String exported = vp61.getFormattedFloatValue();
            bean.setExported(exported);
        }

        // Lost volume: vmd - vca or distributedVolume - authorizedConsumedVolume
        double lost = 0;
        if (distributedVolume == null || authorizedConsumedVolume == null ||
            distributedVolume.equals(Double.NaN) || authorizedConsumedVolume.equals(Double.NaN)) {
            bean.setLost(valueNotSet);
        } else {
            lost = distributedVolume - authorizedConsumedVolume;
            String lostStr = NumberUtils.doubleToString(lost, 0, false);
            bean.setLost(lostStr);
        }

        // Not counted
        double notCounted = lost;

        // Consumed without estimations: VP.221
        DataFloatValueDto vp221 = (DataFloatValueDto) pSpeaLot.getDataValueByCode("VP.221");
        if (vp221 == null || vp221.isEmpty()) {
            bean.setConsumed(valueNotSet);
        } else {
            notCounted = notCounted + vp221.getFloatValue();
            String consumed = vp221.getFormattedFloatValue();
            bean.setConsumed(consumed);
        }

        // Service volume: VP.220
        DataFloatValueDto vp220 = (DataFloatValueDto) pSpeaLot.getDataValueByCode("VP.220");
        if (vp220 == null || vp220.isEmpty()) {
            bean.setServiceVolume(valueNotSet);
        } else {
            notCounted = notCounted + vp220.getFloatValue();
            String serviceVolume = vp220.getFormattedFloatValue();
            bean.setServiceVolume(serviceVolume);
        }

        // Volume not counted: [calcul] lost + consumed + volume service
        String notCountedStr = NumberUtils.doubleToString(notCounted, 0, false);
        bean.setNotCounted(notCountedStr);

        dwnSchema = bean;
    }

    /**
     * Load the active data lists on the current territory
     *
     * @throws SispeaException
     */
    protected void loadDataLists() throws SispeaException {
        List<DataListDto> lLists = indicatorService.readDataLists();
        //List<Integer> lTerritories = lot.getSpea().getTerritory().getIncludedTerritoryIds();
        TerritoryDto lSpeaTerritory = territoryService.readTerritory(spea.getId(), OrganismType.SPEA);

        ExerciseDto exercise = lot.getExercise();

        CompetenceDto competence = spea.getCompetence();

        ipOrVpCodesPerDataList = HashMultimap.create();

        // Loop through data lists and only keep the active ones on current territory
        for (DataListDto lDataList : lLists) {
            if (lDataList.isActive()) {
                // Current list is active
                if (territoryService.checkInclusion(lDataList.getTerritory(), lSpeaTerritory, exercise.getId())) {

                    boolean acceptComptence = acceptDataListCompetence(lDataList, competence);
                    if (!acceptComptence) {
                        if (logger.isDebugEnabled()) {
                            logger.debug("Exclude dataList " + lDataList.getName() + " competence mismatch");
                        }
                        continue;
                    }

                    boolean acceptFromExercice = acceptDataListFromExercice(lDataList, exercise);
                    if (!acceptFromExercice) {
                        if (logger.isDebugEnabled()) {
                            logger.debug("Exclude dataList " + lDataList.getName() + " fromExercice mismatch");
                        }
                        continue;
                    }

                    boolean acceptToExercice = acceptDataListToExercice(lDataList, exercise);
                    if (!acceptToExercice) {
                        if (logger.isDebugEnabled()) {
                            logger.debug("Exclude dataList " + lDataList.getName() + " toExercice mismatch");
                        }
                        continue;
                    }

                    if (logger.isDebugEnabled()) {
                        logger.debug("Keep dataList " + lDataList.getName());
                    }

                    dataLists.add(lDataList);

                    // Fill ipOrVpIdsPerDataList
                    for (IndicatorDto indicatorDto : lDataList.getIps()) {
                        ipOrVpCodesPerDataList.put(lDataList.getId(), indicatorDto.getCode());
                    }
                    for (VPDto vpDto : lDataList.getVps()) {
                        ipOrVpCodesPerDataList.put(lDataList.getId(), vpDto.getCode());
                    }
//                    if (lDataList.getExercise() == null || Objects.equals(lDataList.getExercise().getId(), exerciseId)) {
//                        dataLists.add(lDataList);
//                    }
                }
            }
        }
    }

    /**
     * @param pDataListId
     * @return true if one of the group tab contains at least one data of the data list
     */
    public boolean shouldDisplayDataListForAllTabs(Integer pDataListId) {

        // init
        boolean lShould = false;

        if (xmlGroups != null) {

            // loop on groups to search data
            for (Iterator<XmlGroupTab> iterator = xmlGroups.iterator(); iterator.hasNext() && !lShould; ) {
                XmlGroupTab lGroup = iterator.next();
                lShould = shouldDisplayDataListInTab(pDataListId, lGroup.getId());
            }
        }

        // result
        return lShould;
    }

    /**
     * @param pDataListId id of dataList to test
     * @param pGroupTabId id of tab to test
     * @return true if the group tab contains at least one data of the data list
     */
    public boolean shouldDisplayDataListInTab(Integer pDataListId, String pGroupTabId) {

        XmlGroupTab lTab = getXmlGroup(pGroupTabId);

        DataListDto lDataList = getDataList(pDataListId);

        boolean acceptData = false;

        if (!(lDataList == null || lTab == null)) {

                acceptData = acceptDataListIpOrVp(lDataList, lTab);

                if (!acceptData) {

                    if (logger.isDebugEnabled()) {
                        logger.debug("Reject dataList " + lDataList.getName() + " in tab " + lTab.getTitle() + " (no ip nor vp is matching)");
                    }

                }

        }

        if (acceptData) {
            if (logger.isDebugEnabled()) {
                logger.debug("Accept dataList " + lDataList.getName() + " in tab " + lTab.getTitle());
            }
        }

        return acceptData;

    }

    /**
     * @param pDataListId id of dataList to test
     * @param pCode ip or vp code to test
     * @return true if one the active data lists has the IP/VP passed in parameter
     */
    public boolean isDataInDataList(Integer pDataListId, String pCode) {

        Collection<String> dataListCodes = ipOrVpCodesPerDataList.get(pDataListId);
        boolean result = dataListCodes.contains(pCode);
        return result;

//        boolean lRes = false;
//        try {
//            // Check if the code is in an active list
//
//
//
//            for (DataListDto lDataList : dataLists) {
//                if (Objects.equals(lDataList.getId(), pDataListId)) {
//                    // Check IPs
//                    for (IndicatorDto lIp : lDataList.getIps()) {
//                        if (lIp.getCode().equals(pCode)) {
//                            lRes = true;
//                            break;
//                        }
//                    }
//                    // Check VPs
//                    for (VPDto lVp : lDataList.getVps()) {
//                        if (lVp.getCode().equals(pCode)) {
//                            lRes = true;
//                            break;
//                        }
//                    }
//                }
//            }
//
//        } catch (Exception e) {
//            lRes = false;
//
//        }
//        return lRes;
    }

    /**
     * @return user view right
     */
    protected boolean computeUserViewRight() {

        boolean result = false;
        if (getCurrentUser() != null) {
            // get spea
            SPEADto lSpea = spea;
            if (lSpea == null && lot != null) {
                lSpea = lot.getSpea();
            }
            if (lSpea != null) {
                try {
                    result = userService.checkUserRight(getCurrentUser(), Right.View, lSpea, exercise);
                } catch (SispeaException lEx) {
                    result = false;
                }
            }
        }
        return result;

    }

    /**
     * @return user edit right
     */
    protected boolean computeUserEditRight() {

        boolean result = false;
        if (getCurrentUser() != null) {
            // get spea
            SPEADto lSpea = spea;
            if (lSpea == null && lot != null) {
                lSpea = lot.getSpea();
            }
            if (lSpea != null) {
                try {
                    result = userService.checkUserRight(getCurrentUser(), Right.Edit, lSpea, exercise);
//                    if (result) {
//                        hasEditRight = true;
//                    }
                } catch (SispeaException lEx) {
                    result = false;
                }
            }
        }
        return result;

    }

    /**
     * @return user check right
     */
    protected boolean computeUserCheckRight() {

        boolean result = false;
        if (getCurrentUser() != null) {
            // get spea
            SPEADto lSpea = spea;
            if (lSpea == null && lot != null) {
                lSpea = lot.getSpea();
            }
            if (lSpea != null) {
                try {
                    result = userService.checkUserRight(getCurrentUser(), Right.Check, lSpea, exercise);
//                    if (result) {
//                        hasCheckRight = true;
//                    }
                } catch (SispeaException lEx) {
                    result = false;
                }
            }
        }
        return result;

    }

    /**
     * @return user publish right
     */
    protected boolean computeUserPublishRight() {

        boolean result = false;
        if (getCurrentUser() != null) {
            // get spea
            SPEADto lSpea = spea;
            if (lSpea == null && lot != null) {
                lSpea = lot.getSpea();
            }
            if (lSpea != null) {
                try {
                    result = userService.checkUserRight(getCurrentUser(), Right.Publish, lSpea, exercise);
                } catch (SispeaException lEx) {
                    result = false;
                }
            }
        }
        return result;

    }

    /**
     * @return user unpublish right
     */
    protected boolean computeUserUnpublishRight() {

        boolean result = false;
        if (getCurrentUser() != null) {
            // get spea
            SPEADto lSpea = spea;
            if (lSpea == null && lot != null) {
                lSpea = lot.getSpea();
            }
            if (lSpea != null) {
                try {
                    result = userService.checkUserRight(getCurrentUser(), Right.Unpublish, lSpea, exercise);
                } catch (SispeaException lEx) {
                    result = false;
                }
            }
        }
        return result;

    }

    /**
     * build xml data bean
     *
     * @param pLot
     * @param pCompetenceId
     * @param pHasViewRight
     * @return
     * @throws SispeaException
     */
    public List<XmlGroupTab> buildXmlBean(SPEALotDto pLot, SPEALotDto pOldLot, int pCompetenceId, Boolean pHasViewRight) throws SispeaException {

        //init result
        List<XmlGroupTab> lRes = new ArrayList<>();

        // read xml file
        XmlDataReader xmlReader = new XmlDataReader();
        //XmlData lXmlData = xmlReader.parse("/fr/onema/sispea/struts/referential/action/data.xml");
        XmlData lXmlData = xmlReader.parse("/fr/onema/sispea/struts/referential/action/dataGroups.xml");


        // loop on data group to build display group
        for (XmlGroupTab lXmlGroupTab : lXmlData.getSubGroup()) {
            if (lXmlGroupTab.getCompetence() == null || lXmlGroupTab.getCompetence().equals(pCompetenceId)) {

                // update lXmlGroupTab
                lXmlGroupTab = XmlGroupTab.updateXmlGroupTab(pLot, pOldLot, pCompetenceId, pHasViewRight, lXmlGroupTab, this);

                // add ?
                if (lXmlGroupTab != null && (lXmlGroupTab.isGeneralTab() || !lXmlGroupTab.isEmpty())) {
                    // add !
                    lRes.add(lXmlGroupTab);
                }
            }
        }

        // return result
        return lRes;
    }

//    /**
//     * update lot with group data
//     *
//     * @param pXmlGroupTabs
//     * @param pLot
//     * @throws SispeaException
//     */
//    protected SPEALotDto updateLotWithGroupData(List<XmlGroupTab> pXmlGroupTabs, SPEALotDto pLot) throws SispeaException, SispeaInputException {
//
//        // init
//        SPEALotDto lRes = pLot;
//
//        // loop on tabs to update tab
//        for (XmlGroupTab lXmlGroupTab : pXmlGroupTabs) {
//            // update lot
//            if (lXmlGroupTab.getId().equals(editingGroupId)) {
//                lRes = XmlGroupTab.updateLotWithGroupData(lXmlGroupTab, lRes);
//            }
//        }
//
//        // result
//        return lRes;
//
//    }

    /**
     * sort data detail values by upper bound
     *
     * @param speaLot
     * @return
     */
    public List<Long> getSortedUpperBound(SPEALotDto speaLot) {

        // Get list of DataDetailValues
        List<DataDetailValueDto> lDataDetailValues = new ArrayList<>(speaLot.getDataDetailValues());

        // Sort collection.
        Collections.sort(lDataDetailValues);

        // get Upper Bound
        List<Long> lUpperBoundList = speaLot.getUpperboundList(lDataDetailValues);

        // result
        return lUpperBoundList;
    }


    /**
     * returns true if the real data context should be displayed
     *
     * @return
     */
    public boolean isDisplayRealDataContext() {
        // init
        boolean lRes;

        // test
        lRes = (spea != null && spea.getCompetence().isDrinkingWater());

        // result
        return lRes;
    }

    /**
     * display messages on jsp and returns true if an error was found
     *
     * @param pMessages
     * @return
     */
    protected boolean displayResultMessages(Collection<MessageDto> pMessages) {

        // init
        boolean lErrorFound = false;

        if (isDataTabsDisplayed()) {

            // display messages
            for (MessageDto lMessage : pMessages) {
                // string to display
                String lStrMessage = getText(lMessage.getMessageKey(), lMessage.getArguments());


                if (lMessage.getType() == MessageType.ERROR || lMessage.getType() == MessageType.ERROR_NB) {

                    lErrorFound = (lMessage.getType() == MessageType.ERROR);

                    // main message
                    if (!lMessage.isDisplayOnlyLocally()) {
                        addActionError(lStrMessage);
                    }

                    if (lMessage.getCode() != null && lMessage.getCode().length() > 0) {
                        if (!"P203.3".equals(lMessage.getCode()) &&
                            !"P204.3".equals(lMessage.getCode()) &&
                            !"P205.3".equals(lMessage.getCode()))
                            addFieldError("errorField_" + XmlGroup.escapeCode(lMessage.getCode()), lStrMessage);
                    }
                } else {
                    // main message
                    if (!lMessage.isDisplayOnlyLocally()) {
                        addActionMessage(lStrMessage);
                    }
                    if (lMessage.getCode() != null && lMessage.getCode().length() > 0) {
                        if (lMessage.getType() == MessageType.INFO) {
                            addFieldError("infoField_" + XmlGroup.escapeCode(lMessage.getCode()), lStrMessage);
                        } else {
                            addFieldError("warningField_" + XmlGroup.escapeCode(lMessage.getCode()), lStrMessage);
                        }
                    }
                }
            }
        }

        // result
        return lErrorFound;
    }

    /**
     * @param pGroupId
     * @return the code associated to groupd with passed id
     */
    public String getValueCodeToDetail(String pGroupId) {

        // init
        String lRes = null;

        XmlGroupTableDetail lTableDetail = XmlGroupTab.findTableDetail(pGroupId, xmlGroups, false);

        if (lTableDetail != null) {
            // take the first value
            if (lTableDetail.getGroupTableDetailValues() != null && lTableDetail.getGroupTableDetailValues().size() > 0) {
                lRes = lTableDetail.getGroupTableDetailValues().get(0).getCode();
            }
        }

        // result
        return lRes;
    }

    /**
     * @param pXmlGroups
     * @return the code associated to groupd with passed id
     */
    protected XmlGroupTab getEditedTab(List<XmlGroupTab> pXmlGroups) {
        // init
        XmlGroupTab lRes = null;

        // check tab edited
        for (Iterator<XmlGroupTab> iterator = pXmlGroups.iterator(); iterator.hasNext() && lRes == null; ) {
            XmlGroupTab lXmlGroupTab = iterator.next();

            // update lot
            if (lXmlGroupTab.getId().equals(editingGroupId)) {
                lRes = lXmlGroupTab;
            }
        }

        // result
        return lRes;
    }

    /**
     * @return the list of origins as string formated ready to be displayed in the jsp
     */
    public List<String> getOriginsFormated() {
        List<String> lRes = new ArrayList<>();
        SimpleDateFormat simpleFormat = new SimpleDateFormat(getFormatDate());
        String lKey;
        String lDate;
        String lBy;

        try {
            if (lot != null) {
                for (OriginDto lOrigin : lot.getOrigins()) {
                    lDate = simpleFormat.format(lOrigin.getDate().getTime());
                    List<Object> lParams = new ArrayList<>();
                    lParams.add(lDate);

                    if (lOrigin.getType() == OriginType.inputERU || lOrigin.getType() == OriginType.inputSISEEAU
                        || lOrigin.getType() == OriginType.inputAgency || lOrigin.getType() == OriginType.inputOperator
                        || lOrigin.getType() == OriginType.inputThirdParty) {
                        // This was an administrator import
                        lKey = "fr.onema.sispea.referential.jsp.showSpeaIndicators.field.originType.admin";
                        lBy = getText("fr.onema.sispea.referential.jsp.showSpeaIndicators.field.originType." + lOrigin.getType().name());
                        lParams.add(lBy);
                        // optionnal comment
                        if (lOrigin.getComment() != null) {
                            lParams.add(": " + lOrigin.getComment());
                        } else {
                            lParams.add("");
                        }
                    } else {
                        lKey = "fr.onema.sispea.referential.jsp.showSpeaIndicators.field.originType." + lOrigin.getType().name();
                        List<Object> lByParams = new ArrayList<>();
                        lByParams.add(getText(lOrigin.getUserType().getLabelKey()));
                        lByParams.add(lOrigin.getTerritoryName());
                        lBy = getText("fr.onema.sispea.referential.jsp.showSpeaIndicators.field.originInputType." + lOrigin.getInputType().name(), "", lByParams);
                        lParams.add(lBy);
                    }

                    lRes.add(getText(lKey, "", lParams));
                }
            }

        } catch (Exception e) {
            logException(e);
            return lRes;
        }

        return lRes;
    }


    /**
     * This method generates the RPQS for the SPEA.
     *
     * @param pCurrentUser
     * @param pOrganism
     * @param pExercise
     * @throws SispeaException
     */
    protected void generateRpqs(UserDto pCurrentUser, OrganismDto pOrganism, ExerciseDto pExercise) throws SispeaException {

        String territoryName;
        try {
            // Prepare the export parameters.
            //
            // The export is run with the SPEA territory
            TerritoryDto territory = territoryService.readTerritory(pOrganism);
            territoryName = territory.getName();

            // For huge territories, cut it in smaller ones
            List<TerritoryDto> territories = territoryService.buildExtractTerritoryList(territory, pExercise.getId(), pCurrentUser.getId());

            // View right
            Boolean hasViewRight = userService.checkUserRight(pCurrentUser, Right.View, territory, pExercise);

            ExportParameters parameters = new ExportParameters();
            parameters.setUser(pCurrentUser);
            parameters.setTerritories(territories);
            parameters.setLang(getLocale().getLanguage());
            parameters.setHasViewRight(hasViewRight);
            parameters.setFileType(FileType.RPQS);
            parameters.setWithData(false);
            parameters.setWithReferential(false);
            parameters.setExtractTerritoryName(territoryName);
            parameters.setFileNamePrefix("RPQS_");
            parameters.setFileNameDateFormat("yyyyMMdd_HHmmss");
            parameters.setExercise(pExercise);
            // Do not set any competences. They are not needed to generate the RPQS.
            parameters.setCompetences(null);

            // Generate the export
            fileExchangeService.download(parameters);

            addActionMessage(getText("fr.onema.sispea.jsp.fixRef.showSPEAPresentation.rpqs.generating"));
        } catch (Exception e) {
            logException(e);
            throw new SispeaException(getText("fr.onema.sispea.jsp.fixRef.showSPEAPresentation.rpqs.error", pOrganism.getName()));
        }
    }

//    public DataRatioDto getDisplayRatio() {
//        return speaLotService.getDisplayRatio(lot);
//    }

    /**
     * @return the list of db exercises
     * @throws SispeaException
     */
    protected Map<String, String> buildExerciseList() throws SispeaException {

        TreeMap<String, String> result = exerciseService.readExercisesMapAsc();
        return result;

    }

//    /**
//     * @return the list of db exercises
//     * @throws SispeaException
//     */
//    public Map<String, String> getExerciseList() throws SispeaException {
//        //init tmp not sorted by values map
//        final Map<String, String> tmp = new TreeMap<>();
//
//        // get exercises
//        List<ExerciseDto> lExercises = exerciseService.readExercises();
//
//        for (ExerciseDto lExercise : lExercises) {
//            tmp.put(lExercise.getId().toString(), lExercise.getYear().toString());
//        }
//
//        // Declare a Map with comparator otherwise, sort is made by keys, not by values
//        final Map<String, String> lRes = new TreeMap<>(new Comparator<String>() {
//
//            public int compare(String o1, String o2) {
//                return tmp.get(o1).compareTo(tmp.get(o2));
//            }
//        });
//        lRes.putAll(tmp);
//        return lRes;
//    }

    /**
     * Get Map of data types.
     *
     * @return Map of data types.
     */
    public Map<String, String> getDataTypes() {
        Map<String, String> result = new HashMap<>();

        for (SPEALotInputDataType type : SPEALotInputDataType.values()) {
            result.put(type.name(), getText(type.getLabelKey()));
        }
        return result;
    }

    /**
     * @return Map of autopublish values
     */
    public Map<String, String> getAutoPublishValues() {
        Map<String, String> result = new HashMap<>();

        result.put(Boolean.TRUE.toString(), getText("fr.onema.sispea.referential.datagroup.general.inputAutopublish.true"));
        result.put(Boolean.FALSE.toString(), getText("fr.onema.sispea.referential.datagroup.general.inputAutopublish.false"));

        return result;
    }

    /**
     * @return the time out for display session
     */
    public int getDisplaySessionTimeOut() {
        // init result (minutes)
        int lRes = 15;

        // get session
        HttpServletRequest request = ServletActionContext.getRequest();
        if ((request != null) && (request.getSession() != null)) {
            lRes = request.getSession().getMaxInactiveInterval() / 2 / 60;
        }

        // result
        return lRes;
    }

    /**
     * @return the list of verifications
     */
    public List<Verification> getVerifications() {
        List<Verification> lRes = new ArrayList<>();
        for (int i = 0; i < Verification.values().length; i++) {
            lRes.add(Verification.values()[i]);
        }
        return lRes;
    }

    /**
     * @return the list of reliabilities
     */
    public List<Reliability> getReliabilities() {
        List<Reliability> lRes = new ArrayList<>();
        for (int i = 0; i < Reliability.values().length; i++) {
            lRes.add(Reliability.values()[i]);
        }
        return lRes;
    }

    /**
     * @return Map of reliability types.
     */
    public Map<String, String> getReliabilityTypes() {
        // init result
        Map<String, String> result = new TreeMap<>();

        // build hash map
        for (Reliability type : Reliability.values()) {
            result.put(type.name(), getText(type.getShortLabelKey()));
        }

        // result
        return result;
    }

    public Map<String, String> getTabList(int pTabId, String pTabName) {
        Map<String, String> tabList = new TreeMap<>();

        tabList.put(Integer.valueOf(pTabId).toString(), pTabName);

        return tabList;
    }

    public Integer getExerciseId() {
        return exerciseId;
    }

    public void setExerciseId(Integer pExerciseId) {
        exerciseId = pExerciseId;
    }

    public void setStrExerciseId(String pExerciseId) {
        exerciseId = Integer.valueOf(pExerciseId);
    }

    public Integer getChosenExerciseId() {
        return exerciseId;
    }

    public void setChosenExerciseId(Integer pChosenExerciseId) {
        exerciseId = pChosenExerciseId;
    }

    public Integer getId() {
        return id;
    }

    public void setId(Integer pId) {
        id = pId;
    }

    public Integer getSpeaId() {
        return id;
    }

    public void setSpeaId(Integer pSpeaId) {
        id = pSpeaId;
    }

    public SPEADto getSpea() {
        return spea;
    }

    public void setSpea(SPEADto pSpea) {
        spea = pSpea;
    }

    public List<XmlGroupTab> getXmlGroups() {
        return xmlGroups;
    }

    public void setXmlGroups(List<XmlGroupTab> pXmlGroups) {
        xmlGroups = pXmlGroups;
    }

    public XmlGroupTab getXmlGroup(String pGroupTabId) {
        // init
        XmlGroupTab lGroup = null;

        // loop on data list
        if (xmlGroups != null) {
            for (int i = 0; i < xmlGroups.size() && lGroup == null; i++) {
                XmlGroupTab lTmp = xmlGroups.get(i);
                if (lTmp.getId().equals(pGroupTabId)) {
                    lGroup = lTmp;
                }
            }
        }

        // result
        return lGroup;
    }

    public SPEALotDto getLot() {
        return lot;
    }

    public void setLot(SPEALotDto pLot) {
        lot = pLot;
    }

    public boolean isHasViewRight() throws SispeaException {
//        if (hasViewRight == null) {
//            computeUserViewRight();
//        }
        return hasViewRight;
    }

    public void setHasViewRight(boolean pHasViewRight) {
        hasViewRight = pHasViewRight;
    }

    public boolean isHasEditRight() throws SispeaException {
//        if (hasEditRight == null) {
//            computeUserEditRight();
//        }
        return hasEditRight;
    }

    public boolean isHasCheckRight() throws SispeaException {
//        if (hasCheckRight == null) {
//            computeUserCheckRight();
//        }
        return hasCheckRight;
    }

    public boolean isHasPublishRight() {
//        if (hasPublishRight == null) {
//            computeUserPublishRight();
//        }
        return hasPublishRight;
    }

    public void setHasPublishRight(boolean pHasPublishRight) {
        hasPublishRight = pHasPublishRight;
    }

    public boolean isHasUnpublishRight() {
//        if (hasUnpublishRight == null) {
//            computeUserUnpublishRight();
//        }
        return hasUnpublishRight;
    }

    public void setHasUnpublishRight(boolean pHasUnpublishRight) {
        hasUnpublishRight = pHasUnpublishRight;
    }

    public ExerciseDto getExercise() {
        return exercise;
    }

    public void setExercise(ExerciseDto pExercise) {
        exercise = pExercise;
    }

    public String getEditingGroupId() {
        return editingGroupId;
    }

    public void setEditingGroupId(String pEditingGroupId) {
        editingGroupId = pEditingGroupId;
    }

    public boolean isEditing() {
        return editingGroupId != null;
    }

    public String getGeneralComment() {
        return generalComment;
    }

    public void setGeneralComment(String pGeneralComment) {
        generalComment = pGeneralComment;
    }

    public boolean isEditingComment() {
        return editingComment;
    }

    public void setEditingComment(boolean pEditingComment) {
        editingComment = pEditingComment;
    }

    public boolean isEditingDataToInput() {
        return editingDataToInput;
    }

    public void setEditingDataToInput(boolean pEditingDataToInput) {
        editingDataToInput = pEditingDataToInput;
    }

    public String getDcCode() {
        return dcCode;
    }

    public XmlDisplayGroup getDataContextGroup() {
        return dataContextGroup;
    }

    public void setDataContextGroup(XmlDisplayGroup pDataContextGroup) {
        dataContextGroup = pDataContextGroup;
    }

    public boolean isUnpublishable() throws SispeaException {
//        unpublishable = speaLotService.isSpeaLotUnpublishable(getExerciseId(), getSpeaId());
        return unpublishable;
    }

    public boolean isEditingAutopublish() {
        return editingAutopublish;
    }

    public void setEditingAutopublish(boolean pEditingAutopublish) {
        editingAutopublish = pEditingAutopublish;
    }

    public String getUpperBound() {
        return upperBound;
    }

    public void setUpperBound(String pUpperBound) {
        upperBound = pUpperBound;
    }

    public Long getMaxUpperBound() {
        return DataDetailValueDto.DEFAULT_UPPER_BOUND;
    }

    public void setMaxUpperBound(String pUpperBound) {
        upperBound = pUpperBound;
    }

    public String getUpperBoundVP177() {
        return upperBoundVP177;
    }

    public void setUpperBoundVP177(String pUpperBoundVP177) {
        upperBoundVP177 = pUpperBoundVP177;
    }

    public String getUpperBoundVP178() {
        return upperBoundVP178;
    }

    public void setUpperBoundVP178(String pUpperBoundVP178) {
        upperBoundVP178 = pUpperBoundVP178;
    }

    public Long getLowerBound() {
        return lowerBound;
    }

    public void setLowerBound(Long pLowerBound) {
        lowerBound = pLowerBound;
    }

    public String getGroupToEditId() {
        return groupToEditId;
    }

    public void setGroupToEditId(String pGroupToEditId) {
        groupToEditId = pGroupToEditId;
    }

    public Long getMaxSliceUpperBound() {
        return DataDetailValueDto.DEFAULT_UPPER_BOUND;
    }

    public SPEALotDto getOldLot() {
        return oldLot;
    }

    public void setOldLot(SPEALotDto pOldLot) {
        oldLot = pOldLot;
    }

    public List<DataListDto> getDataLists() {
        return dataLists;
    }

    public void setDataLists(List<DataListDto> pDataLists) {
        dataLists = pDataLists;
    }

    public DataListDto getDataList(Integer pDataListId) {

        // init
        DataListDto lList = null;

        // loop on data list
        if (dataLists != null) {
            for (int i = 0; i < dataLists.size() && lList == null; i++) {
                DataListDto lTmpList = dataLists.get(i);
                if (lTmpList.getId().equals(pDataListId)) {
                    lList = lTmpList;
                }
            }
        }

        // result
        return lList;
    }

    public DrinkingWaterNetworkSchemaBean getDwnSchema() {
        return dwnSchema;
    }

    public String getSlicesToDeleteVP177() {
        return slicesToDeleteVP177;
    }

    public void setSlicesToDeleteVP177(String pSlicesToDeleteVP177) {
        slicesToDeleteVP177 = pSlicesToDeleteVP177;
    }

    public String getSlicesTAddVP177() {
        return slicesToAddVP177;
    }

    public void setSlicesToAddVP177(String pSlicesToAddVP177) {
        slicesToAddVP177 = pSlicesToAddVP177;
    }

    public String getSlicesToDeleteVP178() {
        return slicesToDeleteVP178;
    }

    public void setSlicesToDeleteVP178(String pSlicesToDeleteVP178) {
        slicesToDeleteVP178 = pSlicesToDeleteVP178;
    }

    public String getSlicesTAddVP178() {
        return slicesToAddVP178;
    }

    public void setSlicesToAddVP178(String pSlicesToAddVP178) {
        slicesToAddVP178 = pSlicesToAddVP178;
    }

    public String getEmptyData() {
        return emptyData;
    }

    public void setEmptyData(String pEmptyData) {
        emptyData = pEmptyData;
    }

    public boolean isDataTabsDisplayed() {

        // init
        boolean lRes;

        // IF YOU MODIFY THIS, PLEASE MODIFY ExchangeParameters.canViewLot()
        try {
            lRes = lot != null
                   && (
                           (lot.getLotStatus() == SPEALotStatus.Published && isHasViewRight())
                           || (lot.getLotStatus() == SPEALotStatus.Verified && (isHasPublishRight() || isHasCheckRight()))
                           || (lot.getLotStatus() != SPEALotStatus.Published && (isHasEditRight() || isHasCheckRight()))
                   );
        } catch (SispeaException e) {
            lRes = false;
        }

        // result
        return lRes;

    }

    public IndicatorService getIndicatorService() {
        return indicatorService;
    }

    public PasswordService getPasswordService() {
        return passwordService;
    }

    public FormCalculateService getFormCalculateService() {
        return formCalculateService;
    }

    public ServiceMessages getServiceMessages() {
        return serviceMessages;
    }

    public StrutsMessages getStrutsMessages() {
        return strutsMessages;
    }

    public Map<String, String> getExerciseList() {
        return exerciseList;
    }

    protected Map<String, String> loadExcerciseList() throws SispeaException {
        TreeMap<String, String> exerciseList = exerciseService.readExercisesMapAsc();
        return ImmutableMap.copyOf(exerciseList);
    }

    protected boolean acceptDataListCompetence(DataListDto lDataList, CompetenceDto speaComptence) {

        boolean accept = true;
        CompetenceDto dataListCompetence = lDataList.getCompetence();
        if (dataListCompetence!=null) {

            accept = Objects.equals(dataListCompetence , speaComptence);

        }

        return accept;
    }

    protected boolean acceptDataListFromExercice(DataListDto lDataList, ExerciseDto exercise) {

        boolean accept = true;
        if (lDataList.getFromExercise() != null) {
            Short beginYear = lDataList.getFromExercise().getYear();
            accept = beginYear <= exercise.getYear();
        }
        return accept;

    }

    protected boolean acceptDataListToExercice(DataListDto lDataList, ExerciseDto exercise) {

        boolean accept = true;
        if (lDataList.getToExercise() != null) {
            Short endYear = lDataList.getToExercise().getYear();
            accept = exercise.getYear() <= endYear;
        }
        return accept;

    }

    protected boolean acceptDataListIpOrVp(DataListDto lDataList, XmlGroupTab lTab) {

        boolean accept = false;

        // the data to search
        List<DataDto> lData = new ArrayList<>();
        lData.addAll(lDataList.getIps());
        lData.addAll(lDataList.getVps());

        for (DataDto dataDto : lData) {
            if (lTab.containsData(dataDto)) {
                accept = true;
                break;
            }
        }

        return accept;

    }

}
