package fr.onema.sispea.struts.admin.action;

/*
 * #%L
 * SISPEA web application
 * %%
 * Copyright (C) 2014 - 2015 ONEMA
 * %%
 * ONEMA - Tous droits réservés
 * #L%
 */


import com.opensymphony.xwork2.Action;
import fr.onema.sispea.SispeaException;
import fr.onema.sispea.service.data.IndicatorDto;
import fr.onema.sispea.service.data.IndicatorService;
import fr.onema.sispea.service.data.IndicatorType;
import fr.onema.sispea.service.referential.CompetenceDto;
import fr.onema.sispea.service.referential.ReferentialService;
import fr.onema.sispea.service.user.Right;
import fr.onema.sispea.struts.AbstractSispeaAction;
import fr.onema.sispea.struts.common.menu.MenuConstants;
import fr.onema.sispea.util.constants.NumberUtils;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * This action handles the edition of indicators limits.
 *
 * @author CS
 */
public class EditIndicatorsLimitsAction extends AbstractSispeaAction {

    private static final long serialVersionUID = 1L;

    /**
     * the constants for group editing
     */
    private static final Integer EDITING_ID_EMPTY = -1;

    private static final Logger logger = Logger.getLogger(EditIndicatorsLimitsAction.class);

    @Autowired
    protected transient IndicatorService indicatorService;

    @Autowired
    protected transient ReferentialService referentialService;

    /**
     * the chosen competence id
     */
    protected Integer chosenCompetenceId;

    /**
     * list of competence
     */
    protected List<CompetenceDto> competences;

    /**
     * The competence adisplayed
     */
    protected CompetenceDto competence;

    /**
     * the performance indicators list
     */
    protected List<IndicatorDto> pis = new ArrayList<>();

    /**
     * the descriptive indicator list
     */
    protected List<IndicatorDto> dis = new ArrayList<>();

    /**
     * The data currently edited
     */
    protected Integer editingDataId = EDITING_ID_EMPTY;

    /**
     * The new min value
     */
    protected String newMinValue;

    /**
     * The new max value
     */
    protected String newMaxValue;

    /**
     * The new min value
     */
    protected String newCriticMinValue;

    /**
     * The new max value
     */
    protected String newCriticMaxValue;

    /**
     * Number of decimal
     */
    protected String newDecimalValue;

    protected Map<String, String> competenceList;

    @Override
    protected String doExecute() throws Exception {

        //init
        String lRes = Action.SUCCESS;

        // debug message
        if (logger.isDebugEnabled()) {
            logger.debug("edit indicators limits");
        }

        // open menu
        openMenu(MenuConstants.MENU_KEY_WARNINGLEVELS);

        // check rights
        if (!checkUserRight(Right.Manage)) {
            // message
            throw new SispeaException("fr.onema.sispea.autentication.notAllowed");
        }

        try {

            // competence
            checkCompetence();

            // load indicators
            loadIndicators();

            if (submitValue != null && submitValue.equals(getText("fr.onema.sispea.indicatorsLimits.jsp.table.submit.cancel"))) {

                // cancel
                editingDataId = EDITING_ID_EMPTY;

            } else if (submitValue != null && submitValue.equals(getText("fr.onema.sispea.indicatorsLimits.jsp.table.submit.save"))) {
                // save

                //check min value
                Double lMinValue = checkInputValue(newMinValue,
                                                   "fr.onema.sispea.indicatorsLimits.error.wrongMinValue",
                                                   "fr.onema.sispea.indicatorsLimits.error.emptyMinValue",
                                                   "newMinValue", true);

                //check max value
                Double lMaxValue = checkInputValue(newMaxValue,
                                                   "fr.onema.sispea.indicatorsLimits.error.wrongMaxValue",
                                                   "fr.onema.sispea.indicatorsLimits.error.emptyMaxValue",
                                                   "newMaxValue", false);

                //check critical min value
                Double lCriticMinValue = checkInputValue(newCriticMinValue,
                                                         "fr.onema.sispea.indicatorsLimits.error.wrongCriticMinValue",
                                                         "fr.onema.sispea.indicatorsLimits.error.emptyCriticMinValue",
                                                         "newCriticMinValue", true);

                //check critical max value
                Double lCriticMaxValue = checkInputValue(newCriticMaxValue,
                                                         "fr.onema.sispea.indicatorsLimits.error.wrongCriticMaxValue",
                                                         "fr.onema.sispea.indicatorsLimits.error.emptyCriticMaxValue",
                                                         "newCriticMaxValue", false);

                // -----|-----------|-----------|-----------|-------------
                // ERR    |    WARN    |    OK        |    WARN    | ERR
                //         CRITMIN        WARNMIN        WARNMAX        CRITMAX
                // the min warning must be >= than crit min
                if (lMinValue < lCriticMinValue) {
                    String lMessage = "fr.onema.sispea.indicatorsLimits.error.minCompare";
                    addFieldError("newMinValue", getText(lMessage));
                    addFieldError("newCriticMinValue", getText(lMessage));
                    throw new SispeaException(lMessage);
                }
                // the min warning must be < than max warn
                if (lMinValue >= lMaxValue) {
                    String lMessage = "fr.onema.sispea.indicatorsLimits.error.minMaxCompare";
                    addFieldError("newMinValue", getText(lMessage));
                    addFieldError("newMaxValue", getText(lMessage));
                    throw new SispeaException(lMessage);
                }
                // the max warning must be =< than crit max
                if (lMaxValue > lCriticMaxValue) {
                    String lMessage = "fr.onema.sispea.indicatorsLimits.error.maxCompare";
                    addFieldError("newMaxValue", getText(lMessage));
                    addFieldError("newCriticMaxValue", getText(lMessage));
                    throw new SispeaException(lMessage);
                }
                // the min critic must be < than max critic
                if (lCriticMinValue >= lCriticMaxValue) {
                    String lMessage = "fr.onema.sispea.indicatorsLimits.error.minMaxCriticCompare";
                    addFieldError("newCriticMinValue", getText(lMessage));
                    addFieldError("newCriticMaxValue", getText(lMessage));
                    throw new SispeaException(lMessage);
                }

                //check decimal value
                Integer lDecimal;
                try {
                    if (newDecimalValue == null || newDecimalValue.length() == 0) {
                        newDecimalValue = IndicatorDto.DEFAULT_DECIMAL_VALUE.toString();
                    }
                    lDecimal = Integer.valueOf(newDecimalValue);
                } catch (NumberFormatException lEx) {
                    String lMessage = "fr.onema.sispea.indicatorsLimits.error.wrongDecimalValue";
                    addFieldError("newDecimalValue", getText(lMessage));
                    throw new SispeaException(lMessage);
                }

                // update indicator
                IndicatorDto lIndicator = getIndicator(editingDataId);
                if (lIndicator == null) {
                    throw new SispeaException("fr.onema.sispea.indicatorsLimits.error.save");
                }
                lIndicator.setWarningMin(lMinValue);
                lIndicator.setWarningMax(lMaxValue);
                lIndicator.setCriticMin(lCriticMinValue);
                lIndicator.setCriticMax(lCriticMaxValue);
                lIndicator.setDecimal(lDecimal);

                // save to Db
                indicatorService.saveWarningsMinMax(lIndicator);
                indicatorService.saveCriticMinMax(lIndicator);
                indicatorService.saveDecimal(lIndicator);

                // unedit
                editingDataId = EDITING_ID_EMPTY;

                // message
                addActionMessage(getText("fr.onema.sispea.indicatorsLimits.message.success"));

            } else if (isEditing()) {

                // update min and max value
                IndicatorDto lIndicator = getIndicator(editingDataId);
                if (lIndicator != null) {
                    newMaxValue = lIndicator.getWarningMaxStr();
                    newMinValue = lIndicator.getWarningMinStr();
                    newDecimalValue = lIndicator.getDecimal().toString();
                    newCriticMaxValue = lIndicator.getCriticMaxStr();
                    newCriticMinValue = lIndicator.getCriticMinStr();
                }

            }

        } catch (SispeaException lEx) {
            lRes = Action.INPUT;
            String lMessage = getText(lEx.getMessage());
            addActionError(lMessage);
        }

        // result
        return lRes;
    }

    /**
     * Check the string value and returns a double of this
     *
     * @param pValueToCheck
     * @param pWrongValueKey
     * @param pEmptyValueKey
     * @param pFieldName
     * @return
     * @throws SispeaException
     */
    protected Double checkInputValue(String pValueToCheck, String pWrongValueKey, String pEmptyValueKey, String pFieldName, boolean pRequired) throws SispeaException {

        //check min value
        Double lResultValue = null;
        if (pValueToCheck != null && pValueToCheck.length() > 0) {
            try {
                lResultValue = NumberUtils.stringToDouble(pValueToCheck);
            } catch (NumberFormatException lEx) {
                addFieldError(pFieldName, getText(pWrongValueKey));
                throw new SispeaException(pWrongValueKey);
            }
        }
        if (lResultValue == null) {
            if (pRequired) {
                addFieldError(pFieldName, getText(pEmptyValueKey));
                throw new SispeaException(pEmptyValueKey);
            }
            lResultValue = IndicatorDto.EMPTY_MAX;
        }

        // result
        return lResultValue;

    }

    /**
     * check competence params
     *
     * @throws SispeaException
     */
    protected void checkCompetence() throws SispeaException {

        // load competences
        competences = referentialService.readCompetences();

        // check competence
        if (chosenCompetenceId == null) {
            chosenCompetenceId = CompetenceDto.COMPETENCE_DRINKING_WATER_ID;
        } else {
            competence = referentialService.readCompetence(chosenCompetenceId);
            if (competence == null) {
                throw new SispeaException("fr.onema.sispea.indicatorsLimits.error.wrongCompetenceId");
            }
        }

        competenceList = new HashMap<>();
        if (competences != null) {
            for (CompetenceDto lCompetence : competences) {
                competenceList.put(lCompetence.getId().toString(), lCompetence.getName());
            }
        }

    }

    /**
     * loads indicator
     *
     * @throws SispeaException
     */
    protected void loadIndicators() throws SispeaException {

        // load indicators
        List<IndicatorDto> lIndicators = indicatorService.readIndicators(chosenCompetenceId, null);

        // split in two list
        for (IndicatorDto lIndicator : lIndicators) {
            if (lIndicator.getType() == IndicatorType.Descriptive) {
                dis.add(lIndicator);
            } else {
                pis.add(lIndicator);
            }
        }

        // sort
        Collections.sort(dis);
        Collections.sort(pis);
    }

    public Map<String, String> getCompetenceList() {
//        competenceList = new HashMap<>();
//        if (competences != null) {
//            // add competences
//            for (CompetenceDto lCompetence : competences) {
//                competenceList.put(lCompetence.getId().toString(), lCompetence.getName());
//            }
//        }
        return competenceList;
    }


    /**
     * returns the indicator corresponding to id. Null if not found
     *
     * @param pId
     * @return
     */
    protected IndicatorDto getIndicator(Integer pId) {

        // init
        IndicatorDto lRes = null;

        // parse indicators to find edited one
        List<IndicatorDto> lIndicators = new ArrayList<>();
        lIndicators.addAll(dis);
        lIndicators.addAll(pis);
        for (Iterator<IndicatorDto> iterator = lIndicators.iterator(); iterator.hasNext() && lRes == null; ) {
            IndicatorDto lIndicator = iterator.next();
            if (lIndicator.getId().equals(pId)) {
                lRes = lIndicator;
            }
        }

        // result
        return lRes;
    }

    public String getEmptyMax() {
        return NumberUtils.doubleToString(IndicatorDto.EMPTY_MAX, 0, false);
    }

    public boolean isEditing() {
        return !EDITING_ID_EMPTY.equals(editingDataId);
    }

    public Integer getChosenCompetenceId() {
        return chosenCompetenceId;
    }

    public void setChosenCompetenceId(Integer pChosenCompetenceId) {
        chosenCompetenceId = pChosenCompetenceId;
    }

    public List<CompetenceDto> getCompetences() {
        return competences;
    }

    public void setCompetences(List<CompetenceDto> pCompetences) {
        competences = pCompetences;
    }

    public List<IndicatorDto> getPis() {
        return pis;
    }

    public List<IndicatorDto> getDis() {
        return dis;
    }

    public Integer getEditingDataId() {
        return editingDataId;
    }

    public void setEditingDataId(Integer pEditingDataId) {
        editingDataId = pEditingDataId;
    }

    public CompetenceDto getCompetence() {
        return competence;
    }

    public void setCompetence(CompetenceDto pCompetence) {
        competence = pCompetence;
    }

    public String getNewMinValue() {
        return newMinValue;
    }

    public void setNewMinValue(String pNewMinValue) {
        newMinValue = pNewMinValue;
    }

    public String getNewMaxValue() {
        return newMaxValue;
    }

    public void setNewMaxValue(String pNewMaxValue) {
        newMaxValue = pNewMaxValue;
    }

    public String getNewDecimalValue() {
        return newDecimalValue;
    }

    public void setNewDecimalValue(String pDecimal) {
        newDecimalValue = pDecimal;
    }

    public String getNewCriticMinValue() {
        return newCriticMinValue;
    }

    public void setNewCriticMinValue(String pNewCriticMinValue) {
        newCriticMinValue = pNewCriticMinValue;
    }

    public String getNewCriticMaxValue() {
        return newCriticMaxValue;
    }

    public void setNewCriticMaxValue(String pNewCriticMaxValue) {
        newCriticMaxValue = pNewCriticMaxValue;
    }

}
