package fr.onema.sispea.chart;

/*
 * #%L
 * SISPEA web application
 * %%
 * Copyright (C) 2014 - 2015 ONEMA
 * %%
 * ONEMA - Tous droits réservés
 * #L%
 */


import fr.onema.sispea.service.ServiceMessages;
import fr.onema.sispea.service.synthesis.ChartValueDto;
import fr.onema.sispea.service.synthesis.ScatterChartDto;
import fr.onema.sispea.service.synthesis.ScatterChartFactory;
import fr.onema.sispea.struts.synthesis.bean.BoxAndWhiskerValueBean;
import fr.onema.sispea.struts.synthesis.bean.CategoryChartBean;
import fr.onema.sispea.struts.synthesis.bean.CategoryValueBean;
import fr.onema.sispea.struts.synthesis.bean.PieChartBean;
import fr.onema.sispea.struts.synthesis.bean.PieValueBean;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.CategoryAxis;
import org.jfree.chart.axis.CategoryLabelPositions;
import org.jfree.chart.axis.NumberAxis;
import org.jfree.chart.plot.CategoryPlot;
import org.jfree.chart.plot.PlotOrientation;
import org.jfree.chart.renderer.category.BoxAndWhiskerRenderer;
import org.jfree.chart.renderer.xy.XYBoxAndWhiskerRenderer;
import org.jfree.chart.servlet.ChartDeleter;
import org.jfree.chart.servlet.ServletUtilities;
import org.jfree.data.category.DefaultCategoryDataset;
import org.jfree.data.general.DefaultPieDataset;
import org.jfree.data.statistics.DefaultBoxAndWhiskerCategoryDataset;
import org.jfree.data.xml.DatasetReader;
import org.springframework.beans.factory.annotation.Autowired;
import fr.onema.sispea.service.synthesis.chart.CategoryChart;
import fr.onema.sispea.service.synthesis.chart.PieChart;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.awt.Font;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;

/**
 * <p>
 * The ChartServlet object
 * </p>
 *
 * Cette classe est un dérivé de la classe uk.org.axioma.servlet.ChartServlet de la librairie JWebChart.
 *
 * @author <a href="mailto:luke.trevorrow@gmail.com">Luke Trevorrow</a>
 * @version 0.3
 */
public class ChartServlet extends HttpServlet {

    private static final long serialVersionUID = 1L;

    /**
     * the default font title size
     */
    private static final float FONT_TITLE_SIZE = 14;

    //FIXME tchemit-20150626 Vérifier que l'injection fonctionne bien
    @Autowired
    protected transient ServiceMessages serviceMessages;

    @Override
    public void service(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {

        HttpSession session = request.getSession();
        //java.util.Enumeration en = request.getParameterNames();
        Map<String, String[]> map = new HashMap<>(request.getParameterMap());
        JFreeChart jfc = null;

        String series = null;
        int width = 300, height = 200;
        
        /*PrintWriter out = response.getWriter();
        response.setContentType("text/html");
        out.println("<H4>");*/

        // go through the hash map and extract the core values.
        // extract the rest in the createChart methods
        if (map.containsKey("Series")) {
            // use the getParameter method rather than the map to get the string array as I am sure this will be faster
            series = request.getParameter("Series");
            map.remove("Series");
            //out.println("<br>Graph : " + series);
        }

        if (map.containsKey("Width")) {
            width = Integer.valueOf(request.getParameter("Width"));
            map.remove("Width");
            //out.println("<br>Width : " + width);
        }

        if (map.containsKey("Height")) {
            height = Integer.valueOf(request.getParameter("Height"));
            map.remove("Height");
            //out.println("<br>Height : " + height);
        }

        // create the chart
        if (series != null) {
            if (series.equalsIgnoreCase("Pie")) {
                jfc = createPie(map, request);
            } else if (series.equalsIgnoreCase("Category")) {
                jfc = createCategory(map, request);
            } else if (series.equalsIgnoreCase("BoxAndWhisker")) {
                jfc = createBoxAndWhisker(map, request);
            } else if (series.equalsIgnoreCase("XY")) {
                jfc = createXY(map, request);
            }
        }

        // change title font size
        Font lNewFont = jfc.getTitle().getFont().deriveFont(FONT_TITLE_SIZE);
        jfc.getTitle().setFont(lNewFont);

        // save the image for display
        String filename = ServletUtilities.saveChartAsPNG(jfc, width, height, null);
        //out.println(filename);

        // Check the file exists
        File file = new File(System.getProperty("java.io.tmpdir"), filename);
        if (!file.exists()) {
            throw new ServletException(
                    "File '" + file.getAbsolutePath() + "' does not exist"
            );
        }

        //  Check that the graph being served was created by the current user
        //  or that it begins with "public"
        boolean isChartInUserList = false;
        ChartDeleter chartDeleter = (ChartDeleter) session.getAttribute(
                "JFreeChart_Deleter"
        );
        if (chartDeleter != null) {
            isChartInUserList = chartDeleter.isChartAvailable(filename);
        }

        boolean isChartPublic = false;
        if (filename.length() >= 6) {
            if (filename.substring(0, 6).equals("public")) {
                isChartPublic = true;
            }
        }

        boolean isOneTimeChart = false;
        if (filename.startsWith(ServletUtilities.getTempOneTimeFilePrefix())) {
            isOneTimeChart = true;
        }

        if (isChartInUserList || isChartPublic || isOneTimeChart) {
            //  Serve it up
            ServletUtilities.sendTempFile(file, response);
            if (isOneTimeChart) {
                file.delete();
            }
        } else {
            throw new ServletException("Chart image not found");
        }
    }

    /**
     * Create pie chart method.
     *
     * @param data which can consist of elements such as:
     *             <ul>
     *             <li> Type - the type of chart: either 2D, 3D, or Ring.</li>
     *             <li> Title - the title of the chart.</li>
     *             <li> XMLFile - the XML file name if applicable.</li>
     *             <li> data - the data to plot. Format is name=value&name=value, eg: Fish=70&Chips=66.</li>
     *             </ul>
     * @param req  which is the Http Servlet Request
     * @return JFreeChart
     * @throws ServletException
     */
    private JFreeChart createPie(Map<String, String[]> data, HttpServletRequest req)
            throws ServletException {
        PieChart pie = new PieChart();

        // set the title for the Pie Chart
        if (data.containsKey("Title")) {
            pie.title = req.getParameter("Title");
            data.remove("Title");
        }

        // what type?
        if (data.containsKey("Type")) {
            String type = req.getParameter("Type");
            if (type.equalsIgnoreCase("2D")) {
                pie.type = PieChart.TYPE2D;
            } else if (type.equalsIgnoreCase("3D")) {
                pie.type = PieChart.TYPE3D;
            } else if (type.equalsIgnoreCase("Ring")) {
                pie.type = PieChart.TYPERING;
            }
            data.remove("Type");
        }

        // get the chart data from an XML file or...
        if (data.containsKey("XMLFile")) {
            URL url = getClass().getResource(req.getParameter("XMLFile"));
            try {
                java.io.InputStream inputstream = url.openStream();
                pie.dataset = (DefaultPieDataset) DatasetReader.readPieDatasetFromXML(inputstream);
            } catch (IOException ioexception) {
                System.out.println(ioexception.getMessage());
            }
            data.remove("XMLFile");
        } else {
            // Begin SISPEA
            //
            // Get the data from the session
            PieChartBean pieChart = (PieChartBean) req.getSession().getAttribute("PIE_CHART");
            req.getSession().removeAttribute("PIE_CHART");

            for (ChartValueDto chartValueBean : pieChart.getValues()) {
                PieValueBean value = (PieValueBean) chartValueBean;
                pie.dataset.setValue(value.getName(), new Double(value.getValue()));
            }

            /*
            for (Iterator<Entry<String, String[]>> it=data.entrySet().iterator(); it.hasNext(); ) {
                Map.Entry entry = it.next();
                String key = (String) entry.getKey();
                String value[] = (String[]) entry.getValue();
                pie.dataset.setValue(key,new Double(value[0]));
                //out.println("<br>Key : " + key);
                //out.println("  Value : " + value[0]);
            }
            */
            // End SISPEA
        }

        // create pie chart
        JFreeChart chart = pie.getChart();
        return chart;
    }

    /**
     * Create category chart method.
     *
     * @param map which can consist of elements such as:
     *            <ul>
     *            <li> Type - the type of chart: either Bar, Bar3D, Area, Line, or Line3D.</li>
     *            <li> Title - the title of the chart.</li>
     *            <li> XLabel - the domain title.</li>
     *            <li> YLabel - the range title.<li>
     *            <li> Orientation - the orientation of the chart, either H or V.<li>
     *            <li> Range - the upper and lower value of the range. Format is Range=lower,upper, e.g. Range=68,72.</li>
     *            <li> XMLFile - the XML file name if applicable.</li>
     *            <li> data - the data to plot. Format is name=value&name=value, eg: Fish=70&Chips=66.</li>
     *            </ul>
     * @param req which is the Http Servlet Request
     * @return JFreeChart
     * @throws ServletException
     */
    private JFreeChart createCategory(Map<String, String[]> map, HttpServletRequest req)
            throws ServletException {
        CategoryChart bar = new CategoryChart();

        // set the title for the Category Chart
        if (map.containsKey("Title")) {
            bar.title = req.getParameter("Title");
            map.remove("Title");
        }

        // what type? (default is Bar)
        if (map.containsKey("Type")) {
            String type = req.getParameter("Type");
            if (type.equalsIgnoreCase("Bar")) {
                bar.type = CategoryChart.TYPEBAR;
            } else if (type.equalsIgnoreCase("Bar3D")) {
                bar.type = CategoryChart.TYPEBAR3D;
            } else if (type.equalsIgnoreCase("Area")) {
                bar.type = CategoryChart.TYPEAREA;
            } else if (type.equalsIgnoreCase("Line")) {
                bar.type = CategoryChart.TYPELINE;
            } else if (type.equalsIgnoreCase("Line3D")) {
                bar.type = CategoryChart.TYPELINE3D;
            }
            map.remove("Type");
        }

        // set the range
        if (map.containsKey("YLabel")) {
            bar.range = req.getParameter("YLabel");
            map.remove("YLabel");
        }
        // set the domain
        if (map.containsKey("XLabel")) {
            bar.domain = req.getParameter("XLabel");
            map.remove("XLabel");
        }

        // set the orientation
        if (map.containsKey("Orientation")) {
            String orientation = req.getParameter("Orientation");
            if (orientation.equalsIgnoreCase("V")) {
                bar.orientation = PlotOrientation.VERTICAL;
            } else if (orientation.equalsIgnoreCase("H")) {
                bar.orientation = PlotOrientation.HORIZONTAL;
            }
            map.remove("Orientation");
            //out.println("<br>Orientation : " + orientation);
        }

        // set the lower and upper range
        if (map.containsKey("Range")) {
            String rangeArray[] = req.getParameter("Range").split(",");
            try {
                bar.lower = new Double(rangeArray[0]);
            } catch (Exception lEx) {
                bar.lower = null;
            }
            try {
                bar.upper = new Double(rangeArray[1]);
            } catch (Exception lEx) {
                bar.upper = null;
            }
            map.remove("Range");
            //out.println("<br>Range : " + range);
        }

        // get the chart data from an XML file or...
        if (map.containsKey("XMLFile")) {
            URL url = getClass().getResource(req.getParameter("XMLFile"));
            try {
                java.io.InputStream inputstream = url.openStream();
                bar.dataset = (DefaultCategoryDataset) DatasetReader.readCategoryDatasetFromXML(inputstream);
            } catch (IOException ioexception) {
                System.out.println(ioexception.getMessage());
            }
            map.remove("XMLFile");
        } else {
            // Begin SISPEA
            //
            // Get the data from the session
            CategoryChartBean categoryChart = (CategoryChartBean) req.getSession().getAttribute("CATEGORY_CHART");
            req.getSession().removeAttribute("CATEGORY_CHART");
            for (ChartValueDto chartValueBean : categoryChart.getValues()) {
                CategoryValueBean value = (CategoryValueBean) chartValueBean;
                bar.dataset.setValue(new Double(value.getValue()), value.getName(), value.getLabel());
            }

            // chart title
            if (categoryChart.getTitle() != null) {
                bar.title = categoryChart.getTitle();
            }

            // XLabel
            if (categoryChart.getXLabel() != null) {
                bar.domain = categoryChart.getXLabel();
            }

            // YLabel
            if (categoryChart.getYLabel() != null) {
                bar.range = categoryChart.getYLabel();
            }

            /*
            // get the data
            for (Iterator<Entry<String, String[]>> it=map.entrySet().iterator(); it.hasNext(); ) {
                Map.Entry entry = it.next();
                String key = (String) entry.getKey();
                //out.println("<br>Key : " + key);
                String value[] = (String[]) entry.getValue();
                for (int i=0;i != value.length;i++) {
                    String data[] = value[i].split(",");
                    bar.dataset.setValue(new Double(data[1]),key,data[0]);
                    //out.println("  Value : " + value[i] + "Data : " + data[0]);
                }
            }
            */
            // End SISPEA
        }

        JFreeChart chart = bar.getChart();
        return chart;
    }


    /**
     * Create BoxAndWhisker chart method.
     *
     * @param map which can consist of elements such as:
     *            <ul>
     *            <li> Type - the type of chart: either Bar, Bar3D, Area, Line, or Line3D.</li>
     *            <li> Title - the title of the chart.</li>
     *            <li> XLabel - the domain title.</li>
     *            <li> YLabel - the range title.<li>
     *            <li> Orientation - the orientation of the chart, either H or V.<li>
     *            <li> Range - the upper and lower value of the range. Format is Range=lower,upper, e.g. Range=68,72.</li>
     *            <li> XMLFile - the XML file name if applicable.</li>
     *            <li> data - the data to plot. Format is name=value&name=value, eg: Fish=70&Chips=66.</li>
     *            </ul>
     * @param req which is the Http Servlet Request
     * @return JFreeChart
     * @throws ServletException
     */
    private JFreeChart createBoxAndWhisker(Map<String, String[]> map, HttpServletRequest req)
            throws ServletException {

        // sispea chart bean
        CategoryChartBean lBoxAndWhisker = (CategoryChartBean) req.getSession().getAttribute("BOX_AND_WHISKER_CHART");
        req.getSession().removeAttribute("BOX_AND_WHISKER_CHART");

        // XAxis
        final CategoryAxis xAxis = new CategoryAxis(lBoxAndWhisker.getXLabel());
        xAxis.setCategoryLabelPositions(CategoryLabelPositions.UP_45);

        // YAxis
        final NumberAxis yAxis = new NumberAxis(lBoxAndWhisker.getYLabel());
        yAxis.setAutoRangeIncludesZero(false);

        // renderer
        final BoxAndWhiskerRenderer renderer = new BoxAndWhiskerRenderer();
        renderer.setFillBox(false);
        final XYBoxAndWhiskerRenderer xyRenderer = new XYBoxAndWhiskerRenderer();
        xyRenderer.setFillBox(false);
        xyRenderer.setDefaultEntityRadius(4);

        // build dataset
        final DefaultBoxAndWhiskerCategoryDataset lDataSet = new DefaultBoxAndWhiskerCategoryDataset();
        for (ChartValueDto chartValueBean : lBoxAndWhisker.getValues()) {
            BoxAndWhiskerValueBean lChartValue = (BoxAndWhiskerValueBean) chartValueBean;
            lDataSet.add(lChartValue.getValues(), lChartValue.getName(), lChartValue.getLabel());
        }

        // plotter
        final CategoryPlot plot = new CategoryPlot(lDataSet, xAxis, yAxis, renderer);

        // create chart
        JFreeChart lChart = new JFreeChart(lBoxAndWhisker.getTitle(), plot);

        // result
        return lChart;
    }

    /**
     * Create XY chart method.
     *
     * @param map which can consist of elements such as:
     *            <ul>
     *            <li> Type - the type of chart: either Bar, Step, Area, Line, Scatter, or Time.</li>
     *            <li> Title - the title of the chart.</li>
     *            <li> XLabel - the domain title.</li>
     *            <li> YLabel - the range title.<li>
     *            <li> Orientation - the orientation of the chart, either H or V.<li>
     *            <li> Range - the upper and lower value of the range. Format is Range=lower,upper, e.g. Range=68,72.</li>
     *            <li> data - the data to plot. Format is name=value&name=value, eg: Fish=70&Chips=66.</li>
     *            </ul>
     * @param req which is the Http Servlet Request
     * @return JFreeChart
     * @throws ServletException
     */
    private JFreeChart createXY(Map<String, String[]> map, HttpServletRequest req)
            throws ServletException {

        String lChartName = "";

        // set the title for the Pie Chart
        if (map.containsKey("SessionChartName")) {
            lChartName = req.getParameter("SessionChartName");
            map.remove("SessionChartName");
        }

        // Use the scatter chart factory
        ScatterChartFactory scatterChartFactory = new ScatterChartFactory();
        scatterChartFactory.setServiceMessages(serviceMessages);

        // Get the data from the session
        ScatterChartDto scatterChart = (ScatterChartDto) req.getSession().getAttribute(lChartName);
        req.getSession().removeAttribute(lChartName);

        return scatterChartFactory.createXY(map, scatterChart);
    }

}