package fr.inra.agrosyst.web.actions.managementmodes;

/*
 * #%L
 * Agrosyst :: Web
 * $Id: DecisionRulesEdit.java 4612 2014-12-06 18:43:48Z echatellier $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5/agrosyst-web/src/main/java/fr/inra/agrosyst/web/actions/managementmodes/DecisionRulesEdit.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Strings;
import com.opensymphony.xwork2.Preparable;
import fr.inra.agrosyst.api.entities.AgrosystInterventionType;
import fr.inra.agrosyst.api.entities.BioAgressorType;
import fr.inra.agrosyst.api.entities.CroppingPlanEntry;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.managementmode.DecisionRule;
import fr.inra.agrosyst.api.entities.managementmode.DecisionRuleImpl;
import fr.inra.agrosyst.api.entities.referential.RefBioAgressor;
import fr.inra.agrosyst.api.services.domain.DomainDto;
import fr.inra.agrosyst.api.services.domain.DomainService;
import fr.inra.agrosyst.api.services.managementmode.ManagementModeService;
import fr.inra.agrosyst.web.actions.AbstractAgrosystAction;
import org.apache.commons.lang3.StringUtils;
import org.apache.struts2.convention.annotation.Action;
import org.apache.struts2.convention.annotation.Result;

import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * Decision rule edit action.
 *
 * @author Eric Chatellier
 */
public class DecisionRulesEdit extends AbstractAgrosystAction implements Preparable {

    /**
     * serialVersionUID.
     */
    private static final long serialVersionUID = -6937262808324248162L;

    protected transient ManagementModeService managementModeService;
    protected transient DomainService domainService;

    /**
     * Decision rule id to edit.
     */
    protected String decisionRuleTopiaId;

    /**
     * Decision rule entity to edit.
     */
    protected DecisionRule decisionRule;

    /**
     * domain list (only for create mode).
     */
    protected List<Domain> domains;

    /**
     * Domain id (create mode).
     */
    protected String domainCode;

    protected String domainName;

    /**
     * Liste des cultures associées au système de culture sélectionné.
     */
    protected List<CroppingPlanEntry> croppingPlanEntries;

    /**
     * Selected cropping plan entry code.
     */
    protected String croppingPlanEntryCode;

    /**
     * Type de bio agresseurs.
     */
    protected List<BioAgressorType> bioAgressorTypes;

    /**
     * Sous ensemble de bioagresseur pour le {@code bioAgressorType} sélectionné.
     */
    protected List<? extends RefBioAgressor> bioAgressors;

    /**
     * Selected bio agressor topiaId.
     */
    protected String bioAgressorTopiaId;

    /**
     * Code of the decision rule
     */
    protected String decisionRuleCode;

    /**
     * Related Decision Rules
     */
    protected List<DecisionRule> relatedDecisionRules;

    public void setDomainService(DomainService domainService) {
        this.domainService = domainService;
    }

    public void setManagementModeService(ManagementModeService managementModeService) {
        this.managementModeService = managementModeService;
    }

    public void setDomainCode(String domainCode) {
        this.domainCode = domainCode;
    }

    public DecisionRule getDecisionRule() {
        if (decisionRule == null) {
            // EChatellier 27/06/2013 Fais chier de devoir écrire ça, mais c'est la seule option pour ne pas avoir une grosse dose d'exceptions avec du paramsPrepareParams
            return new DecisionRuleImpl();
        }
        return decisionRule;
    }

    @Override
    public void prepare() throws Exception {
        if (StringUtils.isBlank(decisionRuleTopiaId)) {
            if (StringUtils.isBlank(decisionRuleCode)) {
                // Cas de création d'une règle de décision
                decisionRule = managementModeService.newDecisionRule();
            } else {
                // Cas ou l'on vient de la liste des règles de décision
                decisionRule = managementModeService.getLastDecisionRuleVersion(decisionRuleCode);
                decisionRuleTopiaId = decisionRule.getTopiaId();
            }
        } else {
            // Cas d'une mise à jour d'une règle de décision, on récupère une copie afin de ne pas modifier l'original
            decisionRule = managementModeService.getDecisionRule(decisionRuleTopiaId);
        }
    }

    /**
     * Initialisation de certaines variables pour le premier appel de la page.
     */
    @Override
    @Action("decision-rules-edit-input")
    public String input() throws Exception {

        if (!Strings.isNullOrEmpty(decisionRuleTopiaId)) {
            authorizationService.checkDecisionRuleReadable(decisionRuleTopiaId);
            readOnly = !authorizationService.isDecisionRuleWritable(decisionRuleTopiaId);
        }
        if (readOnly) {
            notificationSupport.decisionRuleNotWritable();
        }

        croppingPlanEntryCode = decisionRule.getCroppingPlanEntryCode();

        if (decisionRule.getBioAgressor() != null) {
            bioAgressorTopiaId = decisionRule.getBioAgressor().getTopiaId();
        }

        initForInput();

        return INPUT;
    }

    /**
     * Initialisation des listes ou autres données à chaque affichage (premier/erreurs).
     */
    @Override
    protected void initForInput() {

        // select combo box
        if (getDecisionRule().isPersisted()) {
            // TODO DCossé 31/01/14 ne pourrais-t-on pas remonter uniquement le nom du domaine ?
            DomainDto domain = domainService.getDomainByCode(getDecisionRule().getDomainCode());
            domainName = domain.getName();

            croppingPlanEntries = managementModeService.getDomainCodeCroppingPlanEntries(getDecisionRule().getDomainCode());
            relatedDecisionRules = managementModeService.getRelatedDecisionRules(getDecisionRule().getCode());
        } else {
            domains = domainService.getActiveWritableDomainsForDecisionRuleCreation();

            croppingPlanEntries = Collections.emptyList();
        }

        bioAgressorTypes = managementModeService.getBioAgressorTypes();
        BioAgressorType bioAgressorType = decisionRule.getBioAgressorType();
        if (bioAgressorType != null) {
            bioAgressors = managementModeService.getBioAgressors(bioAgressorType);
        }
    }

    @Override
    public void validate() {

        if (StringUtils.isBlank(decisionRule.getName())) {
            addFieldError("decisionRule.name", "Le nom est obligatoire !");
        }

        if (decisionRule.getInterventionType() == null) {
            addFieldError("decisionRule.interventionType", "Le type d'intervention est obligatoire !");
        }

        if (hasErrors()) {
            initForInput();
        }
    }

    @Override
    @Action(results = {@Result(type = "redirectAction", params = {"actionName", "decision-rules-edit-input", "decisionRuleTopiaId", "${decisionRule.topiaId}"})})
    public String execute() throws Exception {

        decisionRule = managementModeService.createOrUpdateDecisionRule(decisionRule, domainCode, croppingPlanEntryCode, bioAgressorTopiaId);
        decisionRuleTopiaId = decisionRule.getTopiaId();
        notificationSupport.decisionRuleSaved(decisionRule);

        return super.execute();
    }

    public Map<AgrosystInterventionType, String> getAgrosystInterventionTypes() {
        return getEnumAsMap(AgrosystInterventionType.values());
    }

    public String getDomainCode() {
        return domainCode;
    }

    public List<Domain> getDomains() {
        return domains;
    }

    public void setDecisionRuleTopiaId(String decisionRuleTopiaId) {
        this.decisionRuleTopiaId = decisionRuleTopiaId;
    }

    public String getDecisionRuleTopiaId() {
        return decisionRuleTopiaId;
    }

    public BioAgressorType[] getBioAgressorTypes() {
        return BioAgressorType.values();
    }

    public List<? extends RefBioAgressor> getBioAgressors() {
        return bioAgressors;
    }

    public List<CroppingPlanEntry> getCroppingPlanEntries() {
        return croppingPlanEntries;
    }

    public String getBioAgressorTopiaId() {
        return bioAgressorTopiaId;
    }

    public void setBioAgressorTopiaId(String bioAgressorTopiaId) {
        this.bioAgressorTopiaId = bioAgressorTopiaId;
    }

    public List<DecisionRule> getRelatedDecisionRules() {
        return relatedDecisionRules;
    }

    public void setDecisionRuleCode(String decisionRuleCode) {
        this.decisionRuleCode = decisionRuleCode;
    }

    public void setCroppingPlanEntryCode(String croppingPlanEntryCode) {
        this.croppingPlanEntryCode = croppingPlanEntryCode;
    }

    public String getCroppingPlanEntryCode() {
        return croppingPlanEntryCode;
    }

    public String getDomainName() {
        return domainName;
    }
}
