package fr.inra.agrosyst.web.actions.managementmodes;

/*
 * #%L
 * Agrosyst :: Web
 * $Id: ManagementModesEdit.java 5195 2015-12-10 13:37:06Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-web/src/main/java/fr/inra/agrosyst/web/actions/managementmodes/ManagementModesEdit.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Strings;
import com.google.common.collect.Collections2;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.gson.reflect.TypeToken;
import com.opensymphony.xwork2.Preparable;
import fr.inra.agrosyst.api.NavigationContext;
import fr.inra.agrosyst.api.entities.AgrosystInterventionType;
import fr.inra.agrosyst.api.entities.BioAgressorType;
import fr.inra.agrosyst.api.entities.CroppingPlanEntry;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.managementmode.CategoryObjective;
import fr.inra.agrosyst.api.entities.managementmode.CategoryStrategy;
import fr.inra.agrosyst.api.entities.managementmode.DecisionRule;
import fr.inra.agrosyst.api.entities.managementmode.ManagementMode;
import fr.inra.agrosyst.api.entities.managementmode.ManagementModeCategory;
import fr.inra.agrosyst.api.entities.managementmode.ManagementModeImpl;
import fr.inra.agrosyst.api.entities.managementmode.Section;
import fr.inra.agrosyst.api.entities.managementmode.SectionType;
import fr.inra.agrosyst.api.entities.managementmode.Strategy;
import fr.inra.agrosyst.api.entities.managementmode.StrategyType;
import fr.inra.agrosyst.api.services.common.HistoryItem;
import fr.inra.agrosyst.api.services.common.HistoryType;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemFilter;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemService;
import fr.inra.agrosyst.api.services.managementmode.ManagementModeService;
import fr.inra.agrosyst.api.services.managementmode.ManagementModes;
import fr.inra.agrosyst.api.services.managementmode.SectionDto;
import fr.inra.agrosyst.api.services.managementmode.StrategyDto;
import fr.inra.agrosyst.api.services.referential.ReferentialService;
import fr.inra.agrosyst.web.actions.AbstractAgrosystAction;
import org.apache.commons.lang3.StringUtils;
import org.apache.struts2.convention.annotation.Action;
import org.apache.struts2.convention.annotation.Result;

import java.lang.reflect.Type;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * Management mode edit action.
 * 
 * @author Eric Chatellier
 */
public class ManagementModesEdit extends AbstractAgrosystAction implements Preparable {

    /** serialVersionUID. */
    private static final long serialVersionUID = -2688592598121764478L;

    protected static final String DATE_FORMAT = "dd/MM/yyyy";

    protected static final int HISTORY_SIZE_LIMIT = 100;

    protected transient ManagementModeService managementModeService;

    protected transient GrowingSystemService growingSystemService;

    protected transient ReferentialService referentialService;

    /** Management mode id to edit. */
    protected String managementModeTopiaId;

    /** Management mode entity to edit. */
    protected ManagementMode managementMode;

    /** Growing system list (only for edit mode). */
    protected List<GrowingSystem> growingSystems;

    /** Domain id (of selected growing system). */
    protected String domainTopiaId;

    /** Growing system id (edit mode). */
    protected String growingSystemTopiaId;

    /** Current management mode sections. */
    protected List<SectionDto> sections;
    protected String sectionsJson;

    /** Type de bio agresseurs. */
    protected List<BioAgressorType> bioAgressorTypes;

    /** Liste des cultures associées au système de culture sélectionné. */
    protected List<CroppingPlanEntry> croppingPlanEntries;

    /** Liste des regles associés au système de culture sélectionné. */
    protected Collection<DecisionRule> decisionRules;

    protected List<ManagementMode> relatedManagementModes;

    protected List<ManagementModeCategory> availableManagementModeCategories;

    protected List<String> histories;

    public void setManagementModeService(ManagementModeService managementModeService) {
        this.managementModeService = managementModeService;
    }

    public void setGrowingSystemService(GrowingSystemService growingSystemService) {
        this.growingSystemService = growingSystemService;
    }

    public void setReferentialService(ReferentialService referentialService) {
        this.referentialService = referentialService;
    }

    public ManagementMode getManagementMode() {
        if (managementMode == null) {
            // EChatellier 27/06/2013 Fais chier de devoir écrire ça, mais c'est la seule option pour ne pas avoir une grosse dose d'exceptions avec du paramsPrepareParams
            return new ManagementModeImpl();
        }
        return managementMode;
    }
    
    @Override
    public void prepare() throws Exception {
        if (StringUtils.isEmpty(managementModeTopiaId)) {
            // Cas de création d'un growingSystem
            managementMode = managementModeService.newManagementMode();
        } else {
            // Cas d'une mise à jour de growingSystem
            managementMode = managementModeService.getManagementMode(managementModeTopiaId);
        }
    }
    
    /**
     * Initialisation de certaines variables pour le premier appel de la page.
     */
    @Override
    @Action("management-modes-edit-input")
    public String input() throws Exception {

        if (!Strings.isNullOrEmpty(managementModeTopiaId)) {
            authorizationService.checkManagementModeReadable(managementModeTopiaId);
            readOnly = !authorizationService.isManagementModeWritable(managementModeTopiaId);
        }
        if (readOnly) {
            notificationSupport.managementModeNotWritable();
        }

        if (getManagementMode().getGrowingSystem() != null) {
            growingSystemTopiaId = managementMode.getGrowingSystem().getTopiaId();
        }

        sections = getManagementModeSections(managementMode);

        initForInput();

        return INPUT;
    }

    protected List<SectionDto> getManagementModeSections(ManagementMode managementMode) {
        List<SectionDto> sections;
        if (managementMode.getSections() != null) {
            Collection<Section> mmSections = managementMode.getSections();
            sections = Lists.newArrayList(Collections2.transform(mmSections, ManagementModes.SECTION_TO_DTO));
        } else {
            sections = Collections.emptyList();
        }
        return sections;
    }

    /**
     * Initialisation des listes ou autres données à chaque affichage (premier/erreurs).
     */
    @Override
    protected void initForInput() {
        
        // select combo box
        if (!getManagementMode().isPersisted()) {
            GrowingSystemFilter growingSystemFilter = new GrowingSystemFilter();
            NavigationContext navigationContext = getNavigationContext();
            growingSystemFilter.setNavigationContext(navigationContext);

            growingSystems = managementModeService.getGrowingSystemsForManagementMode(navigationContext);

            histories = Lists.newArrayList();
        } else {
            translateHistorical();
        }

        if (StringUtils.isNotBlank(growingSystemTopiaId)) {
            GrowingSystem growingSystem = growingSystemService.getGrowingSystem(growingSystemTopiaId);
            domainTopiaId = growingSystem.getGrowingPlan().getDomain().getTopiaId();

            croppingPlanEntries = managementModeService.getGrowingSystemCroppingPlanEntries(growingSystemTopiaId);
            decisionRules = managementModeService.getGrowingSystemDecisionRules(growingSystemTopiaId);
            availableManagementModeCategories = managementModeService.getAvailableManagementModeCategories(growingSystemTopiaId);
            relatedManagementModes = managementModeService.getRelatedManagementModes(managementMode.getGrowingSystem());
        } else {
            availableManagementModeCategories = Lists.newArrayList(ManagementModeCategory.values());
        }

        bioAgressorTypes = managementModeService.getBioAgressorTypes();
    }

    protected void translateHistorical() {
        String historical = managementMode.getHistorical();

        if(!Strings.isNullOrEmpty(historical)) {
            Type type = new TypeToken<List<HistoryItem>>() {}.getType();
            List<HistoryItem> items = getGson().fromJson(historical, type);
            int limit = Math.min(items.size(), getHistorySizeLimit());

            histories = Lists.newArrayListWithCapacity(limit);
            SimpleDateFormat simpleFormat = new SimpleDateFormat(DATE_FORMAT);

            for (HistoryItem item: Iterables.limit(Lists.reverse(items), limit)) {
                Date date = item.getDate();
                String stDate = simpleFormat.format(date);
                List<String> args = item.getArgs();
                HistoryType historyType = item.getType();

                String format = getText(HistoryType.class.getName() + "." + historyType.name());
                switch (historyType) {
                    case MANAGEMENT_MODE_ADD:
                        // [toManagementModeCategory]
                        String toCategory = args.get(0);
                        String translatedToCategories = getText(ManagementModeCategory.class.getName() + "." + toCategory);
                        histories.add(String.format(format, stDate, translatedToCategories));
                        break;
                    case MANAGEMENT_MODE_COPY:
                        // [fromManagementModeCategory, toManagementModeCategory]
                        String fromCategory = args.get(0);
                        toCategory = args.get(1);
                        String translatedFromCategories = getText(ManagementModeCategory.class.getName() + "." + fromCategory);
                        translatedToCategories = getText(ManagementModeCategory.class.getName() + "." + toCategory);
                        histories.add(String.format(format, stDate, translatedToCategories, translatedFromCategories));
                        break;
                    case MANAGEMENT_MODE_EXTEND:
                        // [fromManagementModeCategory, toManagementModeCategory]
                        fromCategory = args.get(0);
                        toCategory = args.get(1);
                        translatedFromCategories = getText(ManagementModeCategory.class.getName() + "." + fromCategory);
                        translatedToCategories = getText(ManagementModeCategory.class.getName() + "." + toCategory);
                        histories.add(String.format(format, stDate, translatedToCategories, translatedFromCategories));
                        break;
                    case SECTION_ADD:
                    case SECTION_REMOVE:
                        // [SectionType, BioAgressorType, RefBioAgressor]
                        String stSectionType = args.get(0);
                        String translatedSectionType = getText(SectionType.class.getName() + "." + stSectionType);

                        String stBioAgressorType = args.get(1);
                        String translatedBioAgressorType = Strings.isNullOrEmpty(stBioAgressorType) ? "-" : getText(BioAgressorType.class.getName() + "." + stBioAgressorType);

                        //managementModeService.getBioAgressors()
                        String bioAgressorId = args.get(2);
                        String bioAgressorName = Strings.isNullOrEmpty(bioAgressorId) ? "-" : referentialService.getBioAgressor(bioAgressorId).getLabel();

                        histories.add(String.format(format, stDate, translatedSectionType, translatedBioAgressorType, bioAgressorName));
                        break;
                    case STRATEGY_ADD:
                    case STRATEGY_REMOVE:
                        // [multiannual, Explanation, CroppingPlanEntryTopiaId]
                        Boolean multiannual = Boolean.valueOf(args.get(0));
                        String strategyType = getText(Strategy.class.getName() + "." + Strategy.PROPERTY_MULTIANNUAL + "." + multiannual);

                        // index 1: explanation

                        String croppingPlanEntryTopiaId = args.get(2);
                        String croppingPlanEntryName = Strings.isNullOrEmpty(croppingPlanEntryTopiaId) ? "-" : managementModeService.getCroppingPlanEntries(croppingPlanEntryTopiaId).getName();

                        histories.add(String.format(format, stDate, strategyType, croppingPlanEntryName));
                        break;
                    case RULE_ADD:
                    case RULE_REMOVE:
                        // [TopiaId, VersionNumber]

                        String ruleId = args.get(0);
                        String ruleName = Strings.isNullOrEmpty(ruleId) ? "-" : managementModeService.getDecisionRule(ruleId).getName();

                        Integer ruleVersion = Integer.valueOf(args.get(1));

                        histories.add(String.format(format, stDate, ruleName, ruleVersion));

                        break;
                }
            }
        }
    }
    
    @Override
    public void validate() {

        // growing system mandatory only during edit mode
        if (!getManagementMode().isPersisted() && StringUtils.isBlank(growingSystemTopiaId)) {
            addFieldError("growingSystemTopiaId", "Le système de culture est obligatoire !");
        }

        if (managementMode.getCategory()==null) {
            addFieldError("managementMode.category", "La catégorie est obligatoire !");
        }

        try {
            convertSectionsJson(sectionsJson);
        } catch (Exception ex) {
            addActionError("Exception:" + ex.toString() + "\n/!\\ Désolé les données saisies depuis votre dernier enregistrement concernant les rubriques n'ont pu être récupérées !");
        }

        if (sections == null) {
            sections = getManagementModeSections(managementMode);
        }

        for (SectionDto section : sections) {
            if (section.getSectionType() == null) {
                addFieldError("editedSection.sectionType", "Le type de rubrique est obligatoire !");
                addActionError("Le type de rubrique est obligatoire !");
            } else {
                List<StrategyDto> strategies = section.getStrategiesDto();
                if (strategies != null){
                    for (Strategy strategie : strategies) {
                        if (StringUtils.isBlank(strategie.getExplanation())) {
                            addFieldError("editedStrategy.explanation", "Une stratégie n'a pas d'explication de renseignée !");
                            addActionError("Une stratégie n'a pas d'explication de renseignée !");
                        }
                        if (section.getSectionType() != null && (section.getSectionType() == SectionType.ADVENTICES || section.getSectionType() == SectionType.MALADIES || section.getSectionType() == SectionType.RAVAGEURS)){
                            if (strategie.getStrategyType() == null) {
                                addFieldError("strategyTypeField", "Les stratégies de type 'Maitrise des maladies', 'Maitrise des adventices' et 'Maitrise des ravageurs' doivent avoir un type de gestion de défini !");
                                addActionError("Cette stratégie doit avoir un type de gestion de défini !");
                            }
                        }
                    }
                }
            }
        }

        if (hasErrors()) {
            initForInput();
        }
    }
    
    @Override
    @Action(results = {@Result(type = "redirectAction", params = {"actionName", "management-modes-edit-input", "managementModeTopiaId", "${managementMode.topiaId}"})})
    public String execute() throws Exception {

        managementMode = managementModeService.createOrUpdateManagementMode(managementMode, growingSystemTopiaId, sections);
        if (Strings.isNullOrEmpty(managementModeTopiaId)){
            notificationSupport.newManagementModeCreated(managementMode);
        } else {
            notificationSupport.managementModeSaved(managementMode);
        }
        return super.execute();
    }

    public void convertSectionsJson(String json) {
        Type type = new TypeToken<List<SectionDto>>() {
        }.getType();
        this.sections = getGson().fromJson(json, type);
    }

    public String getManagementModeTopiaId() {
        return managementModeTopiaId;
    }

    public void setManagementModeTopiaId(String managementModeTopiaId) {
        this.managementModeTopiaId = managementModeTopiaId;
    }

    public String getGrowingSystemTopiaId() {
        return growingSystemTopiaId;
    }

    public void setGrowingSystemTopiaId(String growingSystemTopiaId) {
        this.growingSystemTopiaId = growingSystemTopiaId;
    }

    public String getDomainTopiaId() {
        return domainTopiaId;
    }

    public List<GrowingSystem> getGrowingSystems() {
        return growingSystems;
    }
    
    public Map<ManagementModeCategory, String> getManagementModeCategories() {
        return getEnumAsMap(ManagementModeCategory.values());
    }

    public Map<SectionType, String> getSectionTypes() {
        return getEnumAsMap(SectionType.values());
    }
    
    public Map<CategoryObjective, String> getCategoryObjectives() {
        return getEnumAsMap(CategoryObjective.values());
    }
    
    public Map<CategoryStrategy, String> getCategoryStrategies() {
        return getEnumAsMap(CategoryStrategy.values());
    }
    
    public Map<StrategyType, String> getStrategyTypes() {
        return getEnumAsMap(StrategyType.values());
    }

    public Map<BioAgressorType, String> getBioAgressorTypes() {
        return getEnumAsMap(BioAgressorType.values());
    }

    public List<SectionDto> getSections() {
        return sections;
    }

    public void setSections(String json) {
        sectionsJson = json;
    }

    public List<CroppingPlanEntry> getCroppingPlanEntries() {
        return croppingPlanEntries;
    }

    public Collection<DecisionRule> getDecisionRules() {
        return decisionRules;
    }

    public List<ManagementModeCategory> getAvailableManagementModeCategories() {
        return availableManagementModeCategories;
    }

    public List<ManagementMode> getRelatedManagementModes() {
        return relatedManagementModes;
    }

    public ManagementModeCategory[] getCategories(){
        return ManagementModeCategory.values();
    }

    public Map<AgrosystInterventionType, String> getAgrosystInterventionTypes() {
        return getEnumAsMap(AgrosystInterventionType.values());
    }

    public List<String> getHistories() {
        return histories;
    }

    public int getHistorySizeLimit() {
        return HISTORY_SIZE_LIMIT;
    }
}
