package fr.inra.agrosyst.web.actions.itk;

/*
 * #%L
 * Agrosyst :: Web
 * $Id: AbstractItkAction.java 5098 2015-10-09 16:15:41Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-web/src/main/java/fr/inra/agrosyst/web/actions/itk/AbstractItkAction.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import fr.inra.agrosyst.api.entities.AgrosystInterventionType;
import fr.inra.agrosyst.api.entities.BioAgressorType;
import fr.inra.agrosyst.api.entities.action.AbstractAction;
import fr.inra.agrosyst.api.entities.action.AbstractInput;
import fr.inra.agrosyst.api.entities.action.BiologicalControlActionImpl;
import fr.inra.agrosyst.api.entities.action.BiologicalProductInput;
import fr.inra.agrosyst.api.entities.action.CapacityUnit;
import fr.inra.agrosyst.api.entities.action.FertiOrgaUnit;
import fr.inra.agrosyst.api.entities.action.HarvestingAction;
import fr.inra.agrosyst.api.entities.action.HarvestingYeald;
import fr.inra.agrosyst.api.entities.action.IrrigationActionImpl;
import fr.inra.agrosyst.api.entities.action.MineralProductInput;
import fr.inra.agrosyst.api.entities.action.MineralProductUnit;
import fr.inra.agrosyst.api.entities.action.OrganicProductInput;
import fr.inra.agrosyst.api.entities.action.OrganicProductUnit;
import fr.inra.agrosyst.api.entities.action.OtherProductInput;
import fr.inra.agrosyst.api.entities.action.PesticideProductInput;
import fr.inra.agrosyst.api.entities.action.PesticidesSpreadingActionImpl;
import fr.inra.agrosyst.api.entities.action.PhytoProductUnit;
import fr.inra.agrosyst.api.entities.action.SeedPlantUnit;
import fr.inra.agrosyst.api.entities.action.SeedType;
import fr.inra.agrosyst.api.entities.action.SeedingAction;
import fr.inra.agrosyst.api.entities.action.SeedingActionSpecies;
import fr.inra.agrosyst.api.entities.action.SeedingProductInput;
import fr.inra.agrosyst.api.entities.action.YealdCategory;
import fr.inra.agrosyst.api.entities.action.YealdUnit;
import fr.inra.agrosyst.api.entities.referential.RefFertiMinUNIFA;
import fr.inra.agrosyst.api.entities.referential.RefFertiOrga;
import fr.inra.agrosyst.api.entities.referential.RefInterventionAgrosystTravailEDI;
import fr.inra.agrosyst.api.services.input.Inputs;
import fr.inra.agrosyst.api.services.itk.Itk;
import fr.inra.agrosyst.api.services.referential.MineralProductType;
import fr.inra.agrosyst.api.services.referential.ReferentialService;
import fr.inra.agrosyst.web.actions.AbstractAgrosystAction;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @author Arnaud Thimel : thimel@codelutin.com
 */
public abstract class AbstractItkAction extends AbstractAgrosystAction {

    private static final long serialVersionUID = -6057182538439163217L;

    protected transient ReferentialService referentialService;


    protected List<MineralProductType> mineralProductTypes;
    protected List<RefFertiOrga> organicProductTypes;
    protected Map<AgrosystInterventionType, List<String>> actaTreatmentProductTypes;
    protected List<RefInterventionAgrosystTravailEDI> agrosystActionsFullList;

    public final void setReferentialService(ReferentialService referentialService) {
        this.referentialService = referentialService;
    }

    @Override
    protected void initForInput() {

        agrosystActionsFullList = referentialService.findAllActiveAgrosystActions();
        mineralProductTypes = referentialService.findAllActiveMineralProductTypes();
        organicProductTypes = referentialService.findAllActiveOrganicProductTypes();
        actaTreatmentProductTypes = referentialService.getAllActiveActaTreatmentProductTypes();

        super.initForInput();
    }

    protected Map<AgrosystInterventionType, List<AbstractAction>> validActions(Collection<AbstractAction> actions) {
        Map<AgrosystInterventionType, List<AbstractAction>> result = Maps.newHashMap();
        if (CollectionUtils.isEmpty(actions)) {
            addActionError("Une intervention doit avoir au moins une action");
        } else {
            List<AgrosystInterventionType> usedActionTypes = new ArrayList<AgrosystInterventionType>();
            for (AbstractAction action : actions) {
                if (action.getMainAction() == null) {
                    addActionError("Une action n'a pas de type d'intervention");
                } else {
                    AgrosystInterventionType interventionType = action.getMainAction().getIntervention_agrosyst();

                    addActionToActionsByType(result, action, interventionType);

                    if (interventionType != AgrosystInterventionType.ENTRETIEN_TAILLE_VIGNE_ET_VERGER && usedActionTypes.contains(interventionType)) {
                        addActionError("Il n'est possible de définir qu'une seule action de type "+getText(interventionType.getClass().getName()+"."+interventionType.toString()));
                    } else {
                        usedActionTypes.add(interventionType);
                    }

                    switch (action.getMainAction().getIntervention_agrosyst()) {
                        case RECOLTE:
                            HarvestingAction harvestingAction = (HarvestingAction)action;
                            if (CollectionUtils.isEmpty(harvestingAction.getHarvestingYealds())) {
                                addActionError("Un rendement est obligatoire pour l'action de récolte !");
                            } else {
                                for (HarvestingYeald harvestingYeald : harvestingAction.getHarvestingYealds()) {
                                    if (harvestingYeald.getYealdCategory() == null) {
                                        addActionError("La categorie de rendement est obligatoire !");
                                    }
                                    if (harvestingYeald.getYealdUnit() == null) {
                                        addActionError("L'unité de rendement est obligatoire !");
                                    }
                                }
                            }
                            break;
                        case APPLICATION_DE_PRODUITS_PHYTOSANITAIRES:
                            PesticidesSpreadingActionImpl pesticidesSpreadingAction = (PesticidesSpreadingActionImpl)action;
                            Double boiledQuantity = pesticidesSpreadingAction.getBoiledQuantity();
                            if (boiledQuantity == null) {
                                addActionError("Le volume moyen de bouillie par hectare est obligatoire pour l'action de traitement phytosanitaire !");
                            }
                            if (pesticidesSpreadingAction.getProportionOfTreatedSurface() <= 0){
                                addActionError("La proportion de surface traitée au sein de la surface de l'intervention est obligatoire pour l'action de traitement phytosanitaire !");
                            }
                            break;
                        case IRRIGATION:
                            IrrigationActionImpl irrigationAction = (IrrigationActionImpl)action;
                            double waterQuantityAverage = irrigationAction.getWaterQuantityAverage();
                            if (waterQuantityAverage <= 0) {
                                addActionError("La quantité d'eau apportée est obligatoire pour l'action d'irrigation !");
                            }
                            break;
                        case LUTTE_BIOLOGIQUE:
                            BiologicalControlActionImpl biologicalControlAction = (BiologicalControlActionImpl)action;
                            if (biologicalControlAction.getProportionOfTreatedSurface() <= 0) {
                                addActionError("La proportion de surface traitée au sein de la surface de l'intervention est obligatoire pour l'action lutte biologique !");
                            }
                            break;
                        case SEMIS:
                            SeedingAction seedingAction = (SeedingAction)action;
                            Collection<SeedingActionSpecies> seedingActionSpecieses = seedingAction.getSeedingSpecies();
                            if (seedingActionSpecieses != null) {
                                for (SeedingActionSpecies seedingActionSpecies : seedingActionSpecieses) {
                                    if (seedingActionSpecies.isBiologicalSeedInoculation() || seedingActionSpecies.isTreatment()) {
                                        if (seedingActionSpecies.getQuantity() == 0 || seedingActionSpecies.getSeedPlantUnit() == null) {
                                            addActionError("Lorsqu'un traitement de semence (chimique ou biologique) est déclaré, la quantité semée ne doit pas être égale à zero et une unité doit être choisie !");
                                        }
                                    }
                                }
                            }
                        default:
                            break;
                    }
                }
            }
        }
        return result;
    }

    private static void addActionToActionsByType(Map<AgrosystInterventionType, List<AbstractAction>> result, AbstractAction action, AgrosystInterventionType interventionType) {
        List<AbstractAction> sameTypeActions = result.get(interventionType);
        if (sameTypeActions == null) {
            sameTypeActions = Lists.newArrayList();
            result.put(interventionType, sameTypeActions);
        }
        sameTypeActions.add(action);
    }

    protected void validInputs(Map<AgrosystInterventionType, List<AbstractAction>> actionsByAgrosystInterventionType, Collection<AbstractInput> inputs) {
        if (!CollectionUtils.isEmpty(inputs)) {
            for (AbstractInput input : inputs) {
                if (input instanceof MineralProductInput) {

                    List<AbstractAction> actions = actionsByAgrosystInterventionType.get(AgrosystInterventionType.APPLICATION_DE_PRODUITS_FERTILISANTS_MINERAUX);
                    if (CollectionUtils.isEmpty(actions)) {
                        addActionError("Pas d'action 'Application de produits minéraux' de défini pour l'intrant de type 'Engrais/amendement (organo)minéral' !");
                    }

                    RefFertiMinUNIFA product = ((MineralProductInput) input).getMineralProduct();
                    if (product == null) {
                        addActionError("Les intrant de type 'Engrais/amendement (organo)minéral' doivent avoir une composition de l'engrais de saisie !");
                    } else {
                        double totalWeight = 0;
                        totalWeight += product.getN() != null ? product.getN() : 0;
                        totalWeight += product.getP2O5() != null ? product.getP2O5() : 0;
                        totalWeight += product.getBore() != null ? product.getBore() : 0;
                        totalWeight += product.getCalcium() != null ? product.getCalcium() : 0;
                        totalWeight += product.getCuivre() != null ? product.getCuivre() : 0;
                        totalWeight += product.getFer() != null ? product.getFer() : 0;
                        totalWeight += product.getK2O() != null ? product.getK2O() : 0;
                        totalWeight += product.getManganese() != null ? product.getManganese() : 0;
                        totalWeight += product.getMgO() != null ? product.getMgO() : 0;
                        totalWeight += product.getMolybdene() != null ? product.getMolybdene() : 0;
                        totalWeight += product.getOxyde_de_sodium() != null ? product.getOxyde_de_sodium() : 0;
                        totalWeight += product.getsO3() != null ? product.getsO3() : 0;
                        totalWeight += product.getZinc() != null ? product.getZinc() : 0;
                        if (totalWeight > 100) {
                            addActionError(" La somme des % poids ne peut être supérieure à 100 pour un intrant de type 'Engrais/amendement (organo)minéral'");
                        }
                    }

                } else if (input instanceof OtherProductInput) {
                    int nbValidAction = actionsByAgrosystInterventionType.size();
                    nbValidAction = nbValidAction - (actionsByAgrosystInterventionType.get(AgrosystInterventionType.TRANSPORT) == null ? 0 : 1);
                    nbValidAction = nbValidAction - (actionsByAgrosystInterventionType.get(AgrosystInterventionType.TRAVAIL_DU_SOL) == null ? 0 : 1);

                    if (nbValidAction == 0) {
                        addActionError("Aucune action pour l'intrant de type 'Autre' !");
                    }

                    OtherProductInput otherProductInput = (OtherProductInput) input;

                    if (otherProductInput.getOtherAction() == null &&
                            otherProductInput.getHarvestingAction() == null &&
                            otherProductInput.getMaintenancePruningVinesAction() == null &&
                            otherProductInput.getIrrigationAction() == null &&
                            otherProductInput.getMineralFertilizersSpreadingAction() == null &&
                            otherProductInput.getOrganicFertilizersSpreadingAction() == null &&
                            otherProductInput.getPesticidesSpreadingAction() == null &&
                            otherProductInput.getBiologicalControlAction() == null &&
                            otherProductInput.getSeedingAction() == null) {
                        addActionError("Aucune action pour l'intrant de type 'Autre' !");
                    }

                    if (StringUtils.isBlank(otherProductInput.getProductName())) {
                        addActionError("Le nom du produit est requis pour un intrant de type 'Autre' !");
                    }

                } else if (input instanceof OrganicProductInput) {
                    List<AbstractAction> actions = actionsByAgrosystInterventionType.get(AgrosystInterventionType.EPANDAGES_ORGANIQUES);
                    if (CollectionUtils.isEmpty(actions)) {
                        addActionError("Pas d'action 'Épandage organique' de défini pour l'intrant de type 'Engrais/amendement organique' !");
                    }
                } else if (input instanceof SeedingProductInput) {
                    List<AbstractAction> actions = actionsByAgrosystInterventionType.get(AgrosystInterventionType.SEMIS);
                    if (CollectionUtils.isEmpty(actions)) {
                        addActionError("Pas d'action 'Semis' de défini pour l'intrant de type 'Traitement de semence' !");
                    } else {
                        SeedingAction seedingAction = (SeedingAction) actions.get(0);
                        Collection<SeedingActionSpecies> specieses = seedingAction.getSeedingSpecies();
                        if (CollectionUtils.isNotEmpty(specieses)) {
                            SeedingProductInput seedingProductInput = (SeedingProductInput) input;

                            if (StringUtils.isNotBlank(seedingProductInput.getProductType())) {
                                List<String> productTypes = getActaTreatmentProductTypes().get(AgrosystInterventionType.SEMIS);
                                Set<String> allowedProductTypes = Sets.newHashSet();
                                for (SeedingActionSpecies species : specieses) {
                                    if (species.isTreatment()) {
                                        //Fongicides
                                        allowedProductTypes.add(productTypes.get(0));
                                        //Insecticides
                                        allowedProductTypes.add(productTypes.get(2));
                                    }
                                    if (species.isBiologicalSeedInoculation()) {
                                        // "Inoculation biologique de semences ou plants"
                                        allowedProductTypes.add(productTypes.get(1));
                                    }
                                }

                                if (!allowedProductTypes.contains(seedingProductInput.getProductType())) {
                                    addActionError(String.format("Aucune espece de l'action 'Semis' n'a de traitement sélectionné pour ce type d'intrant '%s' !", seedingProductInput.getProductType()));
                                }
                            }
                        } else {
                            addActionError("L'action 'Semis', n'a aucune espèce sur laquel s'applique le traitement!");
                        }
                    }
                } else if (input instanceof BiologicalProductInput) {
                    List<AbstractAction> actions = actionsByAgrosystInterventionType.get(AgrosystInterventionType.LUTTE_BIOLOGIQUE);
                    if (CollectionUtils.isEmpty(actions)) {
                        addActionError("Pas d'action 'Lutte biologique' de défini pour l'intrant de type 'Lutte biologique' !");
                    }
                } else if (input instanceof PesticideProductInput) {
                    List<AbstractAction> actions = actionsByAgrosystInterventionType.get(AgrosystInterventionType.APPLICATION_DE_PRODUITS_PHYTOSANITAIRES);
                    if (CollectionUtils.isEmpty(actions)) {
                        addActionError("Pas d'action 'Application de produits phytosanitaires' de défini pour l'intrant de type 'Phytosanitaires' !");
                    }
                }
            }
        }
    }



    ////////////// ITK specific data //////////////

    public Map<YealdCategory, String> getYealdCategories(){
        return getEnumAsMap(YealdCategory.values());
    }

    public Map<CapacityUnit, String> getCapacityUnits(){
        return getEnumAsMap(CapacityUnit.values());
    }

    public List<MineralProductType> getMineralProductTypes() {
        return mineralProductTypes;
    }

    public Map<BioAgressorType, String> getTreatmentTargetCategories() {
        return getEnumAsMap(Itk.TREATMENT_TARGET_CATEGORIES);
    }

    public Map<SeedType, String> getSeedTypes() {
        return getEnumAsMap(SeedType.values());
    }

    public Map<SeedPlantUnit, String> getSeedPlantUnits() {
        return getEnumAsMap(SeedPlantUnit.values());
    }

    public Map<YealdUnit, String> getYealdUnits() {
        return getEnumAsMap(YealdUnit.values());
    }

    public List<RefFertiOrga> getOrganicProductTypes() {
        return organicProductTypes;
    }

    public Map<MineralProductUnit, String> getMineralProductUnits() {
        return getEnumAsMap(MineralProductUnit.values());
    }

    public Map<OrganicProductUnit, String> getOrganicProductUnits() {
        return getEnumAsMap(OrganicProductUnit.values());
    }

    public Map<PhytoProductUnit, String> getPhytoProductUnits() {
        return getEnumAsMap(PhytoProductUnit.values());
    }

    public Map<AgrosystInterventionType, List<String>> getActaTreatmentProductTypes() {
        if (actaTreatmentProductTypes == null) {
            actaTreatmentProductTypes = referentialService.getAllActiveActaTreatmentProductTypes();
        }
        return actaTreatmentProductTypes;
    }

    public Map<FertiOrgaUnit, String> getFertiOrgaUnits() {
        return getEnumAsMap(FertiOrgaUnit.values());
    }

    public Map<AgrosystInterventionType, String> getInputTypesLabels() {
        Map<AgrosystInterventionType, String> result = Maps.newLinkedHashMap();
        for (AgrosystInterventionType value : Inputs.INPUT_TYPES) {
            String key = "InputType." + value.name();
            String trans = getText(key);
            result.put(value, trans);
        }
        return result;
    }

    public List<RefInterventionAgrosystTravailEDI> getAgrosystActionsFullList() {
        return agrosystActionsFullList;
    }

}
