package fr.inra.agrosyst.web;

/*
 * #%L
 * Agrosyst :: Web
 * $Id: AgrosystWebAuthenticationFilter.java 4440 2014-10-16 17:25:26Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-web/src/main/java/fr/inra/agrosyst/web/AgrosystWebAuthenticationFilter.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Charsets;
import com.google.common.base.Function;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableSet;
import fr.inra.agrosyst.api.services.users.UserDto;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.io.IOException;
import java.net.URLEncoder;
import java.util.Set;

/**
 * @author Arnaud Thimel : thimel@codelutin.com
 */
public class AgrosystWebAuthenticationFilter implements Filter {

    private static final Log log = LogFactory.getLog(AgrosystWebAuthenticationFilter.class);
    public static final String AGROSYST_WEB_LOGIN_ACTION = "/auth/login.action";
    public static final String AGROSYST_WEB_LOGIN_ACTION_INPUT = "/auth/login-input.action";

    public static final String AGROSYST_WEB_CHARTER_ACTION = "/auth/charter.action";
    public static final String AGROSYST_WEB_CHARTER_ACTION_INPUT = "/auth/charter-input.action";

    protected static final Set<String> PUBLIC_URLS = ImmutableSet.of(
            AGROSYST_WEB_LOGIN_ACTION,
            AGROSYST_WEB_LOGIN_ACTION_INPUT,
            AGROSYST_WEB_CHARTER_ACTION,
            AGROSYST_WEB_CHARTER_ACTION_INPUT,
            "/auth/forgotten-password.action",
            "/auth/forgotten-password-input.action",
            "/auth/retrieve-password.action",
            "/auth/retrieve-password-input.action",
            "/auth/logout.action",
            "/auth/legal.action",
            "/commons/help-raw.action"
            //"/font/Dosis-Regular.ttf",
            //"/font/Dosis-Bold.ttf",
            //"/img/logo-Ecophyto-Dephy.png",
            //"/img/Login-Background.png",
            //"/img/Login-Pannel.png",
            //"/img/Sprite-Buttons.png",
            //"/js/endpoints-js.jsp",
            //"/nuiton-js/layout.css",
            //"/nuiton-js/agrosyst-login.css",
            //"/nuiton-js/agrosyst.js",
            //"/help/cession_droits_agriculteurs.pdf"
    );

    protected static final Function<HttpServletRequest, String> GET_FULL_REQUESTED_URI = new Function<HttpServletRequest, String>() {
        @Override
        public String apply(HttpServletRequest input) {
            String contextPath = input.getContextPath();
            String requestURI = input.getRequestURI();
            if (requestURI.startsWith(contextPath + "/js/") || requestURI.startsWith(contextPath + "/help/")
                    || requestURI.startsWith(contextPath + "/img/") || requestURI.startsWith(contextPath + "/font/")
                    || requestURI.startsWith(contextPath + "/webjars/") || requestURI.startsWith(contextPath + "/nuiton-js/")) {
                requestURI = contextPath + "/";
            }
            String queryString = input.getQueryString();
            String result;
            if (queryString == null) {
                result = requestURI;
            } else {
                result = String.format("%s?%s", requestURI, queryString);
            }
            return result;
        }
    };

    protected static final Function<HttpServletRequest, String> GET_REDIRECT_TO_LOGIN_PAGE_URI = new Function<HttpServletRequest, String>() {
        @Override
        public String apply(HttpServletRequest input) {
            String contextPath = input.getContextPath();
            String result;
            if ("/".equals(contextPath)) {
                result = AGROSYST_WEB_LOGIN_ACTION_INPUT;
            } else {
                result = contextPath + AGROSYST_WEB_LOGIN_ACTION_INPUT;
            }
            return result;
        }
    };

    protected static final Function<HttpServletRequest, String> GET_REDIRECT_TO_CHARTER_PAGE_URI = new Function<HttpServletRequest, String>() {
        @Override
        public String apply(HttpServletRequest input) {
            String contextPath = input.getContextPath();
            String result;
            if ("/".equals(contextPath)) {
                result = AGROSYST_WEB_CHARTER_ACTION_INPUT;
            } else {
                result = contextPath + AGROSYST_WEB_CHARTER_ACTION_INPUT;
            }
            return result;
        }
    };

    @Override
    public void init(FilterConfig filterConfig) throws ServletException {

        if (log.isInfoEnabled()) {
            log.info("Initializing " + AgrosystWebAuthenticationFilter.class.getName());
        }
        // Nothing to do
    }

    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException, ServletException {
        if (request instanceof HttpServletRequest && response instanceof HttpServletResponse) {
            HttpServletRequest httpServletRequest = (HttpServletRequest) request;
            HttpServletResponse httpServletResponse = (HttpServletResponse) response;

            String servletPath = httpServletRequest.getServletPath();

            boolean publicUrl = PUBLIC_URLS.contains(servletPath) ||
                    // static url patterns
                    servletPath.startsWith("/js/") ||
                    servletPath.startsWith("/img/") ||
                    servletPath.startsWith("/help/") ||
                    servletPath.startsWith("/font/") ||
                    servletPath.startsWith("/webjars/") ||
                    servletPath.startsWith("/nuiton-js/");

            boolean authenticated = false;

            boolean acceptedCharter = true;

            if (!publicUrl) {
                HttpSession session = httpServletRequest.getSession(false);

                if (session != null) {
                    Object sessionObject = session.getAttribute(AgrosystWebSession.SESSION_PARAMETER);
                    if (sessionObject != null) {
                        AgrosystWebSession agrosystSession = (AgrosystWebSession) sessionObject;
                        authenticated = !Strings.isNullOrEmpty(agrosystSession.getAuthenticationToken());

                        // check if user has validate Charter
                        UserDto user = agrosystSession.getAuthenticatedUser();
                        acceptedCharter = user == null || user.isAcceptedCharter();
                    }
                }
            }

            if (log.isTraceEnabled()) {
                String message = String.format("Is '%s' [publicUrl=%b] [authenticated=%b] [acceptedCharter=%b]",
                        servletPath, publicUrl, authenticated, acceptedCharter);
                log.trace(message);
            }

            if (publicUrl) {
                chain.doFilter(request, response);
            } else if (authenticated && acceptedCharter) {
                chain.doFilter(request, response);
            } else if (!authenticated) {
                redirectToLoginPage(httpServletRequest, httpServletResponse);
            } else {
                redirectToCharterPage(httpServletRequest, httpServletResponse);
            }
        }
    }

    private void redirectToLoginPage(HttpServletRequest httpServletRequest, HttpServletResponse httpServletResponse) throws IOException {

        String redirectPageUri = GET_REDIRECT_TO_LOGIN_PAGE_URI.apply(httpServletRequest);

        String requestedUri = GET_FULL_REQUESTED_URI.apply(httpServletRequest);
        String queryString = "?next=" + URLEncoder.encode(requestedUri, Charsets.UTF_8.name());

        httpServletResponse.sendRedirect(redirectPageUri + queryString);

    }

    private void redirectToCharterPage(HttpServletRequest httpServletRequest, HttpServletResponse httpServletResponse) throws IOException {

        String redirectPageUri = GET_REDIRECT_TO_CHARTER_PAGE_URI.apply(httpServletRequest);

        String requestedUri = GET_FULL_REQUESTED_URI.apply(httpServletRequest);
        String queryString = "?next=" + URLEncoder.encode(requestedUri, Charsets.UTF_8.name());

        httpServletResponse.sendRedirect(redirectPageUri + queryString);
    }

    @Override
    public void destroy() {

        if (log.isInfoEnabled()) {
            log.info("Destroying " + AgrosystWebAuthenticationFilter.class.getName());
        }
        // Nothing to do
    }

}
