package fr.inra.agrosyst.web.actions.practiced;

/*
 * #%L
 * Agrosyst :: Web
 * $Id: PracticedSystemsEdit.java 4972 2015-06-09 13:14:05Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.1/agrosyst-web/src/main/java/fr/inra/agrosyst/web/actions/practiced/PracticedSystemsEdit.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.google.common.base.Strings;
import com.google.common.collect.Collections2;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.gson.reflect.TypeToken;
import com.opensymphony.xwork2.Preparable;
import fr.inra.agrosyst.api.NavigationContext;
import fr.inra.agrosyst.api.entities.AgrosystInterventionType;
import fr.inra.agrosyst.api.entities.CropCyclePhaseType;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.MaterielTransportUnit;
import fr.inra.agrosyst.api.entities.MaterielWorkRateUnit;
import fr.inra.agrosyst.api.entities.OrchardFrutalForm;
import fr.inra.agrosyst.api.entities.PollinatorSpreadMode;
import fr.inra.agrosyst.api.entities.Price;
import fr.inra.agrosyst.api.entities.PriceUnit;
import fr.inra.agrosyst.api.entities.VineFrutalForm;
import fr.inra.agrosyst.api.entities.WeedType;
import fr.inra.agrosyst.api.entities.action.AbstractAction;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystem;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystemImpl;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystemSource;
import fr.inra.agrosyst.api.entities.referential.RefFertiOrga;
import fr.inra.agrosyst.api.entities.referential.RefInterventionAgrosystTravailEDI;
import fr.inra.agrosyst.api.entities.referential.RefOrientationEDI;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.domain.CroppingPlanSpeciesDto;
import fr.inra.agrosyst.api.services.domain.CroppingPlans;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemFilter;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemService;
import fr.inra.agrosyst.api.services.practiced.CropCycleModelDto;
import fr.inra.agrosyst.api.services.practiced.PracticedCropCycleConnectionDto;
import fr.inra.agrosyst.api.services.practiced.PracticedCropCycleNodeDto;
import fr.inra.agrosyst.api.services.practiced.PracticedCropCyclePhaseDto;
import fr.inra.agrosyst.api.services.practiced.PracticedInterventionDto;
import fr.inra.agrosyst.api.services.practiced.PracticedPerennialCropCycleDto;
import fr.inra.agrosyst.api.services.practiced.PracticedSeasonalCropCycleDto;
import fr.inra.agrosyst.api.services.practiced.PracticedSystemService;
import fr.inra.agrosyst.api.services.referential.MineralProductType;
import fr.inra.agrosyst.services.common.CommonService;
import fr.inra.agrosyst.web.actions.effective.EffectiveCropCyclesEdit;
import fr.inra.agrosyst.web.actions.itk.AbstractItkAction;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.struts2.convention.annotation.Action;
import org.apache.struts2.convention.annotation.Result;
import org.apache.struts2.convention.annotation.Results;

import java.lang.reflect.Type;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Action d'édition d'un système synthétisé.
 *
 * Avec:
 * <ul>
 * <li>l'onglet géneralité
 * <li>la liste des cycles pluriannuels assolés
 * <li>la liste des cycles pluriannuels pérennes
 * <li>l'itinéraire technique
 * </ul>
 *
 * @author Eric Chatellier
 */
@Results({
        @Result(name="success", type="redirectAction", params = {"namespace", "/practiced", "actionName", "practiced-systems-edit-input", "practicedSystemTopiaId", "${practicedSystem.topiaId}" })})
public class PracticedSystemsEdit extends AbstractItkAction implements Preparable {

    /**
     * serialVersionUID.
     */
    private static final long serialVersionUID = -5696853256365484417L;

    protected transient PracticedSystemService practicedSystemService;

    protected transient GrowingSystemService growingSystemService;

    /**
     * L'id du systeme synthétisé en cours d'edition.
     */
    protected String practicedSystemId;

    /**
     * L'instance du systeme synthétisé en cours d'edition.
     */
    protected PracticedSystem practicedSystem;

    /**
     * Code du domaine
     */
    protected String domainCode;

    /**
     * La liste de systeme de culture auquel peut être ratachés le SP (en creation).
     */
    protected List<GrowingSystem> growingSystems;

    /**
     * Le systeme de culture sélectionné.
     */
    protected String growingSystemTopiaId;

    /**
     * La liste des campaignes.
     */
    protected String campaigns;

    protected List<PracticedPerennialCropCycleDto> practicedPerennialCropCycleDtos;

    protected List<RefOrientationEDI> refOrientationEDIs;

    protected List<RefInterventionAgrosystTravailEDI> agrosystActionsFullList;

    protected List<MineralProductType> mineralProductTypes;

    protected List<RefFertiOrga> organicProductTypes;

    /**
     * Liste des cycles pluriannuels assolé.
     */
    protected List<PracticedSeasonalCropCycleDto> practicedSeasonalCropCycleDtos;

    /**
     * Contient une map &lt;culture, List&lt;Species&gt;&gt; pour toutes les cultures des systemes de cultures
     * concernés sur les campagnes du systeme synthétisé.
     */
    protected Map<String, List<CroppingPlanSpeciesDto>> practicedSystemCroppingPlanEntryCodesToSpecies;

    /**
     * Les cultures principales disponibles dans le graphique.
     */
    protected List<CropCycleModelDto> practicedSystemMainCropCycleModels;

    /**
     * Les cultures intermédiaires disponible dans le graphique.
     */
    protected List<CropCycleModelDto> practicedSystemIntermediateCropCycleModels;

    protected List<Price> prices;

    protected Map indexedOtherObjectPrices;

    public void setPracticedSystemService(PracticedSystemService practicedSystemService) {
        this.practicedSystemService = practicedSystemService;
    }

    public void setGrowingSystemService(GrowingSystemService growingSystemService) {
        this.growingSystemService = growingSystemService;
    }

    public PracticedSystem getPracticedSystem() {
        if (practicedSystem == null) {
            // EChatellier 27/06/2013 Fais chier de devoir écrire ça, mais c'est la seule option pour ne pas avoir une grosse dose d'exceptions avec du paramsPrepareParams
            return new PracticedSystemImpl();
        }
        return practicedSystem;
    }

    @Override
    public void prepare() {
        if (StringUtils.isNotBlank(practicedSystemId)) {
            practicedSystem = practicedSystemService.getPracticedSystem(practicedSystemId);

        } else {
            // j'offre la possibilité de crééer un PracticedCropCycle
            practicedSystem = new PracticedSystemImpl();
        }
    }

    @Override
    @Action("practiced-systems-edit-input")
    public String input() {
        initForInput();

        // default to empty lists
        practicedSeasonalCropCycleDtos = Lists.newArrayList();
        practicedPerennialCropCycleDtos = Lists.newArrayList();

        if (practicedSystem.isPersisted()) {
            authorizationService.checkPracticedSystemReadable(practicedSystemId);

            readOnly = !authorizationService.isPracticedSystemWritable(practicedSystemId);
            if (readOnly) {
                notificationSupport.practicedSystemNotWritable();
            }

            practicedPerennialCropCycleDtos = practicedSystemService.getAllPracticedPerennialCropCycles(practicedSystemId);

            practicedSeasonalCropCycleDtos = practicedSystemService.getAllPracticedSeasonalCropCycles(practicedSystemId);
            populateSeasonalCropCycles(practicedSeasonalCropCycleDtos);

        }

        prices = practicedSystemService.getPracticedPrices(practicedSystemId);
        if (prices != null) {
            indexedOtherObjectPrices = Maps.uniqueIndex(prices, EffectiveCropCyclesEdit.GET_PRICE_KEY);
        } else {
            indexedOtherObjectPrices = Maps.newHashMap();
        }

        return INPUT;
    }

    protected void populateSeasonalCropCycles(List<PracticedSeasonalCropCycleDto> practicedSeasonalCropCycleDtos) {

        if (practicedSystemIntermediateCropCycleModels != null && practicedSeasonalCropCycleDtos != null) {

            for (PracticedSeasonalCropCycleDto practicedSeasonalCropCycleDto : practicedSeasonalCropCycleDtos) {

                List<PracticedCropCycleConnectionDto> connections = practicedSeasonalCropCycleDto.getCropCycleConnectionDtos();
                // valuation du label des connexions
                if (connections != null) {
                    for (PracticedCropCycleConnectionDto connection : connections) {
                        final String expectedCode = connection.getIntermediateCroppingPlanEntryCode();
                        String label = "";

                        if (!Strings.isNullOrEmpty(expectedCode)) {
                            Optional<CropCycleModelDto> optional = Iterables.tryFind(practicedSystemIntermediateCropCycleModels, new Predicate<CropCycleModelDto>() {
                                @Override
                                public boolean apply(CropCycleModelDto input) {
                                    return expectedCode.equals(input.getCroppingPlanEntryCode());
                                }
                            });
                            if (optional.isPresent()) {
                                if (connection.getCroppingPlanEntryFrequency() != null) {
                                    label = connection.getCroppingPlanEntryFrequency().intValue() + "%<br/><b>CI</b>" +
                                            "<span class='hover-infos'>" + optional.get().getLabel() + "</span>";
                                } else {
                                    label = "<b>CI</b><span class='hover-infos'>" + optional.get().getLabel() + "</span>";
                                }
                            }
                        } else {
                            if (connection.getCroppingPlanEntryFrequency() != null) {
                                label += connection.getCroppingPlanEntryFrequency().intValue() + "%";
                            }
                        }
                        connection.setLabel(label);
                    }
                }

            }

        }

    }

    @Override
    protected void initForInput() {
        if (getPracticedSystem().isPersisted()) {
            // warning, practicedSystem's growingSystem must always be part of growingSystems set
            // even not selected be navigation context
            GrowingSystem growingSystem = getPracticedSystem().getGrowingSystem();
            growingSystemTopiaId = growingSystem.getTopiaId();
            domainCode = growingSystem.getGrowingPlan().getDomain().getCode();
            growingSystems = Collections.singletonList(growingSystem);
            campaigns = practicedSystem.getCampaigns();
        } else {
            GrowingSystemFilter growingSystemFilter = new GrowingSystemFilter();
            NavigationContext navigationContext = getNavigationContext();
            growingSystemFilter.setNavigationContext(navigationContext);
            growingSystemFilter.setActive(Boolean.TRUE);
            growingSystemFilter.setPageSize(GrowingSystemFilter.ALL_PAGE_SIZE);
            ResultList<GrowingSystem> resultList = growingSystemService.getFilteredGrowingSystems(growingSystemFilter);
            growingSystems = resultList.getElements();
        }

        // chargement des référentiels
        refOrientationEDIs = referentialService.findAllReferentielEDI();

        // initialisation des données chargés en json autrement sur la page
        if (!Strings.isNullOrEmpty(growingSystemTopiaId) && !Strings.isNullOrEmpty(campaigns)) {
            // chargement des toutes les cultures pour les systems de cultures et les années
            Map<CropCycleModelDto, List<CroppingPlanSpeciesDto>> modelToSpecies =
                    practicedSystemService.getCropCycleModelMap(growingSystemTopiaId, campaigns, true, getPracticedSystem().isPersisted());
            Set<CropCycleModelDto> croppingPlanEntryDtos = modelToSpecies.keySet();

            // definition de la liste de culture principale
            Iterable<CropCycleModelDto> modelMain = Iterables.filter(croppingPlanEntryDtos, CroppingPlans.IS_NOT_INTERMEDIATE);
            practicedSystemMainCropCycleModels = Lists.newArrayList(modelMain); // force no lazy

            // définition de la liste de culture intermédiaire
            Iterable<CropCycleModelDto> modelIntermediate = Iterables.filter(croppingPlanEntryDtos, CroppingPlans.IS_INTERMEDIATE);
            practicedSystemIntermediateCropCycleModels = Lists.newArrayList(modelIntermediate); // force no lazy

            // chargement de la map 'code culture' > liste d'espece
            practicedSystemCroppingPlanEntryCodesToSpecies = Maps.newHashMap();
            for (Map.Entry<CropCycleModelDto, List<CroppingPlanSpeciesDto>> entry : modelToSpecies.entrySet()) {
                practicedSystemCroppingPlanEntryCodesToSpecies.put(entry.getKey().getCroppingPlanEntryCode(), entry.getValue());
            }
        }
        super.initForInput();
    }

    @Override
    public void validate() {
        if (StringUtils.isBlank(getPracticedSystem().getTopiaId())) {
            if (StringUtils.isBlank(growingSystemTopiaId)) {
                addFieldError("growingSystemTopiaId", "Le système de culture est requis !");
            }
        } else {
            growingSystemTopiaId = getPracticedSystem().getGrowingSystem().getTopiaId();
        }

        if (StringUtils.isBlank(practicedSystem.getName())) {
            addFieldError("practicedSystem.name", "Le nom du système synthétisé est requis !");
        }
        String campaigns = practicedSystem.getCampaigns();
        if (StringUtils.isBlank(campaigns)) {
            addFieldError("practicedSystem.campaigns", "La série de campagne est obligatoire");
        }
        boolean validcampaigns = false;
        if (StringUtils.isNotBlank(campaigns)) {
            validcampaigns = CommonService.ARE_CAMPAIGNS_VALIDS(campaigns);
            if (!validcampaigns) {
                String format = "La série de campagne doit être composée d'années séparées par des virgules (,), " +
                        "espaces ( ) ou point-virgules (;). Les campagnes doivent être contenues entre %d et %d.";
                Pair<Integer, Integer> bounds = CommonService.GET_CAMPAIGNS_BOUNDS();
                addFieldError("practicedSystem.campaigns", String.format(format, bounds.getLeft(), bounds.getRight()));
            }

        }

        List<String> requiredCropcycleCode = Lists.newArrayList();

        if (practicedSeasonalCropCycleDtos != null && !practicedSeasonalCropCycleDtos.isEmpty()) {
            practicedSeasonalCropCycleValidate(requiredCropcycleCode);
        }
        if (practicedPerennialCropCycleDtos != null && !practicedPerennialCropCycleDtos.isEmpty()) {
            practicedPerennialCycleValidate(requiredCropcycleCode);
        }

        if (growingSystemTopiaId != null && validcampaigns) {
            List<String> availableCroppingPlanCodes = practicedSystemService.getCropCodesFromGrowingSystemIdForCampaigns(growingSystemTopiaId, campaigns);
            Boolean allCropCycleFound = availableCroppingPlanCodes.containsAll(requiredCropcycleCode);
            if (!allCropCycleFound) {
                // FIXME echatellier 20140227 : message non compréhensible par un utilisateur
                addActionError("Cultures non présentes pour cette série de campagnes agricoles !");
                addFieldError("practicedSystem.campaigns", "Série de campagnes agricoles non valide !");
            }
        }

        if (hasErrors()) {
            initForInput();
        }
    }

    protected void practicedSeasonalCropCycleValidate(List<String> requiredCropcycleCode) {

        for (PracticedSeasonalCropCycleDto practicedSeasonalCropCycleDto : practicedSeasonalCropCycleDtos) {
            Map<String, PracticedCropCycleNodeDto> nodesById = new HashMap<String, PracticedCropCycleNodeDto>();
            Integer firstRank = null;
            // gestion des proportions de départ de noeud de rang 1
            List<PracticedCropCycleNodeDto> cropCycleNodeDtos = practicedSeasonalCropCycleDto.getCropCycleNodeDtos();
            if (cropCycleNodeDtos != null && cropCycleNodeDtos.size() > 0) {
                // verification de la somme des proportions
                List<PracticedCropCycleNodeDto> nullFrequencyNodes = Lists.newArrayList();
                double sum = 0.0;
                for (PracticedCropCycleNodeDto cropCycleNodeDto : cropCycleNodeDtos) {
                    nodesById.put(cropCycleNodeDto.getNodeId(), cropCycleNodeDto);
                    firstRank= firstRank == null ? cropCycleNodeDto.getX() : (firstRank < cropCycleNodeDto.getX() ? firstRank : cropCycleNodeDto.getX());
                    if (cropCycleNodeDto.getX() == 0) {
                        if (cropCycleNodeDto.getInitNodeFrequency() == null) {
                            nullFrequencyNodes.add(cropCycleNodeDto);
                        } else {
                            sum += cropCycleNodeDto.getInitNodeFrequency();
                        }
                    }
                    requiredCropcycleCode.add(cropCycleNodeDto.getCroppingPlanEntryCode());
                }
                if (firstRank == null || firstRank != 0) {
                    addActionError("Le premier noeud doit se situer sur le premier rang");
                }

                if (sum > 100) {
                    addActionError("La somme des fréquences initiales des cultures de rang 1 dépassent 100%");
                    break;
                }

                // affectation du pourcentage restant si nécéssaire
                if (!nullFrequencyNodes.isEmpty()) {
                    double remaining = 100.0 - sum;
                    int size = nullFrequencyNodes.size();
                    for (PracticedCropCycleNodeDto connection : nullFrequencyNodes) {
                        double value = remaining / size;
                        value = Math.floor(value * 100.0) / 100.0;
                        connection.setInitNodeFrequency(value);
                    }
                    // XXX total may not be 100% here due to ceil
                }
            }

            // gestion des proportions des connections
            List<PracticedCropCycleConnectionDto> cropCycleConnectionDtos = practicedSeasonalCropCycleDto.getCropCycleConnectionDtos();
            if (cropCycleConnectionDtos != null) {

                Multimap<String, PracticedCropCycleConnectionDto> connectionsBySource = HashMultimap.create();

                for (PracticedCropCycleConnectionDto cropCycleConnectionDto : cropCycleConnectionDtos) {

                    PracticedCropCycleNodeDto source = nodesById.get(cropCycleConnectionDto.getSourceId());
                    PracticedCropCycleNodeDto target = nodesById.get(cropCycleConnectionDto.getTargetId());
                    if(source.getX() == target.getX()){
                        addActionError("Deux noeuds reliés entre eux ne peuvent être sur le même rang");
                    } else if (source.getX()>target.getX()) {
                        if (!source.isEndCycle()) {
                            addActionError("Un retour sur cycle ne peut-être créé qu'à partir d'un noeud fin de cycle");
                        }
                        if (target.getX() != 0) {
                            addActionError("Un retour sur cycle ne peut se faire que sur le premier noeud du cycle");
                        }
                    }

                    List<PracticedInterventionDto> interventionDtos = cropCycleConnectionDto.getInterventions();
                    validatePracticedIntervention(interventionDtos);
                    connectionsBySource.put(cropCycleConnectionDto.getSourceId(), cropCycleConnectionDto);
                }

                for (String source : connectionsBySource.keySet()) {
                    Collection<PracticedCropCycleConnectionDto> connections = connectionsBySource.get(source);
                    Collection<PracticedCropCycleConnectionDto> noFrequencyConnections =
                            Collections2.filter(connections, new Predicate<PracticedCropCycleConnectionDto>() {
                                @Override
                                public boolean apply(PracticedCropCycleConnectionDto practicedCropCycleConnectionDto) {
                                    return practicedCropCycleConnectionDto.getCroppingPlanEntryFrequency() == null;
                                }
                            });

                    double sum = 0;
                    for (PracticedCropCycleConnectionDto connection : connections) {
                        Double freq = connection.getCroppingPlanEntryFrequency();
                        if (freq != null) {
                            sum += freq;
                        }
                    }
                    if (sum > 100) {
                        addActionError("La somme des fréquences des connexions sortantes d'une culture " +
                                "du cycle pluriannuel synthétisé de culture assolée doit être égal à 100%");
                        break;
                    }

                    if (!noFrequencyConnections.isEmpty()) {
                        double remaining = 100.0 - sum;
                        int size = noFrequencyConnections.size();
                        for (PracticedCropCycleConnectionDto connection : noFrequencyConnections) {
                            double value = remaining / size;
                            value = Math.floor(value * 100.0) / 100.0;
                            connection.setCroppingPlanEntryFrequency(value);
                        }
                        // XXX total may not be 100% here due to ceil
                    }
                }
            }
        }
    }

    protected void practicedPerennialCycleValidate(List<String> requiredCropcycleCode) {

        double solOccupationPercent = 0d;
        // validate perennial cycle
        for (PracticedPerennialCropCycleDto practicedPerennialCropCycleDto : practicedPerennialCropCycleDtos) {
            if (practicedPerennialCropCycleDto.getPracticedPerennialCropCycle().getSolOccupationPercent() < 0) {
                addActionError("Le pourcentage dans la sole du SdC d'une culture pérenne ne peut être négatif");
            }
            solOccupationPercent += practicedPerennialCropCycleDto.getPracticedPerennialCropCycle().getSolOccupationPercent();
            if(practicedPerennialCropCycleDto.getPracticedPerennialCropCycle().getWeedType() == null) {
                addActionError("Un cycle de culture perenne doit avoir un type d'enherbement de sélectionnée");
            }

            String cropcycleCode = practicedPerennialCropCycleDto.getPracticedPerennialCropCycle().getCroppingPlanEntryCode();
            if (Strings.isNullOrEmpty(cropcycleCode)) {
                addActionError("Un cycle de culture pérenne doit avoir une culture de sélectionnée");
                addFieldError("croppingPlanEntry", "Champ obligatoire");
            } else {
                requiredCropcycleCode.add(cropcycleCode);
            }

            List<PracticedCropCyclePhaseDto> cropCyclePhaseDtos = practicedPerennialCropCycleDto.getCropCyclePhaseDtos();
            if (cropCyclePhaseDtos == null) {
                addActionError("Un cycle de culture pérenne doit avoir une phase de pleine production");
            } else {
                int instCount = 0, montCount = 0, pleiCount = 0, declCount = 0;
                for (PracticedCropCyclePhaseDto cropCyclePhaseDto : cropCyclePhaseDtos) {
                    if (cropCyclePhaseDto.getType() != null) {
                        switch (cropCyclePhaseDto.getType()) {
                            case INSTALLATION:
                                instCount++;
                                break;
                            case MONTEE_EN_PRODUCTION:
                                montCount++;
                                break;
                            case PLEINE_PRODUCTION:
                                pleiCount++;
                                break;
                            case DECLIN_DE_PRODUCTION:
                                declCount++;
                                break;
                            default:
                                break;
                        }

                    }
                    List<PracticedInterventionDto> interventionDtos = cropCyclePhaseDto.getInterventions();
                    validatePracticedIntervention(interventionDtos);
                }
                if (instCount > 1) {
                    addActionError("Un cycle de culture pérenne ne peut avoir qu'une seule phase d'installation");
                }
                if (montCount > 1) {
                    addActionError("Un cycle de culture pérenne ne peut avoir qu'une seule phase de montée en production");
                }
                if (pleiCount == 0) {
                    addActionError("Un cycle de culture pérenne doit avoir une phase de pleine production");
                } else if (pleiCount > 1) {
                    addActionError("Un cycle de culture pérenne ne peut avoir qu'une seule phase de pleine production");
                }
                if (declCount > 1) {
                    addActionError("Un cycle de culture pérenne ne peut avoir qu'une seule phase de déclin de production");
                }
            }
        }

        if (solOccupationPercent > 100d) {
            addActionError("Le pourcentage dans la sole du SdC d'une culture pérenne ne peut être > 100");
        }
        if (solOccupationPercent == 0d) {
            addActionError("Le pourcentage dans la sole du SdC d'une culture pérenne ne peut être égal à 0");
        }
    }

    protected void validatePracticedIntervention(List<PracticedInterventionDto> interventionDtos) {
        if (interventionDtos != null) {
            List<String> availableToolsCouplingCodes;
            String campaigns = practicedSystem.getCampaigns();
            boolean isToolsCouplingsCodeLoadable = false;
            if (StringUtils.isNotBlank(growingSystemTopiaId)) {
                if (StringUtils.isNotBlank(campaigns)) {
                    boolean validCampaigns = CommonService.ARE_CAMPAIGNS_VALIDS(campaigns);
                    if (validCampaigns) {
                        isToolsCouplingsCodeLoadable = true;
                    }
                }
            }
            if (isToolsCouplingsCodeLoadable) {
                availableToolsCouplingCodes = practicedSystemService.getToolsCouplingsFromGrowingSystemAndCampaigns(growingSystemTopiaId, campaigns);
            } else {
                availableToolsCouplingCodes = Lists.newArrayListWithCapacity(0);
            }

            for (PracticedInterventionDto interventionDto : interventionDtos) {
                if (interventionDto.getToolsCouplingCodes() != null) {
                    if (!availableToolsCouplingCodes.containsAll(interventionDto.getToolsCouplingCodes())) {
                        // FIXME echatellier 20140227 : message non compréhensible par un utilisateur
                        addActionError("Les combinaisons d'outils sélectionnées n'existent pas pour cette série de campagnes agricoles !");
                    }
                }

                if (Strings.isNullOrEmpty(interventionDto.getName())) {
                    addActionError("Une intervention doit être nommée");
                }
                if (interventionDto.getType() == null) {
                    addActionError("Le type d'intervention est obligatoire");
                }

                if (Strings.isNullOrEmpty(interventionDto.getStartingPeriodDate())) {
                    addActionError("Une date d'intervention ou date de début est obligatoire");
                }
                if (Strings.isNullOrEmpty(interventionDto.getEndingPeriodDate())) {
                    addActionError("Une date de fin d'intervention est obligatoire");
                }

                Map<AgrosystInterventionType, List<AbstractAction>> actionsByAgrosystInterventionType =  validActions(interventionDto.getActions());

                validInputs(actionsByAgrosystInterventionType, interventionDto.getInputs());
            }
        }
    }

    //@Action(results = {@Result(type = "redirectAction", params = {"actionName", "practiced-systems-edit-input", "practicedSystemTopiaId", "${practicedSystem.topiaId}"})})
    @Override
    public String execute() throws Exception {
        PracticedSystem practicedSystem = getPracticedSystem();
        // can define domain only during create action
        if (StringUtils.isBlank(practicedSystem.getTopiaId())) {
            GrowingSystem growingSystem = growingSystemService.getGrowingSystem(growingSystemTopiaId);
            getPracticedSystem().setGrowingSystem(growingSystem);
        }

        removeEmptySeasonalCropCycle();

        practicedSystem = practicedSystemService.createOrUpdatePracticedSystem(
                practicedSystem,
                practicedPerennialCropCycleDtos,
                practicedSeasonalCropCycleDtos,
                prices);

        notificationSupport.practicedSystemSaved(practicedSystem);

        return SUCCESS;
    }

    /**
     * The last seasonal crop cycle can be empty has one is always created if it's case then it is removed.
     */
    protected void removeEmptySeasonalCropCycle() {
        if (practicedSeasonalCropCycleDtos != null && !practicedSeasonalCropCycleDtos.isEmpty()) {
            List<PracticedSeasonalCropCycleDto> pseccCopy = Lists.newArrayList(practicedSeasonalCropCycleDtos);
            for (PracticedSeasonalCropCycleDto practicedSeasonalCropCycleDto : pseccCopy) {
                if (practicedSeasonalCropCycleDto.getCropCycleNodeDtos() == null || practicedSeasonalCropCycleDto.getCropCycleNodeDtos().isEmpty()) {
                    practicedSeasonalCropCycleDtos.remove(practicedSeasonalCropCycleDto);
                }
            }
        }
    }

    public List<GrowingSystem> getGrowingSystems() {
        return growingSystems;
    }

    public String getGrowingSystemsJson() {
        return getGson().toJson(growingSystems);
    }

    public void setPracticedSystemTopiaId(String practicedSystemTopiaId) {
        this.practicedSystemId = practicedSystemTopiaId;
    }

    public String getGrowingSystemTopiaId() {
        return growingSystemTopiaId;
    }

    public void setGrowingSystemTopiaId(String growingSystemTopiaId) {
        this.growingSystemTopiaId = growingSystemTopiaId;
    }

    public Map<PracticedSystemSource, String> getSources() {
        return getEnumAsMap(PracticedSystemSource.values());
    }

    public List<RefOrientationEDI> getRefOrientationEDIs() {
        return refOrientationEDIs;
    }

    public void setPracticedSystemId(String practicedSystemId) {
        this.practicedSystemId = practicedSystemId;
    }

    public String getPracticedSystemId() {
        return practicedSystemId;
    }

    public Map<VineFrutalForm, String> getVineFrutalForms() {
        return getEnumAsMap(VineFrutalForm.values());
    }

    public Map<OrchardFrutalForm, String> getOrchardFrutalForms() {
        return getEnumAsMap(OrchardFrutalForm.values());
    }

    public Map<PollinatorSpreadMode, String> getPollinatorSpreadModes() {
        return getEnumAsMap(PollinatorSpreadMode.values());
    }

    public Map<CropCyclePhaseType, String> getPerennialPhaseTypes() {
        return getEnumAsMap(CropCyclePhaseType.values());
    }

    public Map<WeedType, String> getWeedTypes() {
        return getEnumAsMap(WeedType.values());
    }

    public String getCampaigns() {
        String result = "";
        if (practicedSystem != null) {
            result = practicedSystem.getCampaigns();
        }
        return getGson().toJson(result);
    }

    public Map<AgrosystInterventionType, String> getAgrosystInterventionTypes() {
        return getEnumAsMap(AgrosystInterventionType.values());
    }

    public void setPracticedPerennialCropCycleDtosJson(String json) {
        Type type = new TypeToken<List<PracticedPerennialCropCycleDto>>() {
        }.getType();
        this.practicedPerennialCropCycleDtos = getGson().fromJson(json, type);
    }

    public void setPracticedSeasonalCropCycleDtosJson(String json) {
        Type type = new TypeToken<List<PracticedSeasonalCropCycleDto>>() {
        }.getType();
        this.practicedSeasonalCropCycleDtos = getGson().fromJson(json, type);
    }

    public List<PracticedPerennialCropCycleDto> getPracticedPerennialCropCycleDtos() {
        return practicedPerennialCropCycleDtos;
    }

    public List<PracticedSeasonalCropCycleDto> getPracticedSeasonalCropCycleDtos() {
        return practicedSeasonalCropCycleDtos;
    }

    public Map<String, List<CroppingPlanSpeciesDto>> getPracticedSystemCroppingPlanEntryCodesToSpecies() {
        return practicedSystemCroppingPlanEntryCodesToSpecies;
    }

    public List<CropCycleModelDto> getPracticedSystemMainCropCycleModels() {
        return practicedSystemMainCropCycleModels;
    }

    public List<CropCycleModelDto> getPracticedSystemIntermediateCropCycleModels() {
        return practicedSystemIntermediateCropCycleModels;
    }

    public List<Price> getPrices() {
        return prices;
    }

    public void setPricesJson(String json) {
        Type type = new TypeToken<List<Price>>() {
        }.getType();
        this.prices = getGson().fromJson(json, type);
    }

    public String getDomainCode() {
        return domainCode;
    }

    public Map<PriceUnit, String> getPriceUnits() {
        return getEnumAsMap(PriceUnit.values());
    }

    public Map getIndexedOtherObjectPrices() {
        return indexedOtherObjectPrices;
    }

    public Map<MaterielWorkRateUnit, String> getMaterielWorkRateUnits() {
        Map<MaterielWorkRateUnit, String> materielWorkRateUnits = getEnumAsMap(MaterielWorkRateUnit.values());
        return materielWorkRateUnits;
    }

    public Map<MaterielTransportUnit, String> getMaterielTransportUnits() {
        Map<MaterielTransportUnit, String> materielTransportUnits = getEnumAsMap(MaterielTransportUnit.values());
        return materielTransportUnits;
    }
}
