package fr.inra.agrosyst.web.actions.domains;

/*
 * #%L
 * Agrosyst :: Web
 * $Id: DomainsEdit.java 5101 2015-10-15 10:13:49Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.1/agrosyst-web/src/main/java/fr/inra/agrosyst/web/actions/domains/DomainsEdit.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Predicate;
import com.google.common.base.Strings;
import com.google.common.collect.Collections2;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.gson.reflect.TypeToken;
import com.opensymphony.xwork2.Preparable;
import fr.inra.agrosyst.api.entities.AgrosystInterventionType;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.DomainImpl;
import fr.inra.agrosyst.api.entities.DomainType;
import fr.inra.agrosyst.api.entities.Entities;
import fr.inra.agrosyst.api.entities.Equipment;
import fr.inra.agrosyst.api.entities.GeoPoint;
import fr.inra.agrosyst.api.entities.Ground;
import fr.inra.agrosyst.api.entities.MaterielTransportUnit;
import fr.inra.agrosyst.api.entities.MaterielType;
import fr.inra.agrosyst.api.entities.MaterielWorkRateUnit;
import fr.inra.agrosyst.api.entities.Plot;
import fr.inra.agrosyst.api.entities.Price;
import fr.inra.agrosyst.api.entities.PriceUnit;
import fr.inra.agrosyst.api.entities.ToolsCoupling;
import fr.inra.agrosyst.api.entities.WeatherStation;
import fr.inra.agrosyst.api.entities.Zoning;
import fr.inra.agrosyst.api.entities.referential.RefInterventionAgrosystTravailEDI;
import fr.inra.agrosyst.api.entities.referential.RefLegalStatus;
import fr.inra.agrosyst.api.entities.referential.RefLocation;
import fr.inra.agrosyst.api.entities.referential.RefMaterielAutomoteur;
import fr.inra.agrosyst.api.entities.referential.RefMaterielIrrigation;
import fr.inra.agrosyst.api.entities.referential.RefStationMeteo;
import fr.inra.agrosyst.api.services.common.UsageList;
import fr.inra.agrosyst.api.services.domain.CroppingPlanEntryDto;
import fr.inra.agrosyst.api.services.domain.CroppingPlanSpeciesDto;
import fr.inra.agrosyst.api.services.domain.DomainService;
import fr.inra.agrosyst.api.services.plot.PlotService;
import fr.inra.agrosyst.api.services.referential.ReferentialService;
import fr.inra.agrosyst.services.common.CommonService;
import fr.inra.agrosyst.web.actions.AbstractAgrosystAction;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.ReflectionToStringBuilder;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts2.convention.annotation.Action;
import org.apache.struts2.convention.annotation.Result;

import java.lang.reflect.Type;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * Action d'édition d'un domaine.
 *
 * @author Eric Chatellier
 */
public class DomainsEdit extends AbstractAgrosystAction implements Preparable {

    protected static final Predicate<CroppingPlanSpeciesDto> HAS_NO_SPECIES = new Predicate<CroppingPlanSpeciesDto>() {

        @Override
        public boolean apply(CroppingPlanSpeciesDto speciesDto) {
            return speciesDto.getSpeciesId() == null;
        }
    };

    protected static final String SIRET_REGEX = "[0-9]{3}[\\s]?[0-9]{3}[\\s]?[0-9]{3}[\\s]?[0-9]{5}";


    private static final Log log = LogFactory.getLog(DomainsEdit.class);

    private static final String REQUIRED_FIELD = "Champ obligatoire";

    private static final long serialVersionUID = 390873886885798716L;

    protected transient ReferentialService referentialService;

    protected transient DomainService domainService;

    protected transient PlotService plotService;
    
    protected Integer campaign;

    protected Domain domain;

    protected LinkedHashMap<Integer, String> relatedDomains; // TODO AThimel 18/10/13 Using implementation to make sure order is kept

    protected List<GeoPoint> geoPoints;

    protected String domainTopiaId;

    protected String commune;

    protected String legalStatusId;

    protected List<RefLegalStatus> allRefLegalStatus;

    protected RefLocation location;

    protected String departement;
    protected String petiteRegionAgricole;
    protected String petiteRegionAgricoleName;

    protected String locationTopiaId;

    protected Map<MaterielType, List<String>> materielType1List;
    protected List<Equipment> equipments;
    protected List<ToolsCoupling> toolsCouplings;
    protected UsageList<ToolsCoupling> toolsCouplingsUsageList;

    protected Map<Integer, String> solArvalisRegions;

    protected List<Ground> grounds;

    protected List<CroppingPlanEntryDto> croppingPlans;
    protected UsageList<CroppingPlanEntryDto> croppingPlansUsageList;
    protected Map<String, Boolean> croppingPlanSpeciesUsageMap;

    protected Integer otex18;

    protected Integer otex70;

    protected Map<Integer, String> otex18s;

    protected List<Plot> plots;

    protected Collection<WeatherStationDto> weatherStationDtos;

    protected Map<String, String> allRefStationMeteoIdsAndNames;

    protected List<Price> prices;

    protected List<RefInterventionAgrosystTravailEDI> agrosystActionsFullList;

    protected boolean canEditPlots;

    public void setReferentialService(ReferentialService referentialService) {
        this.referentialService = referentialService;
    }

    public void setDomainService(DomainService domainService) {
        this.domainService = domainService;
    }

    public void setPlotService(PlotService plotService) {
        this.plotService = plotService;
    }


    public void setDomainTopiaId(String domainTopiaId) {
        this.domainTopiaId = domainTopiaId;
    }

    public void setLocationTopiaId(String locationTopiaId) {
        this.locationTopiaId = locationTopiaId;
    }

    public void setCommune(String commune) {
        this.commune = commune;
    }

    public void setCampaign(Integer campaign) {
        this.campaign = campaign;
    }

    public void setGeoPointsJson(String gpsDataDtos) {
        Type type = new TypeToken<List<GeoPoint>>() {
        }.getType();
        this.geoPoints = getGson().fromJson(gpsDataDtos, type);
    }

    public void setLegalStatusId(String legalStatusId) {
        this.legalStatusId = legalStatusId;
    }

    public void setOtex18(Integer otex18) {
        this.otex18 = otex18;
    }

    public void setOtex70(Integer otex70) {
        this.otex70 = otex70;
    }

    public void setWeatherStationsJson(String json) {
        Type type = new TypeToken<List<WeatherStationDto>>() {
        }.getType();
        this.weatherStationDtos = getGson().fromJson(json, type);
    }

    public void setGroundsJson(String json) {
        Type type = new TypeToken<List<Ground>>() {
        }.getType();
        this.grounds = getGson().fromJson(json, type);
    }

    public void setEquipmentsJson(String json) {
        Type type = new TypeToken<List<Equipment>>() {
        }.getType();
        this.equipments = getGson().fromJson(json, type);
    }

    public void setToolsCouplingsJson(String json) {
        Type type = new TypeToken<List<ToolsCoupling>>() {
        }.getType();
        this.toolsCouplings = getGson().fromJson(json, type);
    }

    public void setCroppingPlansJson(String json) {
        Type type = new TypeToken<List<CroppingPlanEntryDto>>() {
        }.getType();
        this.croppingPlans = getGson().fromJson(json, type);
    }

    @Override
    public void prepare() throws Exception {

        if (StringUtils.isEmpty(domainTopiaId)) {
            // Cas de création d'un domain
            domain = domainService.newDomain();

        } else {

            // TODO AThimel 07/10/13 May be included directly in the service ?
            authorizationService.checkDomainReadable(domainTopiaId);

            // Cas d'une mise à jour de domain
            readOnly = !authorizationService.isDomainWritable(domainTopiaId);
            if (readOnly) {
                notificationSupport.domainNotWritable();
            }

            domain = domainService.getDomain(domainTopiaId);

            if (StringUtils.isBlank(locationTopiaId)) {
                // Cas ou la commune n'a pas étée modifiée
                locationTopiaId = domain.getLocation().getTopiaId();
            }

        }

        if (!StringUtils.isBlank(locationTopiaId)) {
            location = referentialService.getRefLocation(locationTopiaId);
            domain.setLocation(location);

            departement = location.getDepartement();

            petiteRegionAgricole = location.getPetiteRegionAgricoleCode();
            petiteRegionAgricoleName = location.getPetiteRegionAgricoleNom();
        }

        campaign = domain.getCampaign() == 0 ? null : domain.getCampaign();
        prices = domainService.getDomainPrices(domainTopiaId);

    }

    public Domain getDomain() {
        if (domain == null) {
            // AThimel 26/06/2013 Fais chier de devoir écrire ça, mais c'est la seule option pour ne pas avoir une grosse dose d'exceptions avec du paramsPrepareParams
            return new DomainImpl();
        }
        return domain;
    }

    @Override
    protected void initForInput() {

        allRefLegalStatus = domainService.getAllRefLegalStatus();
        allRefStationMeteoIdsAndNames = referentialService.getAllRefStationMeteoMap();

        materielType1List = referentialService.getTypeMateriel1List();

        solArvalisRegions = referentialService.getSolArvalisRegions();

        otex18s = referentialService.findAllActiveOtex18Code();

        agrosystActionsFullList = referentialService.findAllActiveAgrosystActions();

        if (StringUtils.isNotEmpty(domainTopiaId)) {
            relatedDomains = domainService.getRelatedDomains(domain.getCode());
            plots = plotService.findAllByDomain(domain);

            canEditPlots = authorizationService.areDomainPlotsEditable(domainTopiaId);

            toolsCouplingsUsageList = domainService.getToolsCouplingAndUsage(domainTopiaId);
            croppingPlansUsageList = domainService.getCroppingPlanEntryDtoAndUsage(domainTopiaId);
            croppingPlanSpeciesUsageMap = domainService.getCroppingPlanSpeciesUsage(domainTopiaId);
        } else {
            canEditPlots = true;
        }
    }

    @Override
    @Action("domains-edit-input")
    public String input() throws Exception {

        initForInput();

        if (StringUtils.isNotEmpty(domainTopiaId)) {
            geoPoints = domainService.getGeoPoints(domainTopiaId);
            equipments = domainService.getEquipments(domainTopiaId);
            grounds = domainService.getGrounds(domainTopiaId);

            toolsCouplings = toolsCouplingsUsageList.getElements();
            croppingPlans = croppingPlansUsageList.getElements();

            if (domain.getLegalStatus() != null) {
                legalStatusId = domain.getLegalStatus().getTopiaId();
            }
            Collection<WeatherStation> weatherStations = domain.getWeatherStations();
            String defaultWeatherStationId = null;

            if (domain.getDefaultWeatherStation() != null) {
                defaultWeatherStationId = domain.getDefaultWeatherStation().getTopiaId();
            }

            weatherStationDtos = Collections2.transform(weatherStations, WeatherStations.getFunctionWeatherStationToDto(defaultWeatherStationId));
        } else {
            geoPoints = Collections.emptyList();
            equipments = Collections.emptyList();
            grounds = Collections.emptyList();
            croppingPlans = Collections.emptyList();
            toolsCouplings = Collections.emptyList();
        }

        return INPUT;
    }

    @Override
    public void validate() {

        // campagne
        if (campaign == null || campaign == 0) {
            addFieldError("campaign", REQUIRED_FIELD);
        } else {
            if (!CommonService.ARE_CAMPAIGNS_VALIDS(Integer.toString(campaign))) {
                addFieldError("campaign", "La campagne n'est pas valide");
            }
        }

        // nom du domain
        if (StringUtils.isBlank(domain.getName())) {
            addFieldError("domain.name", REQUIRED_FIELD);
        }

        // nom de l'interlocuteur principal
        if (StringUtils.isBlank(domain.getMainContact())) {
            addFieldError("domain.mainContact", REQUIRED_FIELD);
        }

        // commune
        if (StringUtils.isBlank(locationTopiaId)) {
            addFieldError("commune", REQUIRED_FIELD);
        }

        // le type de domaine
        if (domain.getType() == null) {
            // we set a default value to avoid npe due to flushing in initForInput();
            domain.setType(DomainType.DOMAINE_EXPERIMENTAL);
            addFieldError("domain.type", REQUIRED_FIELD);
            addActionError("Le type de domaine est obligatoire !");
        }


        if (StringUtils.isNotBlank(domain.getSiret()) && !domain.getSiret().matches(SIRET_REGEX)) {
            addActionError("Le numéro SIRET défini sur l'onglet 'Caractéristiques' n'est pas valide !");
        }

        // validation des stations méteo
        if (weatherStationDtos != null) {
            List<WeatherStationDto> emptyWeatherStationDtos = Lists.newArrayList();
            weatherStationDtos.removeAll(Collections.singleton(null));
            for (WeatherStationDto weatherStationDto : weatherStationDtos) {
                if (StringUtils.isBlank(weatherStationDto.getRefStationMeteoTopiaId())){
                    if (StringUtils.isNotBlank(weatherStationDto.getComment())
                            || StringUtils.isNotBlank(weatherStationDto.getData())) {
                        addActionError("A la saisie d'une station météorologique, le nom de la station INRA est obligatoire !");
                    } else {
                        emptyWeatherStationDtos.add(weatherStationDto);
                    }
                }
            }
            weatherStationDtos.removeAll(emptyWeatherStationDtos);
        }

        // validation des materiels
        if (equipments != null) {
            equipments.removeAll(Collections.singleton(null));
            for (Equipment equipment : equipments) {
                if (StringUtils.isBlank(equipment.getName())) {
                    addActionError("Le nom du matériel est obligatoire !");
                }
            }
        }

        // validation des attelages
        if (toolsCouplings != null) {
            toolsCouplings.removeAll(Collections.singleton(null));
            for (ToolsCoupling toolsCoupling : toolsCouplings) {
                if (toolsCoupling.getMainsActions() == null || toolsCoupling.getMainsActions().isEmpty()) {
                    addFieldError("editedToolsCoupling-mainsActions", "Au moins une action principale doit être définie sur une combinaison d'outils");
                    addActionError("Au moins une action principale doit être définie sur une combinaison d'outils");
                }
                if (StringUtils.isBlank(toolsCoupling.getToolsCouplingName())) {
                    addActionError("Une combinaison d'outils doit avoir un nom");
                }
                if (!toolsCoupling.isManualIntervention()) {
                    Collection<Equipment>  tcEquipments = toolsCoupling.getEquipments();
                    boolean isIrrigationEquipments = false;
                    boolean isNonIrrigationEquipments = false;
                    if (tcEquipments != null && tcEquipments.size() > 0) {
                        for (Equipment tcEquipment : tcEquipments) {
                            boolean isIrrigationEquipment = tcEquipment.getRefMateriel() != null && tcEquipment.getRefMateriel() instanceof RefMaterielIrrigation;
                            if (isIrrigationEquipment) {
                                isIrrigationEquipments = true;
                            } else {
                                isNonIrrigationEquipments = true;
                            }
                        }
                    }

                    Equipment tractor = toolsCoupling.getTractor();
                    // no tractor, can be valid if equipments only contains irrigation tools.
                    if (tractor == null) {
                        if (!isIrrigationEquipments) {
                            addFieldError("editedToolsCoupling-mainsActions", "Combinaison d'outils non valide, aucun tracteur/automoteur/matériel d'irrigation présent");
                            addActionError("Combinaison d'outils non valide, aucun tracteur/automoteur/matériel d'irrigation présent");
                        } else {
                            // if no tractor equipments can only be irrigation type ones.
                            if (CollectionUtils.isNotEmpty(tcEquipments)) {
                                if (isNonIrrigationEquipments) {
                                    addFieldError("editedToolsCoupling-mainsActions", "Du matériel d'irrigation ne peut être associé avec du matériel autre qu'un tracteur ou un automoteur");
                                    addActionError("Combinaison d'outils non valide, du matériel d'irrigation ne peut être associé avec du matériel autre qu'un tracteur ou un automoteur");
                                }
                            }
                        }
                    } else {
                        // if tractor is not auto-motor's type, it must be associated to 1+n equipment
                        if (tractor.getRefMateriel() == null || !(tractor.getRefMateriel() instanceof RefMaterielAutomoteur)) {
                            if (CollectionUtils.isEmpty(tcEquipments)) {
                                addActionError("Au moins un équipement doit être associé à un tracteur sur une combinaison d'outils comportant du matriel autre que pour l'irrigation");
                            }
                        }
                        if (CollectionUtils.isNotEmpty(tcEquipments)) {
                            if (isIrrigationEquipments && isNonIrrigationEquipments) {
                                addFieldError("editedToolsCoupling-mainsActions", "Du matériel d'irrigation ne peut être associé avec du matériel autre qu'un tracteur ou un automoteur");
                                addActionError("Combinaison d'outils non valide, du matériel d'irrigation ne peut être associé avec du matériel autre qu'un tracteur ou un automoteur");
                            }
                        }
                    }

                }
            }
        }

        if (geoPoints != null) {
            geoPoints.removeAll(Collections.singleton(null));
            List<GeoPoint> emptyGeoPoints = Lists.newArrayList();
            for (GeoPoint geoPoint : geoPoints) {
                if (Strings.isNullOrEmpty(geoPoint.getName())) {
                    if (geoPoint.getLatitude() != 0d
                            || geoPoint.getLongitude() != 0d
                            || StringUtils.isNotBlank(geoPoint.getDescription())){
                        addActionError("A la saisie d'une coordonnée GPS, le nom du centre est obligatoire !");
                    } else {
                        emptyGeoPoints.add(geoPoint);
                    }
                }
            }
            geoPoints.removeAll(emptyGeoPoints);
        }

        // validation des sols
        if (grounds != null) {
            grounds.removeAll(Collections.singleton(null));
            for (Ground ground : grounds) {
                if (ground.getRefSolArvalis() == null) {
                    addActionError("La région est obligatoire sur un sol !");
                }
            }
        }

        if (croppingPlans != null) {
            croppingPlans.removeAll(Collections.singleton(null));
            for (CroppingPlanEntryDto croppingPlanEntryDto : croppingPlans) {
                if (StringUtils.isBlank(croppingPlanEntryDto.getName())) {
                    addActionError("Le nom d'une culture est obligatoire !");
                }
                croppingPlanEntryDto.getSpecies().removeAll(Collections.singleton(null));
                Iterables.removeIf(croppingPlanEntryDto.getSpecies(), HAS_NO_SPECIES);
            }
        }

        if (prices != null) {
            prices.removeAll(Collections.singleton(null));
            for (Price price : prices) {
                if (Strings.isNullOrEmpty(price.getCategory())) {
                    addActionError("Un prix doit avoir une catégorie de sélectionné!");
                }
            }
        }

        if (hasErrors()) {
            log.debug("validate, action errors : " + getActionErrors().toString());
            log.debug("validate, fields errors : " + getFieldErrors().toString());

            initForInput();
        }
    }

    @Override
    @Action(results = {
            @Result(type = "redirectAction", params = {"actionName", "domains-edit-input", "domainTopiaId", "${domain.topiaId}"})})
    public String execute() throws Exception {

        propagateWeatherStationsChanges();

        domain.setCampaign(campaign);
        domain = domainService.createOrUpdateDomain(domain, locationTopiaId, legalStatusId, geoPoints, croppingPlans,
                otex18, otex70, grounds, equipments, toolsCouplings, prices);
        notificationSupport.domainSaved(domain);

        if (Strings.isNullOrEmpty(domainTopiaId)) {
            navigationContextEntityCreated(domain);
        }

        return SUCCESS;
    }

    /**
     * Propagates changes on {@link Domain#getWeatherStations()}.
     */
    protected void propagateWeatherStationsChanges() {
        Collection<WeatherStation> weatherStations = getDomain().getWeatherStations();
        List<WeatherStation> nonDeleted = Lists.newArrayList();

        if (weatherStations == null) {
            weatherStations = Lists.newArrayList();
            getDomain().setWeatherStations(weatherStations);
        }

        Map<String, WeatherStation> indexWeatherStations = Maps.uniqueIndex(weatherStations, Entities.GET_TOPIA_ID);

        getDomain().setDefaultWeatherStation(null);

        if (weatherStationDtos != null) {
            for (WeatherStationDto weatherStationDto : weatherStationDtos) {
                String topiaId = weatherStationDto.getTopiaId();
                WeatherStation weatherStation;

                if (Strings.isNullOrEmpty(topiaId)) {
                    weatherStation = domainService.newWeatherStation();
                } else {
                    weatherStation = indexWeatherStations.get(topiaId);
                }

                if (weatherStation != null) {
                    String refStationId = weatherStationDto.getRefStationMeteoTopiaId();
                    RefStationMeteo refStation = domainService.findRefStationMeteoByTopiaId(refStationId);
                    WeatherStations.dtoToWeatherStation(weatherStationDto, weatherStation, refStation);

                    if (StringUtils.isBlank(topiaId)) {
                        weatherStations.add(weatherStation);
                    }

                    nonDeleted.add(weatherStation);

                    if (log.isDebugEnabled()) {
                        log.debug("adding weatherStation : ID : " + weatherStation.getTopiaId() + ", " + ReflectionToStringBuilder.toString(weatherStationDto));
                    }

                    // sets the default weather station
                    if (weatherStationDto.isDefaultSelected()) {
                        getDomain().setDefaultWeatherStation(weatherStation);
                    }
                }
            }
        }

        weatherStations.retainAll(nonDeleted);
    }

    /**
     * Get all the type that a a domain could be.
     *
     * @return all the type
     */
    public Map<DomainType, String> getTypes() {
        return getEnumAsMap(DomainType.values());
    }

    /**
     * Get all the agrosystInterventionType
     *
     * @return all the agrosystInterventionType
     */
    public Map<AgrosystInterventionType, String> getAgrosystInterventionTypes() {
        Map<AgrosystInterventionType, String> agrosystInterventionTypes = getEnumAsMap(AgrosystInterventionType.values());
        return agrosystInterventionTypes;
    }

    /**
     * Get all the VulnerableArea that a a domain could be.
     *
     * @return all the VulnerableArea
     */
    public Map<Zoning, String> getZoningValues() {
        return getEnumAsMap(Zoning.values());
    }

    public String getDepartement() {
        return departement;
    }

    public String getFormatedDepartement() {
        if (StringUtils.isBlank(departement)) {
            return "";
        }
        String depNumber = Strings.padStart(departement, 2, '0');
        String key = "departement." + depNumber;
        String result = getText(key);
        result = result + " (" + depNumber + ")";
        return result;
    }

    public String getPetiteRegionAgricole() {
        return petiteRegionAgricole;
    }

    public String getPetiteRegionAgricoleName() {
        return petiteRegionAgricoleName;
    }

    public String getFormatedPetiteRegionAgricoleName() {
        if (StringUtils.isBlank(petiteRegionAgricoleName)) {
            return "";
        }
        String result = petiteRegionAgricoleName + " (" + petiteRegionAgricole + ")";
        return result;
    }

    public List<GeoPoint> getGeoPoints() {
        return geoPoints;
    }

    public String getCommune() {
        return commune;
    }

    public Integer getCampaign() {
        return campaign;
    }

    public String getLegalStatusId() {
        return legalStatusId;
    }

    public List<RefLegalStatus> getAllRefLegalStatus() {
        return allRefLegalStatus;
    }

    public Map<String, String> getAllRefStationMeteoIdsAndNames() {
        return allRefStationMeteoIdsAndNames;
    }

    public String getLocationTopiaId() {
        return locationTopiaId;
    }

    public List<Equipment> getEquipments() {
        return equipments;
    }

    public Map<MaterielType, List<String>> getMaterielType1List() {
        return materielType1List;
    }

    public List<Ground> getGrounds() {
        return grounds;
    }

    public Map<Integer, String> getSolArvalisRegions() {
        return solArvalisRegions;
    }

    public Map<Integer, String> getRelatedDomains() {
        return relatedDomains;
    }

    public List<CroppingPlanEntryDto> getCroppingPlans() {
        return croppingPlans;
    }

    public Map<String, Boolean> getCroppingPlansUsageMap() {
        if (croppingPlansUsageList != null) {
            return croppingPlansUsageList.getUsageMap();
        } else {
            return Maps.newHashMap();
        }
    }

    public Integer getOtex18() {
        return otex18;
    }

    public Map<Integer, String> getOtex18s() {
        return otex18s;
    }

    public List<Plot> getPlots() {
        return plots;
    }

    public Collection<WeatherStationDto> getWeatherStations() {
        return weatherStationDtos;
    }

    public List<Price> getPrices() {
        return prices;
    }

    public Map<AgrosystInterventionType, String> getAgrosystInterventionType() {
        return getEnumAsMap(AgrosystInterventionType.values());
    }

    public List<RefInterventionAgrosystTravailEDI> getAgrosystActionsFullList() {
        return agrosystActionsFullList;
    }

    public Map<PriceUnit, String> getPriceUnits() {
        return getEnumAsMap(PriceUnit.values());
    }

    public void setPricesJson(String json) {
        Type type = new TypeToken<List<Price>>() {
        }.getType();
        this.prices = getGson().fromJson(json, type);
    }

    public boolean isCanEditPlots() {
        return canEditPlots;
    }

    public List<ToolsCoupling> getToolsCouplings() {
        return toolsCouplings;
    }

    public Map<String, Boolean> getToolsCouplingsUsageMap() {
        if (toolsCouplingsUsageList != null) {
            return toolsCouplingsUsageList.getUsageMap();
        } else {
            return Maps.newHashMap();
        }
    }

    public Map<String, Boolean> getCroppingPlanSpeciesUsageMap() {
        if (croppingPlanSpeciesUsageMap == null) {
            croppingPlanSpeciesUsageMap = Maps.newHashMap();
        }
        return croppingPlanSpeciesUsageMap;
    }

    public Map<MaterielWorkRateUnit, String> getMaterielWorkRateUnits() {
        Map<MaterielWorkRateUnit, String> materielWorkRateUnits = getEnumAsMap(MaterielWorkRateUnit.values());
        return materielWorkRateUnits;
    }

    public Map<MaterielTransportUnit, String> getMaterielTransportUnits() {
        Map<MaterielTransportUnit, String> materielTransportUnits = getEnumAsMap(MaterielTransportUnit.values());
        return materielTransportUnits;
    }
}
