package fr.inra.agrosyst.web.actions.itk;

/*
 * #%L
 * Agrosyst :: Web
 * $Id: AbstractItkAction.java 4537 2014-11-07 14:44:55Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.4.1/agrosyst-web/src/main/java/fr/inra/agrosyst/web/actions/itk/AbstractItkAction.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import com.google.common.collect.Maps;

import fr.inra.agrosyst.api.entities.AgrosystInterventionType;
import fr.inra.agrosyst.api.entities.BioAgressorType;
import fr.inra.agrosyst.api.entities.action.AbstractAction;
import fr.inra.agrosyst.api.entities.action.AbstractInput;
import fr.inra.agrosyst.api.entities.action.BiologicalControlActionImpl;
import fr.inra.agrosyst.api.entities.action.CapacityUnit;
import fr.inra.agrosyst.api.entities.action.FertiOrgaUnit;
import fr.inra.agrosyst.api.entities.action.HarvestingAction;
import fr.inra.agrosyst.api.entities.action.HarvestingYeald;
import fr.inra.agrosyst.api.entities.action.IrrigationActionImpl;
import fr.inra.agrosyst.api.entities.action.MineralProductInput;
import fr.inra.agrosyst.api.entities.action.MineralProductUnit;
import fr.inra.agrosyst.api.entities.action.OrganicProductUnit;
import fr.inra.agrosyst.api.entities.action.PesticidesSpreadingActionImpl;
import fr.inra.agrosyst.api.entities.action.PhytoProductUnit;
import fr.inra.agrosyst.api.entities.action.SeedPlantUnit;
import fr.inra.agrosyst.api.entities.action.SeedType;
import fr.inra.agrosyst.api.entities.action.SeedingAction;
import fr.inra.agrosyst.api.entities.action.SeedingActionSpecies;
import fr.inra.agrosyst.api.entities.action.YealdCategory;
import fr.inra.agrosyst.api.entities.action.YealdUnit;
import fr.inra.agrosyst.api.entities.referential.RefFertiMinUNIFA;
import fr.inra.agrosyst.api.entities.referential.RefFertiOrga;
import fr.inra.agrosyst.api.entities.referential.RefInterventionAgrosystTravailEDI;
import fr.inra.agrosyst.api.services.itk.Itk;
import fr.inra.agrosyst.api.services.input.Inputs;
import fr.inra.agrosyst.api.services.referential.MineralProductType;
import fr.inra.agrosyst.api.services.referential.ReferentialService;
import fr.inra.agrosyst.web.actions.AbstractAgrosystAction;
import org.apache.commons.collections4.CollectionUtils;

/**
 * @author Arnaud Thimel : thimel@codelutin.com
 */
public abstract class AbstractItkAction extends AbstractAgrosystAction {

    private static final long serialVersionUID = -6057182538439163217L;

    protected transient ReferentialService referentialService;


    protected List<MineralProductType> mineralProductTypes;
    protected List<RefFertiOrga> organicProductTypes;
    protected Map<AgrosystInterventionType, List<String>> actaTreatmentProductTypes;
    protected List<RefInterventionAgrosystTravailEDI> agrosystActionsFullList;

    public final void setReferentialService(ReferentialService referentialService) {
        this.referentialService = referentialService;
    }

    @Override
    protected void initForInput() {

        agrosystActionsFullList = referentialService.findAllActiveAgrosystActions();
        mineralProductTypes = referentialService.findAllActiveMineralProductTypes();
        organicProductTypes = referentialService.findAllActiveOrganicProductTypes();
        actaTreatmentProductTypes = referentialService.getAllActiveActaTreatmentProductTypes();

        super.initForInput();
    }

    protected void validateActions(Collection<AbstractAction> actions) {
        if (CollectionUtils.isEmpty(actions)) {
            addActionError("Une intervention doit avoir au moins une action");
        } else {
            List<AgrosystInterventionType> usedActionTypes = new ArrayList<AgrosystInterventionType>();
            for (AbstractAction action : actions) {
                if (action.getMainAction() == null) {
                    addActionError("Une action n'a pas de type d'intervention");
                } else {
                    AgrosystInterventionType interventionType = action.getMainAction().getIntervention_agrosyst();
                    if (interventionType != AgrosystInterventionType.ENTRETIEN_TAILLE_VIGNE_ET_VERGER && usedActionTypes.contains(interventionType)) {
                        addActionError("Il n'est possible de définir qu'une seule action de type "+getText(interventionType.getClass().getName()+"."+interventionType.toString()));
                    } else {
                        usedActionTypes.add(interventionType);
                    }
                    switch (action.getMainAction().getIntervention_agrosyst()) {
                        case RECOLTE:
                            HarvestingAction harvestingAction = (HarvestingAction)action;
                            if (CollectionUtils.isEmpty(harvestingAction.getHarvestingYealds())) {
                                addActionError("Un rendement est obligatoire pour l'action de récolte !");
                            } else {
                                for (HarvestingYeald harvestingYeald : harvestingAction.getHarvestingYealds()) {
                                    if (harvestingYeald.getYealdCategory() == null) {
                                        addActionError("La categorie de rendement est obligatoire !");
                                    }
                                    if (harvestingYeald.getYealdUnit() == null) {
                                        addActionError("L'unité de rendement est obligatoire !");
                                    }
                                }
                            }
                            break;
                        case APPLICATION_DE_PRODUITS_PHYTOSANITAIRES:
                            PesticidesSpreadingActionImpl pesticidesSpreadingAction = (PesticidesSpreadingActionImpl)action;
                            Double boiledQuantity = pesticidesSpreadingAction.getBoiledQuantity();
                            if (boiledQuantity == null) {
                                addActionError("Le volume moyen de bouillie par hectare est obligatoire pour l'action de traitement phytosanitaire !");
                            }
                            if (pesticidesSpreadingAction.getProportionOfTreatedSurface() <= 0){
                                addActionError("La proportion de surface traitée au sein de la surface de l'intervention est obligatoire pour l'action de traitement phytosanitaire !");
                            }
                            break;
                        case IRRIGATION:
                            IrrigationActionImpl irrigationAction = (IrrigationActionImpl)action;
                            double waterQuantityAverage = irrigationAction.getWaterQuantityAverage();
                            if (waterQuantityAverage <= 0) {
                                addActionError("La quantité d'eau apportée est obligatoire pour l'action d'irrigation !");
                            }
                            break;
                        case LUTTE_BIOLOGIQUE:
                            BiologicalControlActionImpl biologicalControlAction = (BiologicalControlActionImpl)action;
                            if (biologicalControlAction.getProportionOfTreatedSurface() <= 0) {
                                addActionError("La proportion de surface traitée au sein de la surface de l'intervention est obligatoire pour l'action lutte biologique !");
                            }
                            break;
                        case SEMIS:
                            SeedingAction seedingAction = (SeedingAction)action;
                            Collection<SeedingActionSpecies> seedingActionSpecieses = seedingAction.getSeedingSpecies();
                            if (seedingActionSpecieses != null) {
                                for (SeedingActionSpecies seedingActionSpecies : seedingActionSpecieses) {
                                    if (seedingActionSpecies.isBiologicalSeedInoculation() || seedingActionSpecies.isTreatment()) {
                                        if (seedingActionSpecies.getQuantity() == 0 || seedingActionSpecies.getSeedPlantUnit() == null) {
                                            addActionError("La quantité de produit n'est pas définie sur l'action de type 'Semis' dont au moins une espèce est déclarée avoir un traitement chimique des semences / plants soit une inoculation biologique des semences / plants !");
                                        }
                                    }
                                }
                            }
                        default:
                            break;
                    }
                }
            }
        }
    }

    protected void validateInputs(Collection<AbstractInput> inputs) {
        if (!CollectionUtils.isEmpty(inputs)) {
            for (AbstractInput input : inputs) {
                if (input instanceof MineralProductInput) {
                    RefFertiMinUNIFA product = ((MineralProductInput) input).getMineralProduct();
                    if (product == null) {
                        addActionError("Les intrant de type 'Engrais/amendement (organo)minéral' doivent avoir une composition de l'engrais de saisie !");
                    } else {
                        double totalWeight = 0;
                        totalWeight += product.getN() != null ? product.getN() : 0;
                        totalWeight += product.getP2O5() != null ? product.getP2O5() : 0;
                        totalWeight += product.getBore() != null ? product.getBore() : 0;
                        totalWeight += product.getCalcium() != null ? product.getCalcium() : 0;
                        totalWeight += product.getCuivre() != null ? product.getCuivre() : 0;
                        totalWeight += product.getFer() != null ? product.getFer() : 0;
                        totalWeight += product.getK2O() != null ? product.getK2O() : 0;
                        totalWeight += product.getManganese() != null ? product.getManganese() : 0;
                        totalWeight += product.getMgO() != null ? product.getMgO() : 0;
                        totalWeight += product.getMolybdene() != null ? product.getMolybdene() : 0;
                        totalWeight += product.getOxyde_de_sodium() != null ? product.getOxyde_de_sodium() : 0;
                        totalWeight += product.getsO3() != null ? product.getsO3() : 0;
                        totalWeight += product.getZinc() != null ? product.getZinc() : 0;
                        if (totalWeight > 100) {
                            addActionError(" La somme des % poids ne peut être supérieure à 100 pour un intrant de type 'Engrais/amendement (organo)minéral'");
                        }
                    }

                }
            }
        }
    }



    ////////////// ITK specific data //////////////

    public Map<YealdCategory, String> getYealdCategories(){
        return getEnumAsMap(YealdCategory.values());
    }

    public Map<CapacityUnit, String> getCapacityUnits(){
        return getEnumAsMap(CapacityUnit.values());
    }

    public List<MineralProductType> getMineralProductTypes() {
        return mineralProductTypes;
    }

    public Map<BioAgressorType, String> getTreatmentTargetCategories() {
        return getEnumAsMap(Itk.TREATMENT_TARGET_CATEGORIES);
    }

    public Map<SeedType, String> getSeedTypes() {
        return getEnumAsMap(SeedType.values());
    }

    public Map<SeedPlantUnit, String> getSeedPlantUnits() {
        return getEnumAsMap(SeedPlantUnit.values());
    }

    public Map<YealdUnit, String> getYealdUnits() {
        return getEnumAsMap(YealdUnit.values());
    }

    public List<RefFertiOrga> getOrganicProductTypes() {
        return organicProductTypes;
    }

    public Map<MineralProductUnit, String> getMineralProductUnits() {
        return getEnumAsMap(MineralProductUnit.values());
    }

    public Map<OrganicProductUnit, String> getOrganicProductUnits() {
        return getEnumAsMap(OrganicProductUnit.values());
    }

    public Map<PhytoProductUnit, String> getPhytoProductUnits() {
        return getEnumAsMap(PhytoProductUnit.values());
    }

    public Map<AgrosystInterventionType, List<String>> getActaTreatmentProductTypes() {
        return actaTreatmentProductTypes;
    }

    public Map<FertiOrgaUnit, String> getFertiOrgaUnits() {
        return getEnumAsMap(FertiOrgaUnit.values());
    }

    public Map<AgrosystInterventionType, String> getInputTypesLabels() {
        Map<AgrosystInterventionType, String> result = Maps.newLinkedHashMap();
        for (AgrosystInterventionType value : Inputs.INPUT_TYPES) {
            String key = "InputType." + value.name();
            String trans = getText(key);
            result.put(value, trans);
        }
        return result;
    }

    public List<RefInterventionAgrosystTravailEDI> getAgrosystActionsFullList() {
        return agrosystActionsFullList;
    }

}
