package fr.inra.agrosyst.web;

/*
 * #%L
 * Agrosyst :: Web
 * $Id: AgrosystWebSessionListener.java 4085 2014-06-05 09:19:10Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.4.1/agrosyst-web/src/main/java/fr/inra/agrosyst/web/AgrosystWebSessionListener.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.Date;

import javax.servlet.http.HttpSession;
import javax.servlet.http.HttpSessionEvent;
import javax.servlet.http.HttpSessionListener;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.google.common.base.Charsets;
import com.google.common.base.Strings;
import com.google.common.base.Supplier;
import com.google.common.hash.Hashing;

import fr.inra.agrosyst.api.entities.AgrosystTopiaApplicationContext;
import fr.inra.agrosyst.api.entities.AgrosystTopiaPersistenceContext;
import fr.inra.agrosyst.api.services.security.AuthenticationService;
import fr.inra.agrosyst.services.AgrosystServiceConfig;
import fr.inra.agrosyst.services.DefaultServiceContext;

/**
 * @author Arnaud Thimel (Code Lutin)
 */
public class AgrosystWebSessionListener implements HttpSessionListener {

    private static final Log log = LogFactory.getLog(AgrosystWebSessionListener.class);

    public static String obfuscateSessionId(String sessionId) {
        String sessionIdHashed = Hashing.sha1().hashString(sessionId, Charsets.UTF_8).toString();
        String result = "SID=" + sessionIdHashed.substring(24);
        return result;
    }

    @Override
    public void sessionCreated(HttpSessionEvent se) {
        if (log.isInfoEnabled()) {
            String sessionId = se.getSession().getId();
            log.info(String.format("[%s] New session created", obfuscateSessionId(sessionId)));
        }
    }

    @Override
    public void sessionDestroyed(HttpSessionEvent se) {

        HttpSession session = se.getSession();
        String sessionId = session.getId();
        sessionId = obfuscateSessionId(sessionId);

        if (log.isInfoEnabled()) {
            Date created = new Date(session.getCreationTime());
            Date lastAccessed = new Date(session.getLastAccessedTime());

            String message = String.format("[%s] Session about to be destroyed. Created on %s. Last accessed on: %s.",
                    sessionId, created, lastAccessed);
            log.info(message);
        }

        Object value = session.getAttribute(AgrosystWebSession.SESSION_PARAMETER);
        if (value != null) {
            AgrosystWebSession agrosystWebSession = (AgrosystWebSession) value;
            String authenticationToken = agrosystWebSession.getAuthenticationToken();
            if (!Strings.isNullOrEmpty(authenticationToken)) {

                if (log.isInfoEnabled()) {
                    String message = String.format("[%s] User was authenticated, trigger logout", sessionId);
                    log.info(message);
                }

                Object scValue = se.getSession().getServletContext().getAttribute(AgrosystWebApplicationContext.APPLICATION_CONTEXT_PARAMETER);
                if (scValue != null) {
                    AgrosystWebApplicationContext applicationContext = (AgrosystWebApplicationContext) scValue;
                    final AgrosystTopiaApplicationContext topiaApplicationContext = applicationContext.getApplicationContext();

                    AgrosystServiceConfig serviceConfig = (AgrosystServiceConfig) applicationContext.getServiceConfig();
                    Supplier<AgrosystTopiaPersistenceContext> transactionSupplier = new Supplier<AgrosystTopiaPersistenceContext>() {
                        @Override
                        public AgrosystTopiaPersistenceContext get() {
                            return topiaApplicationContext.newPersistenceContext();
                        }
                    };

                    DefaultServiceContext serviceContext = new DefaultServiceContext(serviceConfig, transactionSupplier);
                    try {
                        AuthenticationService authenticationService = serviceContext.newService(AuthenticationService.class);
                        authenticationService.logout(authenticationToken);
                    } finally {
                        serviceContext.getPersistenceContext().close();
                    }
                }
            }
        }
    }
}
