package fr.inra.agrosyst.web;

/*
 * #%L
 * Agrosyst :: Web
 * $Id: AgrosystWebConfig.java 3962 2014-04-02 09:29:30Z athimel $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.4.1/agrosyst-web/src/main/java/fr/inra/agrosyst/web/AgrosystWebConfig.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.Properties;

import fr.inra.agrosyst.api.exceptions.AgrosystTechnicalException;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.config.ApplicationConfig;
import org.nuiton.config.ArgumentsParserException;
import org.nuiton.config.ConfigOptionDef;

/**
 * Web module configuration.
 *
 * @author Eric Chatellier
 */
public class AgrosystWebConfig {

    private static final Log log = LogFactory.getLog(AgrosystWebConfig.class);

    /**
     * Configuration filename.
     */
    protected static final String AGROSYST_DEFAULT_CONF_FILENAME = "agrosyst-default.properties";

    public static final String NAVIGATION_CONTEXT_COOKIE_NAME = "nav.context";

    /**
     * Delegate application config.
     */
    protected ApplicationConfig config;

    public AgrosystWebConfig(String configFileName) {
        try {
            ApplicationConfig defaultConfig = new ApplicationConfig(AGROSYST_DEFAULT_CONF_FILENAME);
            defaultConfig.loadDefaultOptions(WebConfigOption.values());
            defaultConfig.parse();
            if (StringUtils.isNotBlank(configFileName)) {
                Properties flatOptions = defaultConfig.getFlatOptions(false);

                config = new ApplicationConfig(flatOptions, configFileName);
                config.parse();
            } else {
                if (log.isWarnEnabled()) {
                    log.warn("No specific configuration provided, using the default one");
                }
                config = defaultConfig;
            }
        } catch (ArgumentsParserException ex) {
            throw new AgrosystTechnicalException("Can't read configuration", ex);
        }
    }

    public String getApplicationVersion() {
        return config.getOption(WebConfigOption.APPLICATION_VERSION.key);
    }

    public int getListResultsPerPage() {
        return config.getOptionAsInt(WebConfigOption.LIST_RESULTS_PER_PAGE.key);
    }

    public String getIgnKeyjs() {
        return config.getOption(WebConfigOption.IGN_KEYJS.key);
    }

    public long getUploadMaxSize() {
        return config.getOptionAsLong(WebConfigOption.UPLOAD_MAX_SIZE.key);
    }

    public String getUploadAllowedExtensions() {
        return config.getOption(WebConfigOption.UPLOAD_ALLOWED_EXTENSIONS.key);
    }

    public boolean isServicesRemoteEnabled() {
        return config.getOptionAsBoolean(WebConfigOption.SERVICES_REMOTE_ENABLED.getKey());
    }

    public String getServicesRemoteBaseUrl() {
        String result = config.getOption(WebConfigOption.SERVICES_REMOTE_BASE_URL.getKey());
        if (result != null && !result.endsWith("/")) {
            result += "/";
        }
        return result;
    }
    
    public String getContactEmail() {
        return config.getOption(WebConfigOption.CONTACT_EMAIL.getKey());
    }

    public String getInstanceBannerText() {
        return config.getOption(WebConfigOption.INSTANCE_BANNER_TEXT.key);
    }

    public String getInstanceBannerStyle() {
        return config.getOption(WebConfigOption.INSTANCE_BANNER_STYLE.key);
    }

    static enum WebConfigOption implements ConfigOptionDef {

//        FILENAME(ApplicationConfig.CONFIG_FILE_NAME, AGROSYST_DEFAULT_CONF_FILENAME),

        APPLICATION_VERSION("agrosyst.web.application.version", null),

        LIST_RESULTS_PER_PAGE("agrosyst.web.list.results.per.page", "10"),

        IGN_KEYJS("agrosyst.web.ign.keyjs", null),

        SERVICES_REMOTE_ENABLED("agrosyst.web.services.remote.enabled", "false"),

        SERVICES_REMOTE_BASE_URL("agrosyst.web.services.remote.baseUrl", null),

        CONTACT_EMAIL("agrosyst.web.email.contact", "agrosyst-devel@list.forge.codelutin.com"),

        /**
         * Upload config
         */
        UPLOAD_MAX_SIZE("agrosyst.web.upload.maxSize", "10240000"),

        UPLOAD_ALLOWED_EXTENSIONS("agrosyst.web.upload.allowedExtensions", ""),

        INSTANCE_BANNER_TEXT("agrosyst.web.banner.text", null),

        /** Error, warning, info.*/
        INSTANCE_BANNER_STYLE("agrosyst.web.banner.style", null);

        private String key;
        private String defaultValue;

        private WebConfigOption(String key, String defaultValue) {
            this.key = key;
            this.defaultValue = defaultValue;
        }

        @Override
        public String getKey() {
            return key;
        }

        @Override
        public String getDefaultValue() {
            return defaultValue;
        }

        @Override
        public Class<?> getType() {
            return null;
        }

        @Override
        public String getDescription() {
            return null;
        }

        @Override
        public boolean isTransient() {
            return false;
        }

        @Override
        public boolean isFinal() {
            return false;
        }

        @Override
        public void setDefaultValue(String defaultValue) {

        }

        @Override
        public void setTransient(boolean isTransient) {

        }

        @Override
        public void setFinal(boolean isFinal) {

        }
    }
}
