package fr.inra.agrosyst.web.actions.admin;

/*
 * #%L
 * Agrosyst :: Web
 * $Id: TrackedEventsList.java 3264 2014-02-17 10:17:26Z athimel $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.1.1/agrosyst-web/src/main/java/fr/inra/agrosyst/web/actions/admin/TrackedEventsList.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.lang.reflect.Type;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.base.Function;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.gson.reflect.TypeToken;

import fr.inra.agrosyst.api.entities.security.RoleType;
import fr.inra.agrosyst.api.entities.security.TrackedEvent;
import fr.inra.agrosyst.api.entities.security.TrackedEventType;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.security.TrackedEventFilter;
import fr.inra.agrosyst.api.services.security.TrackerService;
import fr.inra.agrosyst.api.services.security.UserRoleDto;
import fr.inra.agrosyst.api.services.users.Users;

/**
 * @author Arnaud Thimel (Code Lutin)
 */
public class TrackedEventsList extends AbstractAdminAction {

    private static final long serialVersionUID = 5158479141426960541L;
    protected static final Function<UserRoleDto,RoleType> GET_ROLE_TYPE = new Function<UserRoleDto, RoleType>() {
        @Override
        public RoleType apply(UserRoleDto input) {
            return input.getType();
        }
    };

    protected Map<String, Set<RoleType>> rolesCache = Maps.newHashMap();

    protected transient TrackerService trackerService;

    public void setTrackerService(TrackerService trackerService) {
        this.trackerService = trackerService;
    }

    protected ResultList<TrackedEventDto> events;

    protected Function<TrackedEvent, TrackedEventDto> getEventToDtoFunction() {
        return new Function<TrackedEvent, TrackedEventDto>() {
            @Override
            public TrackedEventDto apply(TrackedEvent input) {
                TrackedEventDto result = new TrackedEventDto();
                result.setType(input.getType());
                String logKey = TrackedEvent.class.getName() + "." + input.getType();
                String logFormat = getText(logKey);
                Type type = new TypeToken<List<String>>() {
                }.getType();


                List<String> argsList = getGson().fromJson(input.getArgs(), type);
                if (TrackedEventType.USER_ACTIVATION.equals(input.getType())) {
                    String action = "true".equals(argsList.get(1)) ? "Activation" : "Désactivation";
                    argsList = Lists.newArrayList(action, argsList.get(0));
                } else if (TrackedEventType.USER_MOD.equals(input.getType()) || TrackedEventType.USER_MOD_SELF.equals(input.getType())) {
                    String action = "true".equals(argsList.get(1)) ? "avec" : "sans";
                    argsList = Lists.newArrayList(argsList.get(0), action);
                } else if (TrackedEventType.USER_ROLE_ADD.equals(input.getType()) || TrackedEventType.USER_ROLE_REMOVE.equals(input.getType())) {
                    RoleType roleType = RoleType.valueOf(argsList.get(0));
                    String role = getText(RoleType.class.getName() + "." + roleType.name());
                    argsList.set(0, role);
                    if (RoleType.DOMAIN_RESPONSIBLE.equals(roleType)) {
                        logFormat = getText(logKey + ".long");
                        argsList.add(1, "domaine");
                    } else if (RoleType.GROWING_PLAN_RESPONSIBLE.equals(roleType)) {
                        logFormat = getText(logKey + ".long");
                        argsList.add(1, "dispositif");
                    } else if (RoleType.GS_DATA_PROCESSOR.equals(roleType)) {
                        logFormat = getText(logKey + ".long");
                        argsList.add(1, "système de culture");
                    } else if (RoleType.NETWORK_RESPONSIBLE.equals(roleType)) {
                        logFormat = getText(logKey + ".medium");
                        argsList.add(1, "réseau");
                    }
                }
                String log = String.format(logFormat, argsList.toArray());
                result.setDate(input.getDate());
                result.setLog(log);
                result.setAuthor(Users.TO_USER_DTO.apply(input.getAuthor()));
                if (input.getAuthor() != null && result.getAuthor() != null) {
                    result.getAuthor().setRoles(getUserRoles(input.getAuthor().getTopiaId()));
                }
                return result;
            }
        };
    }

    protected Set<RoleType> getUserRoles(String userId) {
        if (!rolesCache.containsKey(userId)) {
            List<UserRoleDto> userRoles = authorizationService.getUserRoles(userId);
            Set<RoleType> roles = Sets.newHashSet(Iterables.transform(userRoles, GET_ROLE_TYPE));
            rolesCache.put(userId, roles);
        }
        return rolesCache.get(userId);
    }

    @Override
    public String execute() throws Exception {
        checkIsAdmin();

        ResultList<TrackedEvent> trackedEvents = trackerService.list(new TrackedEventFilter());

        List<TrackedEventDto> dtos = Lists.transform(trackedEvents.getElements(), getEventToDtoFunction());

        events = ResultList.of(dtos, trackedEvents.getPager());
        return SUCCESS;
    }

    public ResultList<TrackedEventDto> getEvents() {
        return events;
    }


    public Map<TrackedEventType, String> getEventTypes() {
        return getEnumAsMap(TrackedEventType.values());
    }

    public Map<RoleType, String> getRoleTypes() {
        return getEnumAsMap(RoleType.values());
    }

}
