package fr.inra.agrosyst.web;

/*
 * #%L
 * Agrosyst :: Web
 * $Id: RemoteServiceFactory.java 3052 2014-02-03 14:44:10Z athimel $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.1.1/agrosyst-web/src/main/java/fr/inra/agrosyst/web/RemoteServiceFactory.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.Collection;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableSet;
import com.thoughtworks.paranamer.CachingParanamer;
import com.thoughtworks.paranamer.Paranamer;

import fr.inra.agrosyst.api.services.AgrosystService;
import fr.inra.agrosyst.api.services.ServiceFactory;
import fr.inra.agrosyst.commons.gson.AgrosystGsonSupplier;
import fr.inra.agrosyst.commons.http.RequestBuilder;

/**
 * @author Arnaud Thimel : thimel@codelutin.com
 */
public class RemoteServiceFactory implements ServiceFactory {

    private static final Log log = LogFactory.getLog(RemoteServiceFactory.class);

    protected static final String SERVICES_PREFIX = "fr.inra.agrosyst.api.services.";
    public static final Function<Class<?>, String> GET_SERVICE_URI = new Function<Class<?>, String>() {
        @Override
        public String apply(Class<?> input) {
            String raw = input.getName().substring(SERVICES_PREFIX.length());
            String result = raw.replaceAll("[.]", "/");
            return result;
        }
    };
    public static final Set<String> BYPASSED_METHODS = ImmutableSet.of("toString", "hashcode", "equals", "finalize");

    protected static AgrosystGsonSupplier gsonSupplier;

    protected AgrosystWebConfig config;
    protected String authenticationToken;
    protected CachingParanamer paranamer;

    public RemoteServiceFactory(AgrosystWebConfig config, String authenticationToken) {
        this.config = config;
        this.authenticationToken = authenticationToken;
        this.paranamer = new CachingParanamer();
    }

    public AgrosystGsonSupplier getGsonSupplier() {
        if (gsonSupplier == null) {
            gsonSupplier = new AgrosystGsonSupplier();
        }
        return gsonSupplier;
    }

    @Override
    public <E extends AgrosystService> E newService(Class<E> clazz) {

        InvocationHandler handler = new RemoteInvocationHandler(config, paranamer, clazz);

        E result = (E) Proxy.newProxyInstance(clazz.getClassLoader(),
                new Class[]{clazz},
                handler);

        return result;
    }

    @Override
    public <I> I newInstance(Class<I> clazz) {
        throw new UnsupportedOperationException();
    }

    public class RemoteInvocationHandler implements InvocationHandler {

        protected AgrosystWebConfig config;
        protected Paranamer paranamer;
        protected Class<? extends AgrosystService> serviceClass;

        protected String serviceUrl;

        public RemoteInvocationHandler(AgrosystWebConfig config, Paranamer paranamer, Class<? extends AgrosystService> serviceClass) {
            this.config = config;
            this.paranamer = paranamer;
            this.serviceClass = serviceClass;
            Preconditions.checkArgument(serviceClass.getName().startsWith(SERVICES_PREFIX));
            Preconditions.checkState(!Strings.isNullOrEmpty(config.getServicesRemoteBaseUrl()));

            serviceUrl = config.getServicesRemoteBaseUrl() + GET_SERVICE_URI.apply(serviceClass);
        }

        @Override
        public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
            String methodName = method.getName();
            if ("toString".equals(methodName)) {
                return String.format("%s<%s>.toString()", RemoteInvocationHandler.class.getName(), serviceClass.getName());
            } else if ("hashcode".equals(methodName)) {
                return String.format("%s<%s>.hashcode()", RemoteInvocationHandler.class.getName(), serviceClass.getName()).hashCode();
            } else if ("equals".equals(methodName)) {
                return false;
            } else if ("finalize".equals(methodName)) {
                return null;
            } else {
                String url = String.format("%s/%s", serviceUrl, methodName);
                if (log.isDebugEnabled()) {
                    log.debug("Invoke remote service on endpoint: " + url);
                }
                RequestBuilder requestBuilder = new RequestBuilder(url, RemoteServiceFactory.this.getGsonSupplier());
                String[] strings = paranamer.lookupParameterNames(method);
                int index = 0;
                for (String name : strings) {
                    if (args.length >= (index + 1)) {
                        Object value = args[index];
                        if (value != null && value instanceof String) {
                            requestBuilder.addParameter(name, (String) value); // Make sure String is not converted to JSON
                        } else if (value != null && value instanceof Collection) {
                            requestBuilder.addParameter(name, (Collection) value); // Make sure List is not converted to JSON
                        } else {
                            requestBuilder.addParameter(name, value);
                        }
                    }
                    index++;
                }
                if (!Strings.isNullOrEmpty(authenticationToken)) {
                    requestBuilder.addHeader("authenticationToken", authenticationToken);
                }
                Object result;
                try {
                    if (methodName.startsWith("get") || methodName.startsWith("load")) {
                        result = requestBuilder.getJsonAndCloseConnection(method.getGenericReturnType());
                    } else {
                        result = requestBuilder.postAndGetJsonAndCloseConnection(method.getGenericReturnType());
                    }
                } catch (RuntimeException re) {
                    if (log.isWarnEnabled()) {
                        log.warn("An error occurred during remote service call", re);
                    }
                    throw re;
                }
                return result;
            }
        }
    }
}
