package fr.inra.agrosyst.web;

/*
 * #%L
 * Agrosyst :: Web
 * $Id: AgrosystWebInterceptor.java 4208 2014-07-21 10:09:28Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.1.1/agrosyst-web/src/main/java/fr/inra/agrosyst/web/AgrosystWebInterceptor.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.beans.PropertyDescriptor;
import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts2.ServletActionContext;
import org.nuiton.util.beans.BeanUtil;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.base.Supplier;
import com.opensymphony.xwork2.ActionInvocation;
import com.opensymphony.xwork2.interceptor.Interceptor;

import fr.inra.agrosyst.api.entities.AgrosystTopiaApplicationContext;
import fr.inra.agrosyst.api.entities.AgrosystTopiaPersistenceContext;
import fr.inra.agrosyst.api.exceptions.AgrosystTechnicalException;
import fr.inra.agrosyst.api.services.AgrosystService;
import fr.inra.agrosyst.api.services.ServiceFactory;
import fr.inra.agrosyst.api.services.security.AuthenticationService;
import fr.inra.agrosyst.api.services.users.UserDto;
import fr.inra.agrosyst.services.DefaultServiceFactory;
import fr.inra.agrosyst.services.ServiceContext;
import fr.inra.agrosyst.web.actions.AbstractAgrosystAction;
import fr.inra.agrosyst.web.actions.AbstractJsonAction;

/**
 * Interceptor struts qui inject les parametres de type connu (Session, Service, Config) dans
 * l'action Struts avant son execution.
 * 
 * @author Arnaud Thimel : thimel@codelutin.com
 */
public class AgrosystWebInterceptor implements Interceptor {

    private static final long serialVersionUID = 2035088485913506328L;

    private static final Log log = LogFactory.getLog(AgrosystWebInterceptor.class);

    @Override
    public void init() {
        if (log.isInfoEnabled()) {
            log.info("Initializing Agrosyst Web Interceptor");
        }
    }

    @Override
    public String intercept(ActionInvocation invocation) throws Exception {

        Object action = invocation.getAction();

        if (action instanceof AbstractAgrosystAction) {

            AgrosystWebSession agrosystSession = getAgrosystSession(invocation);


            ServiceFactory serviceFactory = newServiceFactory(invocation, agrosystSession.getAuthenticationToken());
            try {
                populateAuthenticatedUser(agrosystSession, serviceFactory);

                Set<PropertyDescriptor> descriptors =
                        BeanUtil.getDescriptors(
                                action.getClass(),
                                BeanUtil.IS_WRITE_DESCRIPTOR);

                // Iterate over writable properties
                for (PropertyDescriptor propertyDescriptor : descriptors) {

                    Class<?> propertyType = propertyDescriptor.getPropertyType();

                    Object toInject = null;

                    if (AgrosystService.class.isAssignableFrom(propertyType)) {
                        Class<? extends AgrosystService> serviceClass =
                                (Class<? extends AgrosystService>) propertyType;
                        toInject = serviceFactory.newService(serviceClass);
                    } else if (AgrosystWebSession.class.isAssignableFrom(propertyType)) {
                        toInject = agrosystSession;
                    } else if (AgrosystWebConfig.class.isAssignableFrom(propertyType)) {
                        toInject = getAgrosystApplicationContext(invocation).getWebConfig();
                    } else if (AgrosystWebApplicationContext.class.isAssignableFrom(propertyType)) {
                        toInject = getAgrosystApplicationContext(invocation);
                    } else if (AgrosystWebNotificationSupport.class.isAssignableFrom(propertyType)) {
                        toInject = new AgrosystWebNotificationSupport(agrosystSession);
                    }

                    if (toInject != null) {

                        if (log.isTraceEnabled()) {
                            log.trace("injecting " + toInject + " in action " + action);
                        }

                        propertyDescriptor.getWriteMethod().invoke(action, toInject);

                    }
                }

                String result = invocation.invoke();
                checkForLayoutData((AbstractAgrosystAction) action);
                return result;
            } catch (Exception eee) {
                
                // rollback transaction only when an exception happen
                // sinon, le serveur devient inutilisable (
                rollbackTransaction(serviceFactory);

                // Make sure exception is logged because of some strange logging behavior
                if (log.isErrorEnabled()) {
                    String message = String.format("Exception during Agrosyst action invocation (action is '%s'): %s",
                            action.getClass().getName(), eee.getMessage());
                    log.error(message, eee);
                }
                throw eee;
            } finally {
                checkClosedPersistenceContext(serviceFactory);
            }

        } else {

            // not an action, just process
            return invocation.invoke();
        }

    }

    protected void checkForLayoutData(AbstractAgrosystAction action) {
        // Do not prepare layout data in Json actions
        if (!(action instanceof AbstractJsonAction)) {
            // Only prepare layout data for a "POST" with errors or "GET" requests
            if (action.hasErrors() || "GET".equals(ServletActionContext.getRequest().getMethod())) {
                action.initLayoutData();
            }
        }
    }

    protected void populateAuthenticatedUser(AgrosystWebSession agrosystSession, ServiceFactory serviceFactory) {
        if (!Strings.isNullOrEmpty(agrosystSession.getAuthenticationToken()) && agrosystSession.getAuthenticatedUser() == null) {
            if (log.isDebugEnabled()) {
                log.debug("Authenticated user DTO is dirty, reloading from service...");
            }
            AuthenticationService authenticationService = serviceFactory.newService(AuthenticationService.class);
            UserDto authenticatedUser = authenticationService.getAuthenticatedUser(agrosystSession.getAuthenticationToken());
            agrosystSession.setAuthenticatedUser(authenticatedUser);
        }
    }

    protected AgrosystWebSession getAgrosystSession(ActionInvocation invocation) {

        AgrosystWebSession result = (AgrosystWebSession) invocation.getInvocationContext().getSession().get(AgrosystWebSession.SESSION_PARAMETER);

        if (result == null) {
            result = new AgrosystWebSession();
            invocation.getInvocationContext().getSession().put(AgrosystWebSession.SESSION_PARAMETER, result);
        }

        return result;
    }

    protected AgrosystWebApplicationContext getAgrosystApplicationContext(ActionInvocation invocation) {

        AgrosystWebApplicationContext applicationContext =
                (AgrosystWebApplicationContext) invocation
                        .getInvocationContext()
                        .getApplication()
                        .get(AgrosystWebApplicationContext.APPLICATION_CONTEXT_PARAMETER);

        Preconditions.checkState(applicationContext != null, AgrosystWebApplicationContext.MISSING_APPLICATION_CONTEXT);

        return applicationContext;
    }

    protected ServiceFactory newServiceFactory(ActionInvocation invocation, String authenticationToken) {

        AgrosystWebApplicationContext applicationContext = getAgrosystApplicationContext(invocation);

        ServiceFactory result = null;
        AgrosystWebConfig config = applicationContext.getWebConfig();
        if (config.isServicesRemoteEnabled()) {
            result = new RemoteServiceFactory(config, authenticationToken);
        } else {

            final AgrosystTopiaApplicationContext topiaApplicationContext = applicationContext.getApplicationContext();
            Preconditions.checkState(topiaApplicationContext != null);

            try {
                Supplier<AgrosystTopiaPersistenceContext> topiaContextSupplier = new Supplier<AgrosystTopiaPersistenceContext>() {
                    @Override
                    public AgrosystTopiaPersistenceContext get() {
                        return topiaApplicationContext.newPersistenceContext();
                    }
                };
                Class<?> serviceContextClass = Class.forName("fr.inra.agrosyst.services.DefaultServiceContext");
                Class<?> serviceConfigClass = Class.forName("fr.inra.agrosyst.services.AgrosystServiceConfig");
                Constructor<?> constructor = serviceContextClass.getConstructor(serviceConfigClass, Supplier.class, String.class);

                Object serviceContext = constructor.newInstance(applicationContext.getServiceConfig(), topiaContextSupplier, authenticationToken);

                Method getServiceFactoryMethod = serviceContextClass.getMethod("getServiceFactory");
                result = (ServiceFactory)getServiceFactoryMethod.invoke(serviceContext);
            } catch (Exception eee) {
                if (log.isErrorEnabled()) {
                    log.error("Unable to create local serviceFactory", eee);
                }
            }
        }
        Preconditions.checkState(result != null, "ServiceFactory not instantiated, check configuration");
        return result;
    }

    protected void checkClosedPersistenceContext(ServiceFactory serviceFactory) {
        try {
            if (serviceFactory instanceof DefaultServiceFactory) {
                ServiceContext serviceContext = ((DefaultServiceFactory) serviceFactory).getServiceContext();
                serviceContext.close();
            }
        } catch (Exception eee) {
            if (log.isFatalEnabled()) {
                log.fatal("Unable to close persistence context !", eee);
            }
            throw new AgrosystTechnicalException("Unable to close persistence context !", eee);
        }
    }
    
    protected void rollbackTransaction(ServiceFactory serviceFactory) {
        try {
            if (serviceFactory instanceof DefaultServiceFactory) {
                AgrosystTopiaPersistenceContext persistenceContext = ((DefaultServiceFactory) serviceFactory).getServiceContext().getPersistenceContext(false);
                if (persistenceContext != null) {
                    persistenceContext.rollback();
                }
            }
        } catch (Exception eee) {
            if (log.isFatalEnabled()) {
                log.fatal("Unable to rollback persistence context !", eee);
            }
            throw new AgrosystTechnicalException("Unable to rollback persistence context !", eee);
        }
    }

    @Override
    public void destroy() {
        if (log.isInfoEnabled()) {
            log.info("Destroy Interceptor");
        }
    }

}
