package fr.inra.agrosyst.web.actions.practiced;

/*
 * #%L
 * Agrosyst :: Web
 * $Id: PracticedPlotsEdit.java 4009 2014-04-16 12:11:09Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.0.6/agrosyst-web/src/main/java/fr/inra/agrosyst/web/actions/practiced/PracticedPlotsEdit.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.lang.reflect.Type;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.google.common.collect.Collections2;
import org.apache.commons.lang3.StringUtils;
import org.apache.struts2.convention.annotation.Action;
import org.apache.struts2.convention.annotation.Result;

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.gson.reflect.TypeToken;
import com.opensymphony.xwork2.Preparable;

import fr.inra.agrosyst.api.entities.BufferStrip;
import fr.inra.agrosyst.api.entities.Entities;
import fr.inra.agrosyst.api.entities.FrostProtectionType;
import fr.inra.agrosyst.api.entities.HosesPositionning;
import fr.inra.agrosyst.api.entities.IrrigationSystemType;
import fr.inra.agrosyst.api.entities.MaxSlope;
import fr.inra.agrosyst.api.entities.PompEngineType;
import fr.inra.agrosyst.api.entities.SolWaterPh;
import fr.inra.agrosyst.api.entities.WaterFlowDistance;
import fr.inra.agrosyst.api.entities.ZoneType;
import fr.inra.agrosyst.api.entities.practiced.PracticedPlot;
import fr.inra.agrosyst.api.entities.practiced.PracticedPlotImpl;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystem;
import fr.inra.agrosyst.api.entities.referential.RefElementVoisinage;
import fr.inra.agrosyst.api.entities.referential.RefParcelleZonageEDI;
import fr.inra.agrosyst.api.entities.referential.RefSolCaracteristiqueIndigo;
import fr.inra.agrosyst.api.entities.referential.RefSolProfondeurIndigo;
import fr.inra.agrosyst.api.entities.referential.RefSolTextureGeppa;
import fr.inra.agrosyst.api.services.plot.Plots;
import fr.inra.agrosyst.api.services.plot.SolHorizonDto;
import fr.inra.agrosyst.api.services.practiced.PracticedPlotService;
import fr.inra.agrosyst.api.services.referential.ReferentialService;
import fr.inra.agrosyst.web.actions.AbstractAgrosystAction;

/**
 * Action de création/edition d'une parcelle type.
 *
 * @author Eric Chatellier
 */
public class PracticedPlotsEdit extends AbstractAgrosystAction implements Preparable {

    private static final long serialVersionUID = -1618279547328450777L;

    private static final String REQUIRED_FIELD = "Champ obligatoire";

    protected transient PracticedPlotService practicedPlotService;
    
    protected transient ReferentialService referentialService;

    /** Edited plot id. */
    protected String practicedPlotTopiaId;

    /** Edited plot instance. */
    protected PracticedPlot practicedPlot;

    /** Practiced system topia id. */
    protected String practicedSystemTopiaId;

    /** Practiced system list (create mode). */
    protected List<PracticedSystem> practicedSystems;

    /** Selected location id. */
    protected String locationTopiaId;

    /** Onglet "zonage", liste de tous les zonages. */
    protected List<RefParcelleZonageEDI> parcelleZonages;

    /** Identifiant selectionnées dans la liste {@code parcelleZonages}. */
    protected List<String> selectedPlotZoningIds;

    /** Liste de toutes les sol texture geppa disponibles. */
    protected List<RefSolTextureGeppa> solTextures; 

    /** Identifiant du surface texture sélectionné. */
    protected String selectedSurfaceTextureId;

    /** Identifiant du sous sol texture sélectionné. */
    protected String selectedSubSoilTextureId;

    /** Liste de toutes les sol profondeur disponibles. */
    protected List<RefSolProfondeurIndigo> solProfondeurs;

    /** Identifiant du sol profondeur sélectionné. */
    protected String selectedSolProfondeurId;

    /** Liste des sols horizons. */
    protected List<SolHorizonDto> solHorizons;

    /** Caracteristiques des sols. */
    protected List<RefSolCaracteristiqueIndigo> solCaracteristiques;

    /** Referentiel des elements de voisinage. */
    protected List<RefElementVoisinage> adjacentElements;
 
    /** Element de voisinage sélectionnés. */
    protected List<String> adjacentElementIds;

    public void setPracticedPlotService(PracticedPlotService practicedPlotService) {
        this.practicedPlotService = practicedPlotService;
    }

    public void setReferentialService(ReferentialService referentialService) {
        this.referentialService = referentialService;
    }

    @Override
    public void prepare() throws Exception {
        practicedPlot = practicedPlotService.getPracticedPlot(practicedPlotTopiaId);

        if (practicedPlot.isPersisted()) {
            authorizationService.checkPracticedPlotReadable(practicedPlotTopiaId);

            readOnly = !authorizationService.isPracticedPlotWritable(practicedPlotTopiaId);
            if (readOnly) {
                notificationSupport.practicedPlotNotWritable();
            }
        }
    }

    @Override
    @Action("practiced-plots-edit-input")
    public String input() throws Exception {
        initForInput();

        if (practicedPlot.getLocation() != null) {
            locationTopiaId = practicedPlot.getLocation().getTopiaId();
        }
        if (practicedPlot.getPlotZonings() != null) {
            selectedPlotZoningIds = Lists.newArrayList(Iterables.transform(practicedPlot.getPlotZonings(), Entities.GET_TOPIA_ID));
        }
        if (practicedPlot.getSurfaceTexture() != null) {
            selectedSurfaceTextureId = practicedPlot.getSurfaceTexture().getTopiaId();
        }
        if (practicedPlot.getSubSoilTexture() != null) {
            selectedSubSoilTextureId = practicedPlot.getSubSoilTexture().getTopiaId();
        }
        if (practicedPlot.getSolDepth() != null) {
            selectedSolProfondeurId = practicedPlot.getSolDepth().getTopiaId();
        }
        if (practicedPlot.getSolHorizon() != null) {
            solHorizons = Lists.newArrayList(Collections2.transform(practicedPlot.getSolHorizon(), Plots.SOL_HORIZON_TO_DTO));
        } else {
            solHorizons = Lists.newArrayList();
        }
        if (practicedPlot.getAdjacentElements() != null) {
            adjacentElementIds = Lists.newArrayList(Iterables.transform(practicedPlot.getAdjacentElements(), Entities.GET_TOPIA_ID));
        } else {
            adjacentElementIds = Lists.newArrayList();
        }

        return INPUT;
    }

    
    @Override
    protected void initForInput() {

        if (!practicedPlot.isPersisted()) {
            practicedSystems = practicedPlotService.getPracticedSystemsWithoutPracticedPlot(getNavigationContext());
        }

        parcelleZonages = referentialService.getAllActiveParcelleZonage();
        solTextures = referentialService.getAllActiveSolTextures();
        solProfondeurs = referentialService.getAllActiveSolProfondeurs();
        solCaracteristiques = referentialService.getAllActiveSolCaracteristiques();
        adjacentElements = referentialService.getAllActiveElementVoisinages();
    }

    @Override
    public void validate() {
        // domain
        if (!practicedPlot.isPersisted() && StringUtils.isBlank(practicedSystemTopiaId)) {
            addActionError("domainTopiaId is required");
        }
        // name
        if (StringUtils.isBlank(practicedPlot.getName())) {
            addFieldError("plot.name", REQUIRED_FIELD);
        }
        
        if (hasErrors()) {
            initForInput();
        }
    }

    @Override
    @Action(results = { @Result(type = "redirectAction", params = { "actionName", "practiced-plots-edit-input", "practicedPlotTopiaId", "${practicedPlot.topiaId}" }) })
    public String execute() throws Exception {
        practicedPlot = practicedPlotService.createOrUpdatePracticedPlot(practicedPlot, practicedSystemTopiaId,
                locationTopiaId, selectedPlotZoningIds, selectedSurfaceTextureId, selectedSubSoilTextureId,
                selectedSolProfondeurId, solHorizons, adjacentElementIds);
        notificationSupport.practicedPlotSaved(practicedPlot);
        return SUCCESS;
    }
    
    public PracticedPlot getPracticedPlot() {
        if (practicedPlot == null) {
            // AThimel 26/06/2013 Fais chier de devoir écrire ça, mais c'est la seule option pour ne pas avoir une grosse dose d'exceptions avec du paramsPrepareParams
            return new PracticedPlotImpl();
        }
        return practicedPlot;
    }

    public Map<MaxSlope, String> getMaxSlopes() {
        return getEnumAsMap(MaxSlope.values());
    }
    
    public Map<BufferStrip, String> getBufferStrips() {
        return getEnumAsMap(BufferStrip.values());
    }
    
    public Map<WaterFlowDistance, String> getWaterFlowDistances() {
        return getEnumAsMap(WaterFlowDistance.values());
    }

    public String getPracticedPlotTopiaId() {
        return practicedPlotTopiaId;
    }

    public void setPracticedPlotTopiaId(String plotTopiaId) {
        this.practicedPlotTopiaId = plotTopiaId;
    }

    public void setLocationTopiaId(String locationTopiaId) {
        this.locationTopiaId = locationTopiaId;
    }

    public String getLocationTopiaId() {
        return locationTopiaId;
    }

    public List<RefParcelleZonageEDI> getParcelleZonages() {
        return parcelleZonages;
    }

    public void setSelectedPlotZoningIds(List<String> selectedPlotZoningIds) {
        this.selectedPlotZoningIds = selectedPlotZoningIds;
    }

    public List<String> getSelectedPlotZoningIds() {
        return selectedPlotZoningIds;
    }

    public Map<IrrigationSystemType, String> getIrrigationSystemTypes() {
        return getEnumAsMap(IrrigationSystemType.values());
    }

    public Map<PompEngineType, String> getPompEngineTypes() {
        return getEnumAsMap(PompEngineType.values());
    }

    public Map<HosesPositionning, String> getHosesPositionnings() {
        return getEnumAsMap(HosesPositionning.values());
    }

    public Map<FrostProtectionType, String> getFrostProtectionTypes() {
        return getEnumAsMap(FrostProtectionType.values());
    }

    public Map<SolWaterPh, String> getSolWaterPhs() {
        return getEnumAsMap(SolWaterPh.values());
    }

    public List<RefSolTextureGeppa> getSolTextures() {
        return solTextures;
    }

    public List<RefSolProfondeurIndigo> getSolProfondeurs() {
        return solProfondeurs;
    }

    public String getSelectedSurfaceTextureId() {
        return selectedSurfaceTextureId;
    }

    public void setSelectedSurfaceTextureId(String selectedSurfaceTextureId) {
        this.selectedSurfaceTextureId = selectedSurfaceTextureId;
    }

    public String getSelectedSubSoilTextureId() {
        return selectedSubSoilTextureId;
    }

    public void setSelectedSubSoilTextureId(String selectedSubSoilTextureId) {
        this.selectedSubSoilTextureId = selectedSubSoilTextureId;
    }

    public String getSelectedSolProfondeurId() {
        return selectedSolProfondeurId;
    }

    public void setSelectedSolProfondeurId(String selectedSolProfondeurId) {
        this.selectedSolProfondeurId = selectedSolProfondeurId;
    }

    public List<SolHorizonDto> getSolHorizons() {
        return solHorizons;
    }

    public void setSolHorizons(String json) {
        Type type = new TypeToken<List<SolHorizonDto>>() {
        }.getType();
        this.solHorizons = getGson().fromJson(json, type);
    }

    public List<RefSolCaracteristiqueIndigo> getSolCaracteristiques() {
        return solCaracteristiques;
    }

    public Map<ZoneType, String> getZoneTypes() {
        return getEnumAsMap(ZoneType.values());
    }

    public List<RefElementVoisinage> getAdjacentElements() {
        return adjacentElements;
    }

    public List<String> getAdjacentElementIds() {
        return adjacentElementIds;
    }

    public void setAdjacentElementIds(List<String> adjacentElementIds) {
        this.adjacentElementIds = adjacentElementIds;
    }

    public String getPracticedSystemTopiaId() {
        return practicedSystemTopiaId;
    }

    public void setPracticedSystemTopiaId(String practicedSystemTopiaId) {
        this.practicedSystemTopiaId = practicedSystemTopiaId;
    }

    public List<PracticedSystem> getPracticedSystems() {
        return practicedSystems;
    }
}
