package fr.inra.agrosyst.web;

/*
 * #%L
 * Agrosyst :: Web
 * $Id: AgrosystWebApplicationContext.java 4085 2014-06-05 09:19:10Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.0.6/agrosyst-web/src/main/java/fr/inra/agrosyst/web/AgrosystWebApplicationContext.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.io.IOException;
import java.io.InputStream;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import javax.servlet.ServletContext;

import com.google.common.collect.Maps;
import fr.inra.agrosyst.api.entities.AgrosystTopiaApplicationContext;
import fr.inra.agrosyst.api.exceptions.AgrosystTechnicalException;
import fr.inra.agrosyst.services.AgrosystConfigurationHelper;
import fr.inra.agrosyst.services.AgrosystServiceConfig;
import fr.inra.agrosyst.services.demo.DemoDatas;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.persistence.TopiaApplicationContextCache;

/**
 * @author Arnaud Thimel : thimel@codelutin.com
 */
public class AgrosystWebApplicationContext {

    private static final Log log = LogFactory.getLog(AgrosystWebApplicationContext.class);

    public static final String APPLICATION_CONTEXT_PARAMETER = AgrosystWebApplicationContext.class.getName();

    public static final String CONFIG_FILE_PREFIX = "agrosyst-";
    public static final String CONFIG_FILE_EXT = ".properties";
    public static final String DEFAULT_CONTEXT_NAME = "ROOT";

    public static final String MISSING_APPLICATION_CONTEXT =
            AgrosystWebApplicationContext.class.getSimpleName() + " not found. You probably forgot to" +
                    " register " + AgrosystWebApplicationListener.class.getName() + " in your web.xml";

    protected AgrosystWebConfig webConfig;

    protected AgrosystServiceConfig serviceConfig;

    protected AgrosystTopiaApplicationContext applicationContext;

    /** Banner id =&lt; {banner path, banner copyright}. */
    protected Map<String, String[]> bannersMap = Maps.newHashMap();

    public void init(ServletContext servletContext) {

        String configFileName = servletContext.getContextPath();
        if (configFileName != null) {
            String baseName = StringUtils.remove(configFileName, "/");
            if (StringUtils.isBlank(baseName)) {
                baseName = DEFAULT_CONTEXT_NAME;
            }
            configFileName = StringUtils.join(CONFIG_FILE_PREFIX, baseName, CONFIG_FILE_EXT);
        }
        webConfig = new AgrosystWebConfig(configFileName);

        initApplicationContext(configFileName);
    }

    protected void initApplicationContext(String configFileName) {
        if (!webConfig.isServicesRemoteEnabled()) {
            try {
//                Class<?> defaultConfigSupplierClass = Class.forName("fr.inra.agrosyst.services.AgrosystConfigurationHelper");
//                serviceConfig = defaultConfigSupplierClass.getMethod("getDefaultConfiguration").invoke(null);
//
//                Class<?> applicationContextClass = Class.forName("fr.inra.agrosyst.api.entities.AgrosystTopiaApplicationContext");
//                serviceConfig = defaultConfigSupplierClass.getConstructor(Properties.class);
                serviceConfig = new AgrosystServiceConfig(configFileName);

                Properties contextProperties = AgrosystConfigurationHelper.getRootContextProperties(serviceConfig);
                applicationContext = TopiaApplicationContextCache.getContext(contextProperties, AgrosystConfigurationHelper.getCreateTopiaContextFunction(serviceConfig));
                applicationContext.applicationInit();

            } catch (Exception eee) {
                throw new AgrosystTechnicalException("An exception occurred", eee);
            }
        }
    }

    protected void initApplicationBanners(ServletContext servletContext) {
        Set<String> resources = servletContext.getResourcePaths("/img/Header-Backgrounds/");
        for (String resource : resources) {
            String extension = FilenameUtils.getExtension(resource);
            String basename = FilenameUtils.getBaseName(resource);
            if ("jpg".equals(extension)) {
                String metapath = resource.replace("." + extension, ".txt");
                InputStream metastream = null;
                try {
                    metastream = servletContext.getResourceAsStream(metapath);
                    String metacontent = IOUtils.toString(metastream);
                    
                    if (log.isDebugEnabled()) {
                        log.debug("Adding banner " + basename + "(" + metacontent + ")");
                    }

                    // make banner path context relative
                    String resourceContextPath = StringUtils.removeStart(resource, "/");
                    bannersMap.put(basename, new String[]{resourceContextPath, metacontent});
                } catch (IOException ex) {
                    throw new AgrosystTechnicalException("Can't read banner metadata", ex);
                } finally {
                    IOUtils.closeQuietly(metastream);
                }
            }
        }
        if (log.isInfoEnabled()) {
            log.info("Detecting " + bannersMap.size() + " banners");
        }
    }

    protected void checkForDemoData() {
        if (!getWebConfig().isServicesRemoteEnabled()) {
            try {
                DemoDatas.onInit(serviceConfig);
            } catch (Exception eee) {
                throw new AgrosystTechnicalException("Unable to initialize data", eee);
            }
        }
    }

    public AgrosystWebConfig getWebConfig() {
        return webConfig;
    }

    public Object getServiceConfig() {
        return serviceConfig;
    }

    public AgrosystTopiaApplicationContext getApplicationContext() {
        return applicationContext;
    }

    public Map<String, String[]> getBannersMap() {
        return bannersMap;
    }

    public void close() {
        if (applicationContext != null && !applicationContext.isClosed()) {
            applicationContext.close();
        }
    }

}
