package fr.inra.agrosyst.web.actions;

/*
 * #%L
 * Agrosyst :: Web
 * $Id: AbstractAgrosystAction.java 4014 2014-04-17 16:19:02Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.0.5/agrosyst-web/src/main/java/fr/inra/agrosyst/web/actions/AbstractAgrosystAction.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Objects;
import com.google.common.base.Strings;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.gson.Gson;
import com.opensymphony.xwork2.ActionSupport;

import fr.inra.agrosyst.api.NavigationContext;
import fr.inra.agrosyst.api.services.common.AttachmentService;
import fr.inra.agrosyst.api.services.context.NavigationContextService;
import fr.inra.agrosyst.api.services.security.BusinessAuthorizationService;
import fr.inra.agrosyst.api.services.users.UserDto;
import fr.inra.agrosyst.commons.gson.AgrosystGsonSupplier;
import fr.inra.agrosyst.web.AgrosystWebApplicationContext;
import fr.inra.agrosyst.web.AgrosystWebConfig;
import fr.inra.agrosyst.web.AgrosystWebNotification;
import fr.inra.agrosyst.web.AgrosystWebNotificationSupport;
import fr.inra.agrosyst.web.AgrosystWebSession;
import fr.inra.agrosyst.web.AgrosystWebSessionListener;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts2.interceptor.ParameterAware;
import org.apache.struts2.interceptor.ServletRequestAware;
import org.apache.struts2.interceptor.ServletResponseAware;
import org.nuiton.topia.persistence.TopiaEntity;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import java.util.Map;
import java.util.Set;

/**
 * Toutes les actions Struts doivent hériter de cette classe.
 *
 * @author Arnaud Thimel : thimel@codelutin.com
 */
public abstract class AbstractAgrosystAction extends ActionSupport implements ServletRequestAware, ServletResponseAware,
        ParameterAware {

    private static final long serialVersionUID = 4829352350362628085L;

    private static final Log log = LogFactory.getLog(AbstractAgrosystAction.class);

    protected static final AgrosystLayoutData ERROR_LAYOUT_DATA = new AgrosystLayoutData();
    static {
        ERROR_LAYOUT_DATA.setCurrentUserAnAdmin(false);
        ERROR_LAYOUT_DATA.setCurrentUserFirstName("#Error!");
        ERROR_LAYOUT_DATA.setCurrentUserLastName("#Error!");
        ERROR_LAYOUT_DATA.setNavigationContext(new NavigationContext());
    }

    // Injectés par fr.inra.agrosyst.web.AgrosystWebInterceptor
    protected AgrosystWebConfig config;
    protected AgrosystWebSession session;
    protected AgrosystWebNotificationSupport notificationSupport;
    protected AgrosystWebApplicationContext applicationContext;
    protected NavigationContextService navigationContextService;
    protected BusinessAuthorizationService authorizationService;

    protected HttpServletRequest servletRequest;
    protected HttpServletResponse servletResponse;

    protected Map<String, String[]> parameters;

    protected RichNavigationContext richNavigationContext;

    protected boolean readOnly = false;

    protected Boolean currentUserAnAdmin;
    protected Boolean currentUserAnIsDataProcessor;

    /** Attachment management */
    protected AttachmentService attachmentService;

    protected Long attachmentCount;

    /**
     * gson (can be used into non json action to initialize json data).
     */
    protected Gson gson;

    public Gson getGson() {
        if (gson == null) {
            gson = new AgrosystGsonSupplier().get();
        }
        return gson;
    }

    public String toJson(Object element) {
        String result = null;
        try {
            result = getGson().toJson(element);
        } catch (Exception e) {
            log.warn(e);
        }
        return result;
    }

    @Override
    public void setServletRequest(HttpServletRequest servletRequest) {
        this.servletRequest = servletRequest;
    }

    @Override
    public void setServletResponse(HttpServletResponse servletResponse) {
        this.servletResponse = servletResponse;
    }

    @Override
    public void setParameters(Map<String, String[]> parameters) {
        this.parameters = parameters;
    }

    public final void setConfig(AgrosystWebConfig config) {
        this.config = config;
    }

    public final void setSession(AgrosystWebSession session) {
        this.session = session;
    }

    public final void setNavigationContextService(NavigationContextService navigationContextService) { // AThimel 11/10/13 Final to make sure it is not overridden
        this.navigationContextService = navigationContextService;
    }

    public final void setAuthorizationService(BusinessAuthorizationService authorizationService) { // AThimel 11/10/13 Final to make sure it is not overridden
        this.authorizationService = authorizationService;
    }

    public final void setNotificationSupport(AgrosystWebNotificationSupport notificationSupport) {
        this.notificationSupport = notificationSupport;
    }

    public void setApplicationContext(AgrosystWebApplicationContext applicationContext) {
        this.applicationContext = applicationContext;
    }

    // Ne pas supprimer, méthode utilisée depuis les JSP
    public AgrosystWebConfig getConfig() {
        return config;
    }

    // Ne pas supprimer, méthode utilisée depuis les JSP
    public AgrosystWebSession getSession() {
        return session;
    }

    public String getSessionId() {
        String result = null;

        HttpSession httpSession = servletRequest.getSession(false);
        if (httpSession != null) {
            String httpSessionId = httpSession.getId();
            result = AgrosystWebSessionListener.obfuscateSessionId(httpSessionId);
        }
        return result;
    }

    protected NavigationContext getNavigationContext() {
        NavigationContext navigationContext = session.getNavigationContext();

        // 'dirty' permet de savoir si les données du contexte doivent être validées
        boolean dirty = false;
        if (navigationContext == null) {

            // Lecture du cookie
            navigationContext = readNavigationContextCookie();
            dirty = true;

            // Pas de cookie trouvé, création du contexte de navigation par défaut
            if (navigationContext == null) {
                navigationContext = new NavigationContext();
                dirty = false;
            }

        }

        if (dirty) {
            navigationContext = verifyAndSaveNavigationContext(navigationContext);
        }
        return navigationContext;
    }

    protected Map<String, Boolean> toTrueMap(Iterable<String> elements) {
        if (elements == null) {
            return Maps.newHashMap();
        } else {
            Map<String, Boolean> result = Maps.toMap(elements, AbstractJsonAction.GET_TRUE);
            return result;
        }
    }

    protected NavigationContext verifyAndSaveNavigationContext(NavigationContext navigationContext) {
        NavigationContext newNavigationContext = navigationContextService.verify(navigationContext);
        session.setNavigationContext(newNavigationContext);
        writeNavigationContextCookie(newNavigationContext);
        return newNavigationContext;
    }

    protected NavigationContext navigationContextEntityCreated(TopiaEntity newEntity) {
        NavigationContext navigationContext = getNavigationContext();
        NavigationContext newNavigationContext = navigationContextService.verify(navigationContext, newEntity);
        session.setNavigationContext(newNavigationContext);
        writeNavigationContextCookie(newNavigationContext);
        return newNavigationContext;
    }

    protected NavigationContext readNavigationContextCookie() {
        NavigationContext result = null;
        if (servletRequest != null) {
            Cookie[] cookies = servletRequest.getCookies();
            if (cookies != null) {
                for (Cookie cookie : cookies) {
                    String cookieName = cookie.getName();
                    if (AgrosystWebConfig.NAVIGATION_CONTEXT_COOKIE_NAME.equals(cookieName)) {
                        String cookieValue = cookie.getValue();
                        result = getGson().fromJson(cookieValue, NavigationContext.class);
                        break;
                    }
                }
            }
        }
        return result;
    }

    protected void writeNavigationContextCookie(NavigationContext navigationContext) {
        String cookieValue = getGson().toJson(navigationContext);
        String cookieName = AgrosystWebConfig.NAVIGATION_CONTEXT_COOKIE_NAME;
        Cookie cookie = new Cookie(cookieName, cookieValue);
        cookie.setPath(servletRequest.getContextPath());
        servletResponse.addCookie(cookie);
    }

    public RichNavigationContext getRichNavigationContext() {
        if (richNavigationContext == null) {
            NavigationContext navigationContext = getNavigationContext();
            richNavigationContext = new RichNavigationContext(navigationContext, navigationContextService);
        }
        return richNavigationContext;
    }

    /**
     * Transform enumeration values into map with i18n value for each enum value.
     * 
     * i18n key is fqn.NAME
     *
     * @param values values to transform
     * @return map (enum value &gt; i18n text)
     */
    protected <T> Map<T, String> getEnumAsMap(T... values) {
        // TODO AThimel 09/01/14 Introduce cache
        Map<T, String> valuesMap = Maps.newLinkedHashMap();
        for (T value : values) {
            String i18n = value.getClass().getName() + "." + value.toString();
            String trans = getText(i18n);
            valuesMap.put(value, trans);
        }
        return valuesMap;
    }

    protected void initForInput() {

    }

    /**
     * Return session info notification.
     * 
     * Warning : this get method clear resulting collection
     *
     * @return session info notifications
     */
    public Set<AgrosystWebNotification> getInfoNotifications() {
        Set<AgrosystWebNotification> messages = Sets.newHashSet(session.getInfoNotifications());
        session.getInfoNotifications().clear();
        return messages;
    }

    public boolean isInfoNotificationsEmpty() {
        return session.getInfoNotifications().isEmpty();
    }

    /**
     * Return session warning notifications.
     * 
     * Warning : this get method clear resulting collection
     *
     * @return session warning notifications
     */
    public Set<AgrosystWebNotification> getWarningNotifications() {
        Set<AgrosystWebNotification> messages = Sets.newHashSet(session.getWarningNotifications());
        session.getWarningNotifications().clear();
        return messages;
    }

    public boolean isWarningNotificationsEmpty() {
        return session.getWarningNotifications().isEmpty();
    }

    /**
     * Return session error notifications.
     * 
     * Warning : this get method clear resulting collection
     *
     * @return session error notifications
     */
    public Set<AgrosystWebNotification> getErrorNotifications() {
        Set<AgrosystWebNotification> messages = Sets.newHashSet(session.getErrorNotifications());
        session.getErrorNotifications().clear();
        return messages;
    }

    public boolean isErrorNotificationsEmpty() {
        return session.getErrorNotifications().isEmpty();
    }

    public boolean isReadOnly() {
        return readOnly;
    }

    public boolean isCurrentUserAnAdmin() {
        if (currentUserAnAdmin == null) {
            if (Strings.isNullOrEmpty(getSession().getAuthenticationToken())) {
                currentUserAnAdmin = Boolean.FALSE;
            } else {
                currentUserAnAdmin = authorizationService.isAdmin();
            }
        }
        return currentUserAnAdmin;
    }

    public boolean isCurrentUserAnIsDataProcessor() {
        if (currentUserAnIsDataProcessor == null) {
            if (Strings.isNullOrEmpty(getSession().getAuthenticationToken())) {
                currentUserAnIsDataProcessor = Boolean.FALSE;
            } else {
                currentUserAnIsDataProcessor = authorizationService.isIsDataProcessor();
            }
        }
        return currentUserAnIsDataProcessor;
    }

    public void setAttachmentService(AttachmentService attachmentService) {
        this.attachmentService = attachmentService;
    }

    public long getAttachmentCount(String codeOrId) {
        if (attachmentCount == null) {
            attachmentCount = attachmentService.getAttachmentMetadatasCount(codeOrId);
        }
        return attachmentCount;
    }

    public AgrosystLayoutData initLayoutData() {

        AgrosystLayoutData result;
        String bannerName = null;
        try {

            if (this.layoutData == null) {
                Object layoutData = servletRequest.getAttribute("layoutData");
                AgrosystLayoutData instance;
                if (layoutData == null) {
                    instance = new AgrosystLayoutData();
                } else {
                    instance = (AgrosystLayoutData)layoutData;
                }
                boolean admin = false;
                boolean isDp = false;

                UserDto authenticatedUser = session.getAuthenticatedUser();
                if (authenticatedUser != null) {
                    instance.setCurrentUserLastName(authenticatedUser.getLastName());
                    instance.setCurrentUserFirstName(authenticatedUser.getFirstName());
                    admin = isCurrentUserAnAdmin();
                    isDp = isCurrentUserAnIsDataProcessor();

                    instance.setNavigationContext(getNavigationContext());
                    instance.setCampaigns(getRichNavigationContext().getCampaigns());
                    instance.setNetworks(getRichNavigationContext().getNetworks());
                    instance.setDomains(getRichNavigationContext().getDomains());
                    instance.setGrowingPlans(getRichNavigationContext().getGrowingPlans());
                    instance.setGrowingSystems(getRichNavigationContext().getGrowingSystems());
                    
                    bannerName = authenticatedUser.getBanner();

                }
                instance.setCurrentUserAnAdmin(admin);
                instance.setCurrentUserAnIsDataProcessor(isDp);
                // Assign value at the very end to be sure to exception is raised
                this.layoutData = instance;
            }
        } catch (Exception eee) {
            log.warn("Unable to compute layoutData: " + eee.getMessage());
        } finally {
            result = Objects.firstNonNull(this.layoutData, ERROR_LAYOUT_DATA);
            
            // banner map
            if (applicationContext != null) {
                Map<String, String[]> bannersMap = applicationContext.getBannersMap();
                if (!bannersMap.containsKey(bannerName)) {
                    bannerName = "Photo_F_Liagre";
                }
                String[] bannerData = bannersMap.get(bannerName);
                if (bannerData != null) {
                    result.setCurrentUserBannerPath(bannerData[0]);
                    result.setCurrentUserBannerMeta(bannerData[1]);
                }
            }

            if (servletRequest != null) {
                servletRequest.setAttribute("layoutData", result);
            }
        }

        return result;
    }

    protected AgrosystLayoutData layoutData;

    public AgrosystLayoutData getLayoutData() {
        if (layoutData == null) {
            this.layoutData = initLayoutData();
        }
        return this.layoutData;
    }

}
