/*
 * #%L
 * Agrosyst :: Web
 * $Id: PracticedSystemsEdit.java 2490 2013-12-16 15:27:35Z dcosse $
 * $HeadURL: https://forge.codelutin.com/svn/agrosyst/tags/agrosyst-0.8.1/agrosyst-web/src/main/java/fr/inra/agrosyst/web/actions/practiced/PracticedSystemsEdit.java $
 * %%
 * Copyright (C) 2013 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

package fr.inra.agrosyst.web.actions.practiced;

import java.lang.reflect.Type;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.struts2.convention.annotation.Action;
import org.apache.struts2.convention.annotation.Result;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.gson.reflect.TypeToken;
import com.opensymphony.xwork2.Preparable;

import fr.inra.agrosyst.api.NavigationContext;
import fr.inra.agrosyst.api.entities.AgrosystInterventionType;
import fr.inra.agrosyst.api.entities.CapacityUnit;
import fr.inra.agrosyst.api.entities.CropCyclePhaseType;
import fr.inra.agrosyst.api.entities.FertilizerQuantityUnit;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.OrchardFrutalForm;
import fr.inra.agrosyst.api.entities.PollinatorSpreadMode;
import fr.inra.agrosyst.api.entities.PracticedIntervention;
import fr.inra.agrosyst.api.entities.PracticedPerennialCropCycle;
import fr.inra.agrosyst.api.entities.PracticedPerennialCropCycleImpl;
import fr.inra.agrosyst.api.entities.PracticedSeasonalCropCycle;
import fr.inra.agrosyst.api.entities.PracticedSystem;
import fr.inra.agrosyst.api.entities.PracticedSystemImpl;
import fr.inra.agrosyst.api.entities.PracticedSystemSource;
import fr.inra.agrosyst.api.entities.Price;
import fr.inra.agrosyst.api.entities.SeedType;
import fr.inra.agrosyst.api.entities.UnitPlantSeeds;
import fr.inra.agrosyst.api.entities.VineFrutalForm;
import fr.inra.agrosyst.api.entities.WeedType;
import fr.inra.agrosyst.api.entities.YealdCategory;
import fr.inra.agrosyst.api.entities.YealdUnit;
import fr.inra.agrosyst.api.entities.referentiels.RefInterventionAgrosystTravailEDI;
import fr.inra.agrosyst.api.entities.referentiels.RefOrientationEDI;
import fr.inra.agrosyst.api.entities.referentiels.RefUnitesEDI;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.domain.CroppingPlanSpeciesDto;
import fr.inra.agrosyst.api.services.domain.CroppingPlans;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemFilter;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemService;
import fr.inra.agrosyst.api.services.action.ActionDto;
import fr.inra.agrosyst.api.services.practiced.CropCycleModelDto;
import fr.inra.agrosyst.api.services.practiced.PracticedCropCycleConnectionDto;
import fr.inra.agrosyst.api.services.practiced.PracticedCropCycleNodeDto;
import fr.inra.agrosyst.api.services.practiced.PracticedCropCyclePhaseDto;
import fr.inra.agrosyst.api.services.practiced.PracticedCropCycleSpeciesDto;
import fr.inra.agrosyst.api.services.practiced.PracticedInterventionDto;
import fr.inra.agrosyst.api.services.practiced.PracticedPerennialCropCycleDto;
import fr.inra.agrosyst.api.services.practiced.PracticedSeasonalCropCycleDto;
import fr.inra.agrosyst.api.services.practiced.PracticedSystemService;
import fr.inra.agrosyst.api.services.referentiels.ReferentielService;
import fr.inra.agrosyst.web.actions.AbstractAgrosystAction;

/**
 * Action d'édition d'un système synthétisé.
 * 
 * Avec:
 * <ul>
 * <li>l'onglet géneralité
 * <li>la liste des cycles pluriannuels assolés
 * <li>la liste des cycles pluriannuels pérennes
 * <li>l'itinéraire technique
 * </ul>
 * 
 * @author Eric Chatellier
 */
public class PracticedSystemsEdit extends AbstractAgrosystAction implements Preparable {

    /** serialVersionUID. */
    private static final long serialVersionUID = -5696853256365484417L;

    protected PracticedSystemService practicedSystemService;

    protected GrowingSystemService growingSystemService;
        
    protected ReferentielService referentielService;

    /** L'id du systeme synthétisé en cours d'edition. */
    protected String practicedSystemId;

    /** L'instance du systeme synthétisé en cours d'edition. */
    protected PracticedSystem practicedSystem;

    /** La liste de systeme de culture auquel peut être ratachés le SP (en creation). */
    protected List<GrowingSystem> growingSystems;

    /** Le systeme de culture sélectionné. */
    protected String growingSystemId;

    /** La liste des campaignes. */
    protected String campaigns;

    protected List<PracticedPerennialCropCycleDto> practicedPerennialCropCycleDtos;

    /** @deprecated pas de necessité d'être en globale */
    @Deprecated
    protected List<PracticedIntervention> interventions;

    protected List<RefOrientationEDI> refOrientationEDIs;

    protected List<RefInterventionAgrosystTravailEDI> refInterventionAgrosystTravailEDIs;

    protected Map<Integer, String> allActiveFertiMinProductType;

    /** Liste des cycles pluriannuels assolé. */
    protected List<PracticedSeasonalCropCycleDto> practicedSeasonalCropCycleDtos;

    /** Les combinaisons d'outils concernant le système de cultures pour les années sélectionnées. */
    //protected List<ToolsCouplingDto> practicedSystemToolsCouplings;

    /**
     * Contient une map &lt;culture, List&lt;Species&gt;&gt; pour toutes les cultures des systemes de cultures
     * concernés sur les campagnes du systeme synthétisé.
     */
    protected Map<String, List<CroppingPlanSpeciesDto>> practicedSystemCroppingEntryPlanToSpecies;

    /** Les cultures principales disponibles dans le graphique. */
    protected List<CropCycleModelDto> practicedSystemMainCropCycleModels;

    /** Les cultures intermédiaires disponible dans le graphique. */
    protected List<CropCycleModelDto> practicedSystemIntermediateCropCycleModels;

    protected List<RefUnitesEDI> refUnitesEDIs;

    protected List<Price> prices;

    public void setPracticedSystemService(PracticedSystemService practicedSystemService) {
        this.practicedSystemService = practicedSystemService;
    }

    public void setGrowingSystemService(GrowingSystemService growingSystemService) {
        this.growingSystemService = growingSystemService;
    }
    
    public void setReferentielService(ReferentielService referentielService) {
        this.referentielService = referentielService;
    }

    public PracticedSystem getPracticedSystem() {
        if (practicedSystem == null) {
            // EChatellier 27/06/2013 Fais chier de devoir écrire ça, mais c'est la seule option pour ne pas avoir une grosse dose d'exceptions avec du paramsPrepareParams
            return new PracticedSystemImpl();
        }
        return practicedSystem;
    }

    @Override
    public void prepare() {
        if (StringUtils.isNotBlank(practicedSystemId)) {
            practicedSystem = practicedSystemService.getPracticedSystem(practicedSystemId);
            
        } else {
            // j'offre la possibilité de crééer un PracticedCropCycle
            practicedSystem = new PracticedSystemImpl();
        }
    }

    @Override
    @Action("practiced-systems-edit-input")
    public String input() {
        initForInput();

        // default to empty lists
        practicedSeasonalCropCycleDtos = Lists.newArrayList();
        practicedPerennialCropCycleDtos = Lists.newArrayList();

        if (practicedSystem.isPersisted()) {
            authorizationService.checkPracticedSystemReadable(practicedSystemId);

            readOnly = !authorizationService.isPracticedSystemWritable(practicedSystemId);
            if (readOnly) {
                notificationSupport.practicedSystemNotWritable();
            }

            // FIXME AThimel 06/11/13 Utiliser le topiaId et non de l'instance
            List<PracticedPerennialCropCycle> practicedPerennialCropCycles =  practicedSystemService.getAllPracticedPerennialCropCycle(practicedSystem);
            practicedPerennialCyclesInput(practicedPerennialCropCycles);
            // FIXME AThimel 06/11/13 Utiliser le topiaId et non de l'instance
            List<PracticedSeasonalCropCycle> practicedSeasonalCropCycles = practicedSystemService.getAllPracticedSeasonalCropCycle(practicedSystem);
            practicedSeasonalCyclesInput(practicedSeasonalCropCycles);

        }

        prices = practicedSystemService.getPracticedPrices(practicedSystemId);

        return INPUT;
    }

    protected void practicedPerennialCyclesInput(List<PracticedPerennialCropCycle> practicedPerennialCropCycles) {
        for (PracticedPerennialCropCycle practicedPerennialCropCycle : practicedPerennialCropCycles) {
            PracticedPerennialCropCycleDto practicedPerennialCropCycleDto = new PracticedPerennialCropCycleDto();

            //TODO david 31/10/13 find a way to not carring all unnecessary attributs
            PracticedPerennialCropCycle lightPracticedPerennialCropCycle = new PracticedPerennialCropCycleImpl();
            Binder<PracticedPerennialCropCycle, PracticedPerennialCropCycle> binder = BinderFactory.newBinder(PracticedPerennialCropCycle.class);
            binder.copyExcluding(practicedPerennialCropCycle, lightPracticedPerennialCropCycle,
                    PracticedPerennialCropCycle.PROPERTY_CROP_CYCLE_PHASES,
                    PracticedPerennialCropCycle.PROPERTY_PRACTICED_CROP_CYCLE_SPECIES
                    );

            practicedPerennialCropCycleDto.setPracticedPerennialCropCycle(lightPracticedPerennialCropCycle);
            practicedPerennialCropCycleDto.setCroppingPlanEntryName(practicedSystemService.getCroppingPlanEntryName(practicedPerennialCropCycle.getCroppingPlanEntryCode()));

            List<PracticedCropCyclePhaseDto> cropCyclePhaseDtos = practicedSystemService.convertCropCyclePhases(practicedPerennialCropCycle);
            practicedPerennialCropCycleDto.setCropCyclePhaseDtos(cropCyclePhaseDtos);

            List<PracticedCropCycleSpeciesDto> cropCyclePerennialSpeciesDtos = practicedSystemService.getCropCyclePerennialSpecies(practicedPerennialCropCycle.getCroppingPlanEntryCode(),
                    practicedPerennialCropCycle, practicedSystem.getCampaigns());
            practicedPerennialCropCycleDto.setCropCyclePerennialSpeciesDto(cropCyclePerennialSpeciesDtos);

            practicedPerennialCropCycleDtos.add(practicedPerennialCropCycleDto);
        }
    }

    /**
     * Convert database seasonal entity to dto.
     */
    protected void practicedSeasonalCyclesInput(List<PracticedSeasonalCropCycle> practicedSeasonalCropCycles) {

        for (PracticedSeasonalCropCycle practicedSeasonalCropCycle : practicedSeasonalCropCycles) {
            PracticedSeasonalCropCycleDto dto = new PracticedSeasonalCropCycleDto();
            dto.setTopiaId(practicedSeasonalCropCycle.getTopiaId());

            // graph data for each cycle
            List<PracticedCropCycleNodeDto> nodes = Lists.newArrayList();
            Map<PracticedCropCycleNodeDto, List<CroppingPlanSpeciesDto>> nodesToSpecies = practicedSystemService.getNodes(practicedSeasonalCropCycle.getTopiaId());
            for (Map.Entry<PracticedCropCycleNodeDto, List<CroppingPlanSpeciesDto>> entry : nodesToSpecies.entrySet()) {
                PracticedCropCycleNodeDto cycleNodeDto = entry.getKey();
                nodes.add(cycleNodeDto);
            }
            dto.setCropCycleNodeDtos(nodes);
            List<PracticedCropCycleConnectionDto> connections = practicedSystemService.getConnections(practicedSeasonalCropCycle.getTopiaId());
            dto.setCropCycleConnectionDtos(connections);

            // valuation du label des connexions
            if (connections != null && practicedSystemIntermediateCropCycleModels != null) {
                for (PracticedCropCycleConnectionDto connection : connections) {
                    final String expectedCode = connection.getIntermediateCroppingPlanEntryCode();
                    if (!Strings.isNullOrEmpty(expectedCode)) {
                        Optional<CropCycleModelDto> optional = Iterables.tryFind(practicedSystemIntermediateCropCycleModels, new Predicate<CropCycleModelDto>() {
                            @Override
                            public boolean apply(CropCycleModelDto input) {
                                return expectedCode.equals(input.getCroppingPlanEntryCode());
                            }
                        });
                        if (optional.isPresent()) {
                            String label = optional.get().getLabel();
                            if (connection.getCroppingPlanEntryFrequency() != null) {
                                label += " (" + connection.getCroppingPlanEntryFrequency() + "%)";
                            }
                            connection.setLabel(label);
                        }
                    }
                }
            }

            // add dto
            practicedSeasonalCropCycleDtos.add(dto);
        }
    }

    @Override
    protected void initForInput() {
        if (getPracticedSystem().isPersisted()) {
            // warning, practicedSystem's growingSystem must always be part of growingSystems set
            // even not selected be navigation context
            GrowingSystem growingSystem = getPracticedSystem().getGrowingSystem();
            growingSystemId = growingSystem.getTopiaId();
            growingSystems = Collections.singletonList(growingSystem);
            campaigns = practicedSystem.getCampaigns();
        } else {
            GrowingSystemFilter growingSystemFilter = new GrowingSystemFilter();
            NavigationContext navigationContext = getNavigationContext();
            growingSystemFilter.setNavigationContext(navigationContext);
            growingSystemFilter.setActive(Boolean.TRUE);
            growingSystemFilter.setPageSize(GrowingSystemFilter.ALL_PAGE_SIZE);
            ResultList<GrowingSystem> resultList = growingSystemService.getFilteredGrowingSystems(growingSystemFilter);
            growingSystems = resultList.getElements();
        }
        
        // chargement des référentiels
        refOrientationEDIs = referentielService.findAllReferentielEDI();
        refInterventionAgrosystTravailEDIs = referentielService.getRefInterventionAgrosystTravailEDIs();

        allActiveFertiMinProductType = referentielService.findAllActiveFertiMinProductType();

        refUnitesEDIs = referentielService.findAllActiveRefUnitesEDI();

        // initialisation des données chargés en json autrement sur la page
        if (!Strings.isNullOrEmpty(growingSystemId) && !Strings.isNullOrEmpty(campaigns)) {
            // chargement des toutes les cultures pour les systems de cultures et les années
            Map<CropCycleModelDto, List<CroppingPlanSpeciesDto>> modelToSpecies =
                    practicedSystemService.getCropCycleModelMap(growingSystemId, campaigns, true);
            Set<CropCycleModelDto> croppingPlanEntryDtos = modelToSpecies.keySet();

            // definition de la liste de culture principale
            Iterable<CropCycleModelDto> modelMain = Iterables.filter(croppingPlanEntryDtos, CroppingPlans.IS_NOT_INTERMEDIATE);
            practicedSystemMainCropCycleModels = Lists.newArrayList(modelMain); // force no lazy

            // définition de la liste de culture intermédiaire
            Iterable<CropCycleModelDto> modelIntermediate = Iterables.filter(croppingPlanEntryDtos, CroppingPlans.IS_INTERMEDIATE);
            practicedSystemIntermediateCropCycleModels = Lists.newArrayList(modelIntermediate); // force no lazy

            // chargement de la map 'code culture' > liste d'espece
            practicedSystemCroppingEntryPlanToSpecies = Maps.newHashMap();
            for (Map.Entry<CropCycleModelDto, List<CroppingPlanSpeciesDto>> entry : modelToSpecies.entrySet()) {
                practicedSystemCroppingEntryPlanToSpecies.put(entry.getKey().getCroppingPlanEntryCode(), entry.getValue());
            }

            // chargement des toolsCouplings
//            List<ToolsCouplingDto> toolsCouplingsDtos = practicedSystemService.getToolsCouplingModel(growingSystemId, campaigns);
//            practicedSystemToolsCouplings = Lists.newArrayList(toolsCouplingsDtos); // force no lazy
        }
    }

    @Override
    public void validate() {
        if (StringUtils.isBlank(getPracticedSystem().getTopiaId())) {
            if (StringUtils.isBlank(growingSystemId)) {
                addFieldError("growingSystemId", "Le système de culture est requis !");
            }
        }

        if (StringUtils.isBlank(practicedSystem.getName())) {
            addFieldError("practicedSystem.name", "Le nom du système synthétisé est requis !");
        }
        String campaigns = practicedSystem.getCampaigns();
        if (StringUtils.isBlank(campaigns)) {
            addFieldError("practicedSystem.campaigns", "La série de campagne est obligatoire");
        }
        if (StringUtils.isNotBlank(campaigns)) {
            if (!practicedSystemService.areCampaignsValid(campaigns)) {
                String format = "La série de campagne doit être composée d'années séparées par des virgules (,), " +
                        "espaces ( ) ou point-virgules (;). Les campagnes doivent être contenues entre %d et %d.";
                Pair<Integer,Integer> bounds = practicedSystemService.getCampaignsBounds();
                addFieldError("practicedSystem.campaigns", String.format(format, bounds.getLeft(), bounds.getRight()));
            }
        }

        if (practicedSeasonalCropCycleDtos != null) {
            practicedSeasonalCropCycleValidate();
        }
        if (practicedPerennialCropCycleDtos != null){
            practicedPerennialCycleValidate();
        }
        
        if (hasErrors()) {
            initForInput();
        }
    }

    protected void practicedSeasonalCropCycleValidate() {

        for (PracticedSeasonalCropCycleDto practicedSeasonalCropCycleDto : practicedSeasonalCropCycleDtos) {
            List<PracticedCropCycleConnectionDto> cropCycleConnectionDtos = practicedSeasonalCropCycleDto.getCropCycleConnectionDtos();
            if (cropCycleConnectionDtos != null) {
                for (PracticedCropCycleConnectionDto cropCycleConnectionDto: cropCycleConnectionDtos) {
                    List<PracticedInterventionDto> interventionDtos = cropCycleConnectionDto.getInterventions();
                    validatePracticedIntervention(interventionDtos);
                }
            }
        }
    }

    protected void practicedPerennialCycleValidate() {
        // validate perennial cycle
        for (PracticedPerennialCropCycleDto practicedPerennialCropCycleDto : practicedPerennialCropCycleDtos){
            if (Strings.isNullOrEmpty(practicedPerennialCropCycleDto.getPracticedPerennialCropCycle().getCroppingPlanEntryCode())) {
                addFieldError("croppingPlanEntry", "Champ obligatoire");
            }

            List<PracticedCropCyclePhaseDto> cropCyclePhaseDtos = practicedPerennialCropCycleDto.getCropCyclePhaseDtos();
            if (cropCyclePhaseDtos == null) {
                addActionError("Un cycle pluriannuel doit avoir une phase de pleine production");
            } else {
                int instCount = 0, montCount = 0, pleiCount = 0, declCount = 0;
                for (PracticedCropCyclePhaseDto cropCyclePhaseDto : cropCyclePhaseDtos) {
                    if (cropCyclePhaseDto.getCropCyclePhaseType() != null) {
                        switch (cropCyclePhaseDto.getCropCyclePhaseType()) {
                            case INSTALLATION:
                                instCount++;
                                break;
                            case MONTEE_EN_PRODUCTION:
                                montCount++;
                                break;
                            case PLEINE_PRODUCTION:
                                pleiCount++;
                                break;
                            case DECLIN_DE_PRODUCTION:
                                declCount++;
                                break;
                            default:
                                break;
                        }

                    }
                    List<PracticedInterventionDto> interventionDtos = cropCyclePhaseDto.getInterventions();
                    validatePracticedIntervention(interventionDtos);
                }
                if (instCount > 1) {
                    addActionError("Un cycle pluriannuel ne peut avoir qu'une seul phase d'installation");
                }
                if (montCount > 1) {
                    addActionError("Un cycle pluriannuel ne peut avoir qu'une seul phase de montée en production");
                }
                if (pleiCount == 0) {
                    addActionError("Un cycle pluriannuel doit avoir une phase de pleine production");
                }
                else if (pleiCount > 1) {
                    addActionError("Un cycle pluriannuel ne peut avoir qu'une seul phase de pleine production");
                }
                if (declCount > 1) {
                    addActionError("Un cycle pluriannuel ne peut avoir qu'une seul phase de déclin de production");
                }
            }
        }
    }

    protected void validatePracticedIntervention(List<PracticedInterventionDto> interventionDtos) {
        if (interventionDtos != null) {
            for (PracticedInterventionDto interventionDto : interventionDtos) {
                if (Strings.isNullOrEmpty(interventionDto.getIntervention().getName())) {
                    addActionError("Une intervention doit-être nommée");
                }
                if (interventionDto.getIntervention().getAgrosystInterventionType() == null) {
                    addActionError("Le type d'intervention est obligatoire");
                }
                List<ActionDto> actionDtos = interventionDto.getActionDtos();
                if (actionDtos == null || actionDtos.size() == 0) {
                    addActionError("Une intervention a au moins une action");
                } else {
                    for (ActionDto actionDto : actionDtos) {
                        if (actionDto.getMainAction() == null) {
                            addActionError("le type d'action est obligatoire sur une action");
                        }
                    }
                }
                if (Strings.isNullOrEmpty(interventionDto.getIntervention().getStartingPeriodDate())){
                    addActionError("Une date d'intervention ou date de début est obligatoire");
                }
                if (Strings.isNullOrEmpty(interventionDto.getIntervention().getEndingPeriodDate())) {
                    addActionError("Une date de fin d'intervention est obligatoire");
                }
            }
        }
    }

    @Action(results = {@Result(type = "redirectAction", params = {"actionName", "practiced-systems-edit-input", "practicedSystemTopiaId", "${practicedSystem.topiaId}"})})
    @Override
    public String execute() throws Exception {
        PracticedSystem practicedSystem = getPracticedSystem();
        // can define domain only during create action
        if (StringUtils.isBlank(practicedSystem.getTopiaId())) {
            GrowingSystem growingSystem = growingSystemService.getGrowingSystem(growingSystemId);
            getPracticedSystem().setGrowingSystem(growingSystem);
        }

        removeEmptySeasonalCropCycle();

        practicedSystem = practicedSystemService.createOrUpdatePracticedSystem(
                practicedSystem,
                practicedPerennialCropCycleDtos,
                practicedSeasonalCropCycleDtos,
                prices);

        notificationSupport.practicedSystemSaved(practicedSystem);

        return SUCCESS;
    }

    /**
     * The last seasonal crop cycle can be empty has one is always created if it's case then it is removed.
     */
    protected void removeEmptySeasonalCropCycle() {
        if (practicedSeasonalCropCycleDtos != null && !practicedSeasonalCropCycleDtos.isEmpty()){
            List<PracticedSeasonalCropCycleDto> pseccCopy = Lists.newArrayList(practicedSeasonalCropCycleDtos);
            for (PracticedSeasonalCropCycleDto practicedSeasonalCropCycleDto : pseccCopy) {
                if (practicedSeasonalCropCycleDto.getCropCycleNodeDtos() == null || practicedSeasonalCropCycleDto.getCropCycleNodeDtos().isEmpty()){
                    practicedSeasonalCropCycleDtos.remove(practicedSeasonalCropCycleDto);
                }
            }
        }
    }

    public List<GrowingSystem> getGrowingSystems() {
        return growingSystems;
    }
    
    public String getGrowingSystemsJson() {
        return getGson().toJson(growingSystems);
    }

    public void setPracticedSystemTopiaId(String practicedSystemTopiaId) {
        this.practicedSystemId = practicedSystemTopiaId;
    }

    public String getGrowingSystemId() {
        return growingSystemId;
    }

    public void setGrowingSystemId(String growingSystemId) {
        this.growingSystemId = growingSystemId;
    }

    public Map<PracticedSystemSource, String> getSources() {
        return getEnumAsMap(PracticedSystemSource.values());
    }
    
    public List<RefOrientationEDI> getRefOrientationEDIs() {
        return refOrientationEDIs;
    }

    public void setPracticedSystemId(String practicedSystemId) {
        this.practicedSystemId = practicedSystemId;
    }

    public String getPracticedSystemId() {
        return practicedSystemId;
    }

    public Map<VineFrutalForm, String> getVineFrutalForms() {
        return getEnumAsMap(VineFrutalForm.values());
    }

    public Map<OrchardFrutalForm, String> getOrchardFrutalForms() {
        return getEnumAsMap(OrchardFrutalForm.values());
    }

    public Map<PollinatorSpreadMode, String> getPollinatorSpreadModes() {
        return getEnumAsMap(PollinatorSpreadMode.values());
    }

    public Map<CropCyclePhaseType, String> getCropCyclePhaseTypes() {
        return getEnumAsMap(CropCyclePhaseType.values());
    }
    
    public Map<WeedType, String> getWeedTypes() {
        return getEnumAsMap(WeedType.values());
    }

    public Map<FertilizerQuantityUnit, String> getFertilizerQuantityUnit() {
        return getEnumAsMap(FertilizerQuantityUnit.values());
    }

    public String getCampaigns() {
        String result = "";
        if (practicedSystem != null) {
            result = practicedSystem.getCampaigns();
        }
        return getGson().toJson(result);
    }
    
    public String getGrowingSystemIdJson(){
        return getGson().toJson(growingSystemId);
    }
    
    public Map<AgrosystInterventionType, String> getAgrosystInterventionTypes(){
        return getEnumAsMap(AgrosystInterventionType.values());
    }

    public Map<SeedType, String> getSeedTypes() {
        return getEnumAsMap(SeedType.values());
    }

    public Map<UnitPlantSeeds, String> getUnitPlantSeeds() {
        return getEnumAsMap(UnitPlantSeeds.values());
    }

    public Map<YealdUnit, String>  getYealdUnit() {
        return getEnumAsMap(YealdUnit.values());
    }

    public void setPracticedPerennialCropCycleDtosJson(String json) {
        Type type = new TypeToken<List<PracticedPerennialCropCycleDto>>() {}.getType();
        this.practicedPerennialCropCycleDtos = getGson().fromJson(json, type);
    }

    public void setPracticedSeasonalCropCycleDtosJson(String json) {
        Type type = new TypeToken<List<PracticedSeasonalCropCycleDto>>() {}.getType();
        this.practicedSeasonalCropCycleDtos = getGson().fromJson(json, type);
    }

    public List<PracticedPerennialCropCycleDto> getPracticedPerennialCropCycleDtos() {
        return practicedPerennialCropCycleDtos;
    }

    public List<RefInterventionAgrosystTravailEDI> getRefInterventionAgrosystTravailEDIs(){
        return refInterventionAgrosystTravailEDIs;
    }

    public List<PracticedSeasonalCropCycleDto> getPracticedSeasonalCropCycleDtos() {
        return practicedSeasonalCropCycleDtos;
    }

//    public List<ToolsCouplingDto> getPracticedSystemToolsCouplings() {
//        return practicedSystemToolsCouplings;
//    }

    public Map<String, List<CroppingPlanSpeciesDto>> getPracticedSystemCroppingEntryPlanToSpecies() {
        return practicedSystemCroppingEntryPlanToSpecies;
    }

    public List<CropCycleModelDto> getPracticedSystemMainCropCycleModels() {
        return practicedSystemMainCropCycleModels;
    }

    public List<CropCycleModelDto> getPracticedSystemIntermediateCropCycleModels() {
        return practicedSystemIntermediateCropCycleModels;
    }

    public Map<YealdCategory, String> getYealdCategorys(){
        return getEnumAsMap(YealdCategory.values());
    }

    public Map<CapacityUnit, String> getCapacityUnits(){
        return getEnumAsMap(CapacityUnit.values());
    }

    public Map<Integer, String> getAllActiveFertiMinProductType() {
        return allActiveFertiMinProductType;
    }

    public List<RefUnitesEDI> getRefUnitesEDIs() {
        return refUnitesEDIs;
    }

    public List<Price> getPrices() {
        return prices;
    }

    public void setPricesJson(String json) {
        Type type = new TypeToken<List<Price>>() {}.getType();
        this.prices = getGson().fromJson(json, type);
    }

}
