package fr.inra.agrosyst.web.actions.networks;

/*
 * #%L
 * Agrosyst :: Web
 * $Id: NetworksEdit.java 1947 2013-10-29 16:19:38Z athimel $
 * $HeadURL: https://forge.codelutin.com/svn/agrosyst/tags/agrosyst-0.8.1/agrosyst-web/src/main/java/fr/inra/agrosyst/web/actions/networks/NetworksEdit.java $
 * %%
 * Copyright (C) 2013 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.lang.reflect.Type;
import java.util.List;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.apache.struts2.convention.annotation.Action;
import org.apache.struts2.convention.annotation.Result;

import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.gson.reflect.TypeToken;
import com.opensymphony.xwork2.Preparable;

import fr.inra.agrosyst.api.entities.Network;
import fr.inra.agrosyst.api.entities.NetworkImpl;
import fr.inra.agrosyst.api.entities.NetworkManager;
import fr.inra.agrosyst.api.services.network.NetworkManagerDto;
import fr.inra.agrosyst.api.services.network.NetworkService;
import fr.inra.agrosyst.api.services.users.UserDto;
import fr.inra.agrosyst.api.services.users.UserService;
import fr.inra.agrosyst.web.actions.AbstractAgrosystAction;

/**
 * @author cosse
 */
public class NetworksEdit extends AbstractAgrosystAction implements Preparable {

    private static final long serialVersionUID = 6310375684340888705L;

    protected NetworkService networkService;

    protected UserService userService;

    public void setNetworkService(NetworkService networkService) {
        this.networkService = networkService;
    }

    public void setUserService(UserService userService) {
        this.userService = userService;
    }

    protected String networkTopiaId;

    protected Network network;

    protected List<String> parentIds;

    protected List<NetworkManagerDto> managers;

    @Override
    public void prepare() throws Exception {
        if (StringUtils.isBlank(networkTopiaId)) {// new network
            network = networkService.newNetwork();
        } else {
            network = networkService.getNetwork(networkTopiaId);
        }
    }

    @Override
    @Action("networks-edit-input")
    public String input() {
        if (!StringUtils.isBlank(networkTopiaId)) {

            readOnly = !authorizationService.isNetworkWritable(networkTopiaId);
            if (readOnly) {
                notificationSupport.networkNotWritable();
            }
        }

        managers = Lists.newArrayList();
        for (NetworkManager networkManager : network.getManagers()) {
            String userTopiaId = networkManager.getAgrosystUser().getTopiaId();
            UserDto userDto = userService.getUser(userTopiaId);
            NetworkManagerDto result = new NetworkManagerDto(
                    networkManager.getTopiaId(),
                    networkManager.getFromDate(),
                    networkManager.getToDate(),
                    networkManager.isActive(),
                    userDto);
            managers.add(result);
        }

        return INPUT;
    }

    @Override
    public void validate() {
        if (StringUtils.isBlank(network.getName())) {
            addFieldError("name", "Le nom du réseau est obligatoire.");

        } else {
            Set<String> homonymeNetworks = networkService.findNetworksByName(network.getName(), networkTopiaId);
            if (homonymeNetworks != null && !homonymeNetworks.isEmpty()) {
                if (!homonymeNetworks.contains(networkTopiaId)) {
                    addFieldError("name", "Le nom du réseau est déjà utilisé.");
                }
            }
        }
        if (managers.isEmpty()) {
            addFieldError("user", "Un réseau doit avoir au moins un responsable.");
        }
        if (researchParentsCycleProcess()) {
            addFieldError("parent", "Cycle détecté, un réseau ne peut avoir lui-même comme parent ou un réseau parent ayant ce réseau comme parent.");
        }

        if (hasErrors()) {
            initForInput();
        }
    }


    protected Boolean researchParentsCycleProcess() {
        Boolean result = false;
        for (String parentId : parentIds) {
            Network parent = networkService.getNetwork(parentId);
            researchCycle(parent);
        }
        return result;
    }

    protected Boolean researchCycle(Network parent) {
        Boolean result = false;

        if (networkTopiaId.equals(parent.getTopiaId())) {
            result = true;
        } else {
            for (Network grandParent : parent.getParents()) {
                result = researchCycle(grandParent);
            }
        }
        return result;
    }

    @Override
    @Action(results = {@Result(type = "redirectAction", params = {"actionName", "networks-edit-input", "networkTopiaId", "${network.topiaId}"})})
    public String execute() throws Exception {

        network = networkService.createOrUpdateNetwork(network, managers, parentIds);
        notificationSupport.networkSaved(network);

        if (Strings.isNullOrEmpty(networkTopiaId)) {
            navigationContextEntityCreated(network);
        }

        return SUCCESS;
    }

    public String getNetworkTopiaId() {
        return networkTopiaId;
    }

    public void setNetworkTopiaId(String networkTopiaId) {
        this.networkTopiaId = networkTopiaId;
    }

    public List<String> getParentIds() {
        return parentIds;
    }

    public void setParentIdsJson(String json) {
        Type type = new TypeToken<List<String>>() {
        }.getType();
        this.parentIds = getGson().fromJson(json, type);
    }

    public Network getNetwork() {
        if (network == null) {
            // AThimel 26/06/2013 Fais chier de devoir écrire ça, mais c'est la seule option pour ne pas avoir une grosse dose d'exceptions avec du paramsPrepareParams
            return new NetworkImpl();
        }
        return network;
    }

    public void setNetwork(Network network) {
        this.network = network;
    }

    public List<NetworkManagerDto> getManagers() {
        return managers;
    }

    public void setNetworkManagerDtoJson(String json) {
        Type type = new TypeToken<List<NetworkManagerDto>>() {
        }.getType();
        this.managers = getGson().fromJson(json, type);
    }

}
