package fr.inra.agrosyst.web.actions.domains;

/*
 * #%L
 * Agrosyst :: Web
 * $Id: DomainsEdit.java 2472 2013-12-12 22:40:05Z athimel $
 * $HeadURL: https://forge.codelutin.com/svn/agrosyst/tags/agrosyst-0.8.1/agrosyst-web/src/main/java/fr/inra/agrosyst/web/actions/domains/DomainsEdit.java $
 * %%
 * Copyright (C) 2013 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import com.google.common.base.Predicate;
import fr.inra.agrosyst.api.entities.AgrosystInterventionType;
import fr.inra.agrosyst.api.entities.referentiels.RefInterventionAgrosystTravailEDI;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.ReflectionToStringBuilder;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts2.convention.annotation.Action;
import org.apache.struts2.convention.annotation.Result;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.gson.reflect.TypeToken;
import com.opensymphony.xwork2.Preparable;

import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.DomainImpl;
import fr.inra.agrosyst.api.entities.DomainType;
import fr.inra.agrosyst.api.entities.Entities;
import fr.inra.agrosyst.api.entities.GPSData;
import fr.inra.agrosyst.api.entities.Materiel;
import fr.inra.agrosyst.api.entities.MaterielType;
import fr.inra.agrosyst.api.entities.Plot;
import fr.inra.agrosyst.api.entities.Price;
import fr.inra.agrosyst.api.entities.Sol;
import fr.inra.agrosyst.api.entities.ToolsCoupling;
import fr.inra.agrosyst.api.entities.WeatherStation;
import fr.inra.agrosyst.api.entities.Zoning;
import fr.inra.agrosyst.api.entities.referentiels.RefLegalStatus;
import fr.inra.agrosyst.api.entities.referentiels.RefLocation;
import fr.inra.agrosyst.api.entities.referentiels.RefMateriel;
import fr.inra.agrosyst.api.entities.referentiels.RefSolArvalis;
import fr.inra.agrosyst.api.entities.referentiels.RefStationMeteo;
import fr.inra.agrosyst.api.services.domain.CroppingPlanEntryDto;
import fr.inra.agrosyst.api.services.domain.DomainService;
import fr.inra.agrosyst.api.services.domain.ToolsCouplingDto;
import fr.inra.agrosyst.api.services.plot.PlotService;
import fr.inra.agrosyst.api.services.referentiels.ReferentielService;
import fr.inra.agrosyst.web.actions.AbstractAgrosystAction;
import fr.inra.agrosyst.web.actions.commons.AgrosystValidationState;

/**
 * Action d'édition d'un domaine.
 *
 * @author Eric Chatellier
 */
public class DomainsEdit extends AbstractAgrosystAction implements Preparable {

    private static final Log log = LogFactory.getLog(DomainsEdit.class);

    private static final String REQUIRED_FIELD = "Champ obligatoire";

    private static final long serialVersionUID = 390873886885798716L;

    /** Next action edit existing plot. */
    protected static final String ACTION_EDIT_PLOT = "actionEditPlot";

    /** Next action duplicate existing plot. */
    protected static final String ACTION_DUPLICATE_PLOT = "actionDuplicatePlot";

    /** Next action add empty plot. */
    protected static final String ACTION_EMPTY_PLOT = "actionEmptyPlot";

    protected static final Predicate<RefInterventionAgrosystTravailEDI> IS_NOT_PERSITED = new Predicate<RefInterventionAgrosystTravailEDI>() {

        @Override
        public boolean apply(RefInterventionAgrosystTravailEDI activityPeriod) {
            return activityPeriod.getTopiaId() == null;
        }
    };

    protected static final Function<Sol, SolDto> SOL_TO_DTO = new Function<Sol, SolDto>() {
        @Override
        public SolDto apply(Sol input) {
            RefSolArvalis refSolArvalis = input.getRefSolArvalis();
            SolDto result = new SolDto(
                    input.getTopiaId(),
                    input.getName(),
                    input.getComment(),
                    input.getImportance()
            );
            result.setSolArvalisTopiaId(refSolArvalis.getTopiaId());
            result.setSolNom(refSolArvalis.getSol_nom());
            result.setSolTexture(refSolArvalis.getSol_texture());
            result.setSolCalcaire(refSolArvalis.getSol_calcaire());
            result.setSolProfondeur(refSolArvalis.getSol_profondeur());
            result.setSolHydromorphie(refSolArvalis.getSol_hydromorphie());
            result.setSolPierrosite(refSolArvalis.getSol_pierrosite());
            result.setSolRegion(refSolArvalis.getSol_region());
            result.setSolRegionCode(refSolArvalis.getSol_region_code());
            return result;
        }
    };

    protected ReferentielService referentielService;

    protected DomainService domainService;

    protected PlotService plotService;

    protected Domain domain;

    protected LinkedHashMap<Integer, String> relatedDomains; // TODO AThimel 18/10/13 Using implementation to make

    protected List<GPSDataDto> gpsDataDtos;

    protected String domainTopiaId;

    protected String commune;

    protected String legalStatusId;

    protected List<RefLegalStatus> allRefLegalStatus;

    protected RefLocation location;

    protected String departement;
    protected String petiteRegionAgricole;
    protected String petiteRegionAgricoleName;

    protected String locationTopiaId;

    protected List<MaterielDto> domainMaterielDtos;

    protected List<ToolsCouplingDto> toolsCouplingDtos;

    protected Map<MaterielType, List<String>> materielType1List;

    protected Map<Integer, String> solArvalisRegions;

    protected List<SolDto> solDtos;

    protected List<CroppingPlanEntryDto> croppingPlan;

    protected Map<String, Materiel> materiels;

    protected AgrosystInterventionType agrosystInterventionType;

    protected Integer otex18;

    protected Integer otex70;

    protected Map<Integer, String> otex18s;

    protected List<Plot> plots;

    protected List<WeatherStationDto> weatherStationDtos;
    protected List<RefStationMeteo> allRefStationMeteo;

    protected List<Price> prices;

    protected Collection<RefInterventionAgrosystTravailEDI> actionsEDIs;

    protected AgrosystValidationState validationState;

    public void setReferentielService(ReferentielService referentielService) {
        this.referentielService = referentielService;
    }

    public void setDomainService(DomainService domainService) {
        this.domainService = domainService;
    }

    public void setPlotService(PlotService plotService) {
        this.plotService = plotService;
    }

    @Override
    public void prepare() throws Exception {

        if (StringUtils.isEmpty(domainTopiaId)) {
            // Cas de création d'un domain
            domain = domainService.newDomain();

            domain.setCampaign(getNavigationContext().getCampaigns().iterator().next());
        } else {
            // TODO AThimel 07/10/13 May be included directly in the service ?
            authorizationService.checkDomainReadable(domainTopiaId);

            // Cas d'une mise à jour de domain
            readOnly = !authorizationService.isDomainWritable(domainTopiaId);
            if (readOnly) {
                notificationSupport.domainNotWritable();
            }

            domain = domainService.getDomain(domainTopiaId);

            if (StringUtils.isBlank(locationTopiaId)) {
                // Cas ou la commune n'a pas étée modifiée
                locationTopiaId = domain.getLocation().getTopiaId();

                // Dans le cas ou il y a eu modification du choix de la commune
                location = referentielService.getRefLocation(locationTopiaId);

                departement = location.getDepartement();

                petiteRegionAgricole = location.getPetiteRegionAgricoleCode();
                petiteRegionAgricoleName = location.getPetiteRegionAgricoleNom();
            }

        }

        prices = domainService.getDomainPrices(domainTopiaId);

    }

    public Domain getDomain() {
        if (domain == null) {
            // AThimel 26/06/2013 Fais chier de devoir écrire ça, mais c'est la seule option pour ne pas avoir une grosse dose d'exceptions avec du paramsPrepareParams
            return new DomainImpl();
        }
        return domain;
    }

    @Override
    protected void initForInput() {

        allRefLegalStatus = domainService.getAllRefLegalStatus();
        allRefStationMeteo = domainService.getAllRefStationMeteo();

        materielType1List = referentielService.getTypeMateriel1List();

        solArvalisRegions = referentielService.getSolArvalisRegions();

        otex18s = referentielService.findAllActiveOtex18Code();

        actionsEDIs = referentielService.getRefInterventionAgrosystTravailEDIs();

        if (StringUtils.isNotEmpty(domainTopiaId)) {
            relatedDomains = domainService.getRelatedDomains(domain.getCode());
            plots = plotService.findAllByDomain(domain);
        }

        validationState = computeValidationState();
    }

    protected AgrosystValidationState computeValidationState() {
        Preconditions.checkNotNull(domain);
        AgrosystValidationState result = new AgrosystValidationState();
        boolean hasPermission = false;  // An object must be persisted first before been displayed
        if (domain.isPersisted()) {
            hasPermission = authorizationService.isDomainValidable(domainTopiaId);
            Date validationDate = domain.getValidationDate();
            result.setLastValidationDate(validationDate);
            boolean dirty = !domain.isValidated() || validationDate.before(domain.getUpdateDate());
            result.setDirty(dirty);
        }
        result.setUserHasValidationPermission(hasPermission);
        return result;
    }

    @Override
    @Action("domains-edit-input")
    public String input() throws Exception {
        initForInput();

        if (StringUtils.isNotEmpty(domainTopiaId)) {
            gpsDataDtos = Lists.transform(domain.getGpsDatas(), GpsDatas.GPS_DATA_TO_DTO);
            domainMaterielDtos = Lists.transform(domain.getMateriels(), Materiels.MATERIEL_TO_DTO);
            solDtos = Lists.transform(domain.getSols(), SOL_TO_DTO);
            toolsCouplingDtos = Lists.transform(domain.getToolsCoupling(), Materiels.TOOLS_COUPLING_TO_DTO);
            croppingPlan = domainService.getCroppingPlan(domainTopiaId);

            List<WeatherStation> weatherStations = domain.getWeatherStations();
            String defaultWeatherStationId = null;

            if (domain.getDefaultWeatherStation() != null) {
                defaultWeatherStationId = domain.getDefaultWeatherStation().getTopiaId();
            }

            weatherStationDtos = Lists.transform(weatherStations, WeatherStations.getFunctionWeatherStationToDto(defaultWeatherStationId));
        } else {
            gpsDataDtos = Collections.emptyList();
            domainMaterielDtos = Collections.emptyList();
            solDtos = Collections.emptyList();
            croppingPlan = Collections.emptyList();
        }

        return INPUT;
    }

    @Override
    public void validate() {
        // campagne
        if (domain.getCampaign() == 0) {
            addFieldError("domain.campaign", REQUIRED_FIELD);
        }

        // nom du domain
        if (StringUtils.isBlank(domain.getName())) {
            addFieldError("domain.name", REQUIRED_FIELD);
        }

        // nom de l'interlocuteur principal
        if (StringUtils.isBlank(domain.getMainContact())) {
            addFieldError("domain.mainContact", REQUIRED_FIELD);
        }

        // commune
        if (StringUtils.isBlank(locationTopiaId)) {
            addFieldError("commune", REQUIRED_FIELD);
        }

        // validation des materiels
        if (domainMaterielDtos != null) {
            for (MaterielDto domainMaterielDto : domainMaterielDtos) {
                if (StringUtils.isBlank(domainMaterielDto.getMaterielTopiaId())) {
                    addActionError("Le type de matériel associé au matériel est obligatoire !");
                }
                if (StringUtils.isBlank(domainMaterielDto.getName())) {
                    addActionError("Le nom du matériel est obligatoire !");
                }
            }
        }

        if (toolsCouplingDtos != null) {
            for (ToolsCouplingDto toolsCouplingDto: toolsCouplingDtos) {
                if (toolsCouplingDto.getMainsActions() == null || toolsCouplingDto.getMainsActions().isEmpty()) {
                    addFieldError("editedToolsCoupling-mainsActions", "Au moins une action principale doir être définie sur une combinaison d'outils");
                    addActionError("Au moins une action principale doir être définie sur une combinaison d'outils");
                }
                if (StringUtils.isBlank(toolsCouplingDto.getToolsCouplingName())){
                    addActionError("Une combinaison d'outils doit avoir un nom");
                }
            }
        }

        // validation des sols
        // FIXME echatellier 20130724 TODO

        if (croppingPlan != null) {
            for (CroppingPlanEntryDto croppingPlanEntryDto : croppingPlan) {
                if (StringUtils.isBlank(croppingPlanEntryDto.getName())) {
                    addActionError("Le nom d'une culture est obligatoire !");
                }
            }
        }

        if (hasErrors()) {
            log.debug("validate, action errors : " + getActionErrors().toString());
            log.debug("validate, fields errors : " + getFieldErrors().toString());

            initForInput();
        }
    }

    @Override
    @Action(results = {
            @Result(
                name = SUCCESS, type = "redirectAction",
                params = {"actionName", "domains-edit-input", "domainTopiaId", "${domain.topiaId}"}),
            @Result(
                name = DomainsEdit.ACTION_EDIT_PLOT, type = "redirectAction",
                params = {"namespace", "/plots", "actionName", "plots-edit-input", "plotTopiaId", "${nextParamValue}"}),
            @Result(
                name = DomainsEdit.ACTION_DUPLICATE_PLOT, type = "redirectAction",
                params = {"namespace", "/plots", "actionName", "plots-edit-input", "plotTopiaId", "${nextParamValue}"}),
            @Result(
                name = DomainsEdit.ACTION_EMPTY_PLOT, type = "redirectAction",
                params = {"namespace", "/plots", "actionName", "plots-edit-input", "domainTopiaId", "${domain.topiaId}"})})
    public String execute() throws Exception {

        propagateGpsDatasChanges();
        propagateWeatherStationsChanges();
        propagateSolChanges();
        propagateMaterielChanges();
        propagateToolsCouplingChanges();

        domain = domainService.createOrUpdateDomain(domain, locationTopiaId, legalStatusId, croppingPlan, otex18, otex70);
        notificationSupport.domainSaved(domain);

        if (Strings.isNullOrEmpty(domainTopiaId)) {
            navigationContextEntityCreated(domain);
        }

        // if next action is duplicate plot, duplicate before redirect
        if (DomainsEdit.ACTION_DUPLICATE_PLOT.equals(nextActionName)) {
        	Plot newPlot = plotService.duplicatePlot(nextParamValue);
        	nextParamValue = newPlot.getTopiaId();
        	notificationSupport.plotDuplicated(newPlot);
        }
        return nextActionName != null ? nextActionName : SUCCESS;
    }

    /**
     * Propagate changes on domains's GpsDatas.
     */
    protected void propagateGpsDatasChanges() {
        List<GPSData> gpsDatas = getDomain().getGpsDatas();
        if (gpsDatas == null) {
            gpsDatas = Lists.newArrayList();
            domain.setGpsDatas(gpsDatas);
        }

        Map<String, GPSData> indexGpsDatas = Maps.uniqueIndex(gpsDatas, Entities.GET_TOPIA_ID);

        gpsDatas.clear(); // On vide la liste pour mieux la remplir

        if (gpsDataDtos != null) {
            Iterables.removeIf(gpsDataDtos, GpsDatas.IS_INVALID_GPS_DATA);
            for (GPSDataDto gpsDataDto : gpsDataDtos) {
                String topiaId = gpsDataDto.getTopiaId();
                GPSData gpsData;
                if (Strings.isNullOrEmpty(topiaId)) {
                    // nouvelle donnée gps
                    gpsData = domainService.newGpsData();
                } else {
                    // donnée gps déjà existante
                    gpsData = indexGpsDatas.get(topiaId);
                    Preconditions.checkState(gpsData != null, "Invalid topiaId for the given entity: " + topiaId);
                }
                propagateGpsDataChanges(gpsData, gpsDataDto);
                gpsDatas.add(gpsData);
            }
        }
    }

    /**
     * Propagate the GPSData's changes according the gpsDataDto.
     *
     * @param entity the gpsData
     * @param dto    the gpsDataDto
     */
    protected void propagateGpsDataChanges(GPSData entity, GPSDataDto dto) {
        entity.setName(dto.getName());
        entity.setLatitude(dto.getLatitude());
        entity.setLongitude(dto.getLongitude());
        entity.setDescription(dto.getDescription());
    }

    /**
     * Propagates changes on {@link Domain#getWeatherStations()}.
     */
    protected void propagateWeatherStationsChanges() {
        List<WeatherStation> weatherStations = getDomain().getWeatherStations();
        List<WeatherStation> nonDeleted = Lists.newArrayList();

        if (weatherStations == null) {
            weatherStations = Lists.newArrayList();
            getDomain().setWeatherStations(weatherStations);
        }

        Map<String, WeatherStation> indexWeatherStations = Maps.uniqueIndex(weatherStations, Entities.GET_TOPIA_ID);

        getDomain().setDefaultWeatherStation(null);

        if (weatherStationDtos != null) {
            for (WeatherStationDto weatherStationDto : weatherStationDtos) {
                String topiaId = weatherStationDto.getTopiaId();
                WeatherStation weatherStation;

                if (Strings.isNullOrEmpty(topiaId)) {
                    weatherStation = domainService.newWeatherStation();
                } else {
                    weatherStation = indexWeatherStations.get(topiaId);
                }

                if (weatherStation != null) {
                    String refStationId = weatherStationDto.getRefStationMeteoTopiaId();
                    RefStationMeteo refStation = domainService.findRefStationMeteoByTopiaId(refStationId);
                    WeatherStations.dtoToWeatherStation(weatherStationDto, weatherStation, refStation);

                    if (StringUtils.isBlank(topiaId)) {
                        weatherStations.add(weatherStation);
                    }

                    nonDeleted.add(weatherStation);

                    if (log.isDebugEnabled()) {
                        log.debug("adding weatherStation : ID : " + weatherStation.getTopiaId() + ", " + ReflectionToStringBuilder.toString(weatherStationDto));
                    }

                    // sets the default weather station
                    if (weatherStationDto.isDefaultSelected()) {
                        getDomain().setDefaultWeatherStation(weatherStation);
                    }
                }
            }
        }

        weatherStations.retainAll(nonDeleted);
    }

    /**
     * Convert submitted DTO list to up to date entity list.
     */
    protected void propagateMaterielChanges() {

        List<Materiel> domainMateriels = getDomain().getMateriels();
        List<Materiel> nonDeleted = Lists.newArrayList();
        if (domainMateriels == null) {
            domainMateriels = Lists.newArrayList();
            getDomain().setMateriels(domainMateriels);
        }

        // update list with dto
        Map<String, Materiel> domainMaterielMap = Maps.uniqueIndex(domainMateriels, Entities.GET_TOPIA_ID);

        if (domainMaterielDtos != null) {
            materiels = new HashMap<String, Materiel>();
            for (MaterielDto domainMaterielDto : domainMaterielDtos) {
                // find  entity instance
                Materiel domainMateriel;
                String topiaId = domainMaterielDto.getTopiaId();
                if (StringUtils.isNotBlank(topiaId)) {
                    domainMateriel = domainMaterielMap.get(topiaId);
                } else {
                    domainMateriel = domainService.newMateriel();
                    domainMateriel.setCode(UUID.randomUUID().toString());
                }
                materiels.put(domainMaterielDto.getHashKey(), domainMateriel);
                // overwrite entity instance
                domainMateriel.setName(domainMaterielDto.getName());
                domainMateriel.setDescription(domainMaterielDto.getDescription());
                domainMateriel.setMaterielETA(domainMaterielDto.isMaterielETA());

                // manual materiel management
                String materielTopiaId = domainMaterielDto.getMaterielTopiaId();
                if (StringUtils.isNotBlank(materielTopiaId)) {
                    RefMateriel refMateriel = referentielService.findMateriel(domainMaterielDto.getMaterielTopiaId());
                    domainMateriel.setRefMateriel(refMateriel);
                } else {
                    domainMateriel.setRefMateriel(null);
                }

                if (StringUtils.isBlank(topiaId)) {
                    domainMateriels.add(domainMateriel);
                }

                nonDeleted.add(domainMateriel);
            }
        }

        domainMateriels.retainAll(nonDeleted);
    }

    protected void propagateToolsCouplingChanges() {

        List<ToolsCoupling> toolsCouplings = getDomain().getToolsCoupling();
        List<ToolsCoupling> nonDeleted = Lists.newArrayList();
        if (toolsCouplings == null) {
            toolsCouplings = Lists.newArrayList();
            getDomain().setToolsCoupling(toolsCouplings);
        }

        // update list with dto
        Map<String, ToolsCoupling> toolsCouplingsMap = Maps.uniqueIndex(toolsCouplings, Entities.GET_TOPIA_ID);
        Map<String, ToolsCoupling> toolsCouplingsMapToRemove = new HashMap<String, ToolsCoupling>(toolsCouplingsMap);

        if (toolsCouplingDtos != null) {
            for (ToolsCouplingDto toolsCouplingDto : toolsCouplingDtos) {
                // find  entity instance
                ToolsCoupling toolsCoupling;
                String topiaId = toolsCouplingDto.getTopiaId();
                if (StringUtils.isNotBlank(topiaId)) {
                    toolsCoupling = toolsCouplingsMapToRemove.remove(topiaId);
                } else {
                    toolsCoupling = domainService.newToolsCoupling();
                    toolsCoupling.setCode(UUID.randomUUID().toString());
                }
                // overwrite entity instance
                Materiel tractor = materiels.get(toolsCouplingDto.getTractorKey());
                List<Materiel> equipements = new ArrayList<Materiel>();
                if (toolsCouplingDto.getEquipementsKeys() != null) {
                    for (String equipementKey : toolsCouplingDto.getEquipementsKeys()) {
                        equipements.add(materiels.get(equipementKey));
                    }
                }
                toolsCoupling.setTractor(tractor);
                toolsCoupling.setCouplingEquipements(equipements);
                toolsCoupling.setComment(toolsCouplingDto.getComment());
                toolsCoupling.setToolsCouplingName(toolsCouplingDto.getToolsCouplingName());
                toolsCoupling.setWorkforce(toolsCouplingDto.getWorkforce());

                Collection<RefInterventionAgrosystTravailEDI> mainsActions = toolsCouplingDto.getMainsActions();
                Collection<RefInterventionAgrosystTravailEDI> originalMainsActions = toolsCoupling.getMainsActions();
                Iterables.removeIf(mainsActions, IS_NOT_PERSITED);
                if (originalMainsActions != null) {
                    domainService.deleteUnusedMainAction(toolsCoupling.getCode(), originalMainsActions, mainsActions);
                }

                toolsCoupling.setMainsActions(mainsActions);

                if (StringUtils.isBlank(topiaId)) {
                    toolsCouplings.add(toolsCoupling);
                }

                nonDeleted.add(toolsCoupling);
            }
        }
        // David 20/11/2013 add suppression of datas related to the deleted tools couplings.
        domainService.deleteToolsCouplings(toolsCouplingsMapToRemove.values());
        toolsCouplings.retainAll(nonDeleted);
    }

    protected void propagateSolChanges() {
        List<Sol> domainSols = getDomain().getSols();
        List<Sol> nonDeleted = Lists.newArrayList();
        if (domainSols == null) {
            domainSols = Lists.newArrayList();
            getDomain().setSols(domainSols);
        }

        // update list with dto
        Map<String, Sol> domainSolMap = Maps.uniqueIndex(domainSols, Entities.GET_TOPIA_ID);

        if (solDtos != null) {
            for (SolDto solDto : solDtos) {
                String topiaId = solDto.getTopiaId();
                Sol sol;
                if (Strings.isNullOrEmpty(topiaId)) {
                    // nouvelle période d'activité
                    sol = domainService.newSol();
                } else {
                    // période d'activité déjà existante
                    sol = domainSolMap.get(topiaId);
                }

                // propagate changes
                sol.setName(solDto.getName());
                sol.setComment(solDto.getComment());
                sol.setImportance(solDto.getImportance());
                RefSolArvalis refSolArvalis = referentielService.findSolArvalis(solDto.getSolArvalisTopiaId());
                sol.setRefSolArvalis(refSolArvalis);

                // can't do this before findSolArvalis, otherwize hibernate complains !!!
                if (Strings.isNullOrEmpty(topiaId)) {
                    domainSols.add(sol);
                }
                nonDeleted.add(sol);
            }
        }

        domainSols.retainAll(nonDeleted);
    }

    /**
     * Get all the type that a a domain could be.
     *
     * @return all the type
     */
    public Map<DomainType, String> getTypes() {
        return getEnumAsMap(DomainType.values());
    }

    public Map<AgrosystInterventionType, String> getDomainAgrosystInterventionTypes() {
        return getEnumAsMap(AgrosystInterventionType.values());
    }
    /**
     * Get all the agrosystInterventionType
     * @return all the agrosystInterventionType
     */
    public String getAgrosystInterventionTypes() {
        Map<AgrosystInterventionType, String> agrosystInterventionTypes = getEnumAsMap(AgrosystInterventionType.values());
        String interventionTypesJson = getGson().toJson(agrosystInterventionTypes);
        return interventionTypesJson;
    }

    /**
     * Get all the VulnerableArea that a a domain could be.
     *
     * @return all the VulnerableArea
     */
    public Map<Zoning, String> getZoningValues() {
        return getEnumAsMap(Zoning.values());
    }

    public void setLocationTopiaId(String locationTopiaId) {
        this.locationTopiaId = locationTopiaId;
    }

    public String getDepartement() {
        return departement;
    }

    public String getFormatedDepartement() {
        if (StringUtils.isBlank(departement)) {
            return "";
        }
        String depNumber = Strings.padStart(departement, 2, '0');
        String key = "departement." + depNumber;
        String result = getText(key);
        result = result + " (" + depNumber + ")";
        return result;
    }

    public String getPetiteRegionAgricole() {
        return petiteRegionAgricole;
    }

    public String getPetiteRegionAgricoleName() {
        return petiteRegionAgricoleName;
    }

    public String getFormatedPetiteRegionAgricoleName() {
        if (StringUtils.isBlank(petiteRegionAgricoleName)) {
            return "";
        }
        String result = petiteRegionAgricoleName + " (" + petiteRegionAgricole + ")";
        return result;
    }

    public List<GPSDataDto> getGpsDataDtos() {
        return gpsDataDtos;
    }

    public void setGpsDataDtosJson(String gpsDataDtos) {
        Type type = new TypeToken<List<GPSDataDto>>() {
        }.getType();
        this.gpsDataDtos = getGson().fromJson(gpsDataDtos, type);
    }

    public void setDomainTopiaId(String domainTopiaId) {
        this.domainTopiaId = domainTopiaId;
    }

    public String getCommune() {
        return commune;
    }

    public void setCommune(String commune) {
        this.commune = commune;
    }

    public String getLegalStatusId() {
        return legalStatusId;
    }

    public void setLegalStatusId(String legalStatusId) {
        this.legalStatusId = legalStatusId;
    }

    public List<RefLegalStatus> getAllRefLegalStatus() {
        return allRefLegalStatus;
    }

    public List<RefStationMeteo> getAllRefStationMeteo() {
        return allRefStationMeteo;
    }

    public Map<String, String> getAllRefStationMeteoIdsAndNames() {
        Map<String, String> weatherStationIdsAndNames = Maps.newHashMapWithExpectedSize(allRefStationMeteo.size());
        for (RefStationMeteo refStationMeteo : allRefStationMeteo) {
            weatherStationIdsAndNames.put(refStationMeteo.getTopiaId(), refStationMeteo.getCommuneSite());
        }
        return weatherStationIdsAndNames;
    }

    public String getLocationTopiaId() {
        return locationTopiaId;
    }

    public List<MaterielDto> getDomainMaterielDtos() {
        return domainMaterielDtos;
    }

    public void setDomainMaterielDtosJson(String json) {
        Type type = new TypeToken<List<MaterielDto>>() {
        }.getType();
        this.domainMaterielDtos = getGson().fromJson(json, type);
    }

    public List<ToolsCouplingDto> getToolsCouplingDtos() {
        return toolsCouplingDtos;
    }

    public void setToolsCouplingDtosJson(String json) {
        Type type = new TypeToken<List<ToolsCouplingDto>>() {
        }.getType();
        this.toolsCouplingDtos = getGson().fromJson(json, type);
    }

    public Map<MaterielType, List<String>> getMaterielType1List() {
        return materielType1List;
    }

    public List<SolDto> getSolDtos() {
        return solDtos;
    }

    public void setSolDtosJson(String json) {
        Type type = new TypeToken<List<SolDto>>() {
        }.getType();
        this.solDtos = getGson().fromJson(json, type);
    }

    public Map<Integer, String> getSolArvalisRegions() {
        return solArvalisRegions;
    }

    public Map<Integer, String> getRelatedDomains() {
        return relatedDomains;
    }

    public List<CroppingPlanEntryDto> getCroppingPlan() {
        return croppingPlan;
    }

    public void setCroppingPlanJson(String json) {
        Type type = new TypeToken<List<CroppingPlanEntryDto>>() {
        }.getType();
        this.croppingPlan = getGson().fromJson(json, type);
    }

//    public void setSelectedInterventionType(
//            InterventionType selectedInterventionType) {
//        this.selectedInterventionType = selectedInterventionType;
//    }



    public Integer getOtex18() {
        return otex18;
    }

    public void setOtex18(Integer otex18) {
        this.otex18 = otex18;
    }

    public Map<Integer, String> getOtex18s() {
        return otex18s;
    }

    public void setOtex70(Integer otex70) {
        this.otex70 = otex70;
    }

    public List<Plot> getPlots() {
        return plots;
    }

    public List<WeatherStationDto> getWeatherStations() {
        return weatherStationDtos;
    }

    public void setWeatherStationsJson(String json) {
        Type type = new TypeToken<List<WeatherStationDto>>() {
        }.getType();
        this.weatherStationDtos = getGson().fromJson(json, type);
    }

    public List<Price> getPrices() {
        return prices;
    }

    public Map<AgrosystInterventionType, String> getAgrosystInterventionType() {
        return getEnumAsMap(AgrosystInterventionType.values());
    }

    public Collection<RefInterventionAgrosystTravailEDI> getActionsEDIs() {
        return actionsEDIs;
    }

    public AgrosystValidationState getValidationState() {
        return validationState;
    }

}
