package fr.inra.agrosyst.web;

/*
 * #%L
 * Agrosyst :: Web
 * $Id: AgrosystWebAuthenticationFilter.java 2318 2013-11-28 16:26:30Z athimel $
 * $HeadURL: https://forge.codelutin.com/svn/agrosyst/tags/agrosyst-0.8.1/agrosyst-web/src/main/java/fr/inra/agrosyst/web/AgrosystWebAuthenticationFilter.java $
 * %%
 * Copyright (C) 2013 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.io.IOException;
import java.util.Set;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.google.common.base.Function;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableSet;

/**
 * @author Arnaud Thimel : thimel@codelutin.com
 */
public class AgrosystWebAuthenticationFilter implements Filter {

    private static final Log log = LogFactory.getLog(AgrosystWebAuthenticationFilter.class);

    public static final String AGROSYST_WEB_LOGIN_ACTION = "/auth/login.action";
    public static final String AGROSYST_WEB_LOGIN_ACTION_INPUT = "/auth/login-input.action";

    protected static final Set<String> AUTHORIZED_URLS = ImmutableSet.of(
            AGROSYST_WEB_LOGIN_ACTION,
            AGROSYST_WEB_LOGIN_ACTION_INPUT,
            "/js/endpoints-js.jsp",
            "/img/logo-Ecophyto-Dephy.png",
            "/img/Login-Background.png",
            "/img/Login-Pannel.png",
            "/img/Sprite-Buttons.png"
    );

    protected static final Function<HttpServletRequest, String> GET_FULL_REQUESTED_URI = new Function<HttpServletRequest, String>() {
        @Override
        public String apply(HttpServletRequest input) {
            String contextPath = input.getContextPath();
            String requestURI = input.getRequestURI();
            if (requestURI.startsWith(contextPath + "/js/") || requestURI.startsWith(contextPath + "/nuiton-js/") || requestURI.startsWith(contextPath + "/img/")) {
                requestURI = contextPath + "/";
            }
            String queryString = input.getQueryString();
            String result;
            if (queryString == null) {
                result = requestURI;
            } else {
                result = String.format("%s?%s", requestURI, queryString);
            }
            return result;
        }
    };

    protected static final Function<HttpServletRequest, String> GET_REDIRECT_TO_LOGIN_PAGE_URI = new Function<HttpServletRequest, String>() {
        @Override
        public String apply(HttpServletRequest input) {
            String contextPath = input.getContextPath();
            String result;
            if ("/".equals(contextPath)) {
                result = AGROSYST_WEB_LOGIN_ACTION_INPUT;
            } else {
                result = contextPath + AGROSYST_WEB_LOGIN_ACTION_INPUT;
            }
            return result;
        }
    };

    @Override
    public void init(FilterConfig filterConfig) throws ServletException {

        if (log.isInfoEnabled()) {
            log.info("Initializing " + AgrosystWebAuthenticationFilter.class.getName());
        }

        // Nothing to do
    }

    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException, ServletException {
        if (request instanceof HttpServletRequest) {
            HttpServletRequest httpServletRequest = (HttpServletRequest) request;

            String servletPath = httpServletRequest.getServletPath();

            boolean authorized = AUTHORIZED_URLS.contains(servletPath);

            if (!authorized) {
                HttpSession session = httpServletRequest.getSession(false);

                if (session != null) {
                    Object sessionObject = session.getAttribute(AgrosystWebSession.SESSION_PARAMETER);
                    if (sessionObject != null) {
                        AgrosystWebSession agrosystSession = (AgrosystWebSession) sessionObject;
                        authorized = !Strings.isNullOrEmpty(agrosystSession.getAuthenticationToken());
                    }
                }
            }

            if (log.isTraceEnabled()) {
                String message = String.format("Is '%s' authorized ? %b", servletPath, authorized);
                log.trace(message);
            }
            if (authorized) {
                chain.doFilter(request, response);
            } else {
                if (response instanceof HttpServletResponse) {
                    HttpSession session = httpServletRequest.getSession();
                    Object sessionObject = session.getAttribute(AgrosystWebSession.SESSION_PARAMETER);
                    if (sessionObject == null) {
                        AgrosystWebSession agrosystSession = new AgrosystWebSession();
                        session.setAttribute(AgrosystWebSession.SESSION_PARAMETER, agrosystSession);
                    }

                    HttpServletResponse httpServletResponse = (HttpServletResponse) response;
                    String redirectToLoginPageUri = GET_REDIRECT_TO_LOGIN_PAGE_URI.apply(httpServletRequest);

                    String requestedUri = GET_FULL_REQUESTED_URI.apply(httpServletRequest);
                    // FIXME echatellier 20130819 : il faut coder l'url car avec deux parametres, cela
                    // ne fonctionne plus:
                    // /xxx?param1=1&param2=2 => ?next=/xxx?param1=1&param2
                    String queryString = "?next=" + ((HttpServletResponse) response).encodeURL(requestedUri);

                    httpServletResponse.sendRedirect(redirectToLoginPageUri + queryString);
                }
            }
        }

    }

    @Override
    public void destroy() {

        if (log.isInfoEnabled()) {
            log.info("Destroying " + AgrosystWebAuthenticationFilter.class.getName());
        }
        // Nothing to do
    }

}
