/*
 * #%L
 * Agrosyst :: Web
 * $Id: ManagementModesEdit.java 2030 2013-11-06 16:33:46Z echatellier $
 * $HeadURL: https://forge.codelutin.com/svn/agrosyst/tags/agrosyst-0.7.1/agrosyst-web/src/main/java/fr/inra/agrosyst/web/actions/managementmodes/ManagementModesEdit.java $
 * %%
 * Copyright (C) 2013 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */
package fr.inra.agrosyst.web.actions.managementmodes;

import java.lang.reflect.Type;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.apache.struts2.convention.annotation.Action;
import org.apache.struts2.convention.annotation.Result;

import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.gson.reflect.TypeToken;
import com.opensymphony.xwork2.Preparable;

import fr.inra.agrosyst.api.NavigationContext;
import fr.inra.agrosyst.api.entities.CategoryObjective;
import fr.inra.agrosyst.api.entities.CategoryStrategy;
import fr.inra.agrosyst.api.entities.CroppingPlanEntry;
import fr.inra.agrosyst.api.entities.DecisionRule;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.ManagementMode;
import fr.inra.agrosyst.api.entities.ManagementModeCategory;
import fr.inra.agrosyst.api.entities.ManagementModeImpl;
import fr.inra.agrosyst.api.entities.Section;
import fr.inra.agrosyst.api.entities.SectionType;
import fr.inra.agrosyst.api.entities.StrategyType;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.growingplan.GrowingPlanFilter;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemFilter;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemService;
import fr.inra.agrosyst.api.services.managementmode.ManagementModeService;
import fr.inra.agrosyst.api.services.managementmode.ManagementModes;
import fr.inra.agrosyst.api.services.managementmode.SectionDto;
import fr.inra.agrosyst.web.actions.AbstractAgrosystAction;

/**
 * Management mode edit action.
 * 
 * @author Eric Chatellier
 */
public class ManagementModesEdit extends AbstractAgrosystAction implements Preparable {

    /** serialVersionUID. */
    private static final long serialVersionUID = -2688592598121764478L;

    protected ManagementModeService managementModeService;
    protected GrowingSystemService growingSystemService;

    /** Management mode id to edit. */
    protected String managementModeTopiaId;

    /** Management mode entity to edit. */
    protected ManagementMode managementMode;

    /** Growing system list (only for edit mode). */
    protected Set<GrowingSystem> growingSystems;

    /** Growing system id (edit mode). */
    protected String growingSystemTopiaId;

    /** Current management mode sections. */
    protected List<SectionDto> sections;

    /** Type de bio agresseurs. */
    protected List<String> bioAgressorTypes;

    /** Liste des cultures associées au système de culture sélectionné. */
    protected List<CroppingPlanEntry> croppingPlanEntries;

    /** Liste des regles associés au système de culture sélectionné. */
    protected List<DecisionRule> decisionRules;

    public void setManagementModeService(ManagementModeService managementModeService) {
        this.managementModeService = managementModeService;
    }

    public void setGrowingSystemService(GrowingSystemService growingSystemService) {
        this.growingSystemService = growingSystemService;
    }

    public ManagementMode getManagementMode() {
        if (managementMode == null) {
            // EChatellier 27/06/2013 Fais chier de devoir écrire ça, mais c'est la seule option pour ne pas avoir une grosse dose d'exceptions avec du paramsPrepareParams
            return new ManagementModeImpl();
        }
        return managementMode;
    }
    
    @Override
    public void prepare() throws Exception {
        if (StringUtils.isEmpty(managementModeTopiaId)) {
            // Cas de création d'un growingSystem
            managementMode = managementModeService.newManagementMode();
        } else {
            // Cas d'une mise à jour de growingSystem
            managementMode = managementModeService.getManagementMode(managementModeTopiaId);
        }
    }
    
    /**
     * Initialisation de certaines variables pour le premier appel de la page.
     */
    @Override
    @Action("management-modes-edit-input")
    public String input() throws Exception {

        if (!Strings.isNullOrEmpty(managementModeTopiaId)) {
            authorizationService.checkManagementModeReadable(managementModeTopiaId);
            readOnly = !authorizationService.isManagementModeWritable(managementModeTopiaId);
        }
        if (readOnly) {
            notificationSupport.managementModeNotWritable();
        }

        if (getManagementMode().getGrowingSystem() != null) {
            growingSystemTopiaId = managementMode.getGrowingSystem().getTopiaId();
        }

        if (managementMode.getSections() != null) {
            List<Section> mmSections = managementMode.getSections();
            sections = Lists.newArrayList(Lists.transform(mmSections, ManagementModes.SECTION_TO_DTO));
        } else {
            sections = Collections.emptyList();
        }
        initForInput();

        return INPUT;
    }
    
    /**
     * Initialisation des listes ou autres données à chaque affichage (premier/erreurs).
     */
    @Override
    protected void initForInput() {
        
        // select combo box
        if (!getManagementMode().isPersisted()) {
            GrowingSystemFilter growingSystemFilter = new GrowingSystemFilter();
            NavigationContext navigationContext = getNavigationContext();
            growingSystemFilter.setNavigationContext(navigationContext);
            growingSystemFilter.setActive(true);
            growingSystemFilter.setPageSize(GrowingPlanFilter.ALL_PAGE_SIZE);
            ResultList<GrowingSystem> growingSystemListResult = growingSystemService.getFilteredGrowingSystems(growingSystemFilter);
            growingSystems = Sets.newLinkedHashSet(growingSystemListResult.getElements());
        }

        if (StringUtils.isNotBlank(growingSystemTopiaId)) {
            croppingPlanEntries = managementModeService.getGrowingSystemCroppingPlanEntries(growingSystemTopiaId);
            decisionRules = managementModeService.getGrowingSystemDecisionRules(growingSystemTopiaId);
        }

        bioAgressorTypes = managementModeService.getBioAgressorTypes();
    }
    
    @Override
    public void validate() {

        // growing system mandatory only during edit mode
        if (!getManagementMode().isPersisted() && StringUtils.isBlank(growingSystemTopiaId)) {
            addFieldError("growingSystemTopiaId", "Le système de culture est obligatoire !");
        }

        if (hasErrors()) {
            initForInput();
        }
    }
    
    @Override
    @Action(results = {@Result(type = "redirectAction", params = {"actionName", "management-modes-edit-input", "managementModeTopiaId", "${managementMode.topiaId}"})})
    public String execute() throws Exception {

        managementMode = managementModeService.updateManagementMode(managementMode, growingSystemTopiaId, sections);
        notificationSupport.managementModeSaved(managementMode);

        return super.execute();
    }

    public String getManagementModeTopiaId() {
        return managementModeTopiaId;
    }

    public void setManagementModeTopiaId(String managementModeTopiaId) {
        this.managementModeTopiaId = managementModeTopiaId;
    }

    public String getGrowingSystemTopiaId() {
        return growingSystemTopiaId;
    }

    public void setGrowingSystemTopiaId(String growingSystemTopiaId) {
        this.growingSystemTopiaId = growingSystemTopiaId;
    }

    public Set<GrowingSystem> getGrowingSystems() {
        return growingSystems;
    }
    
    public Map<ManagementModeCategory, String> getManagementModeCategories() {
        return getEnumAsMap(ManagementModeCategory.values());
    }
    
    public Map<SectionType, String> getSectionTypes() {
        return getEnumAsMap(SectionType.values());
    }
    
    public Map<CategoryObjective, String> getCategoryObjectives() {
        return getEnumAsMap(CategoryObjective.values());
    }
    
    public Map<CategoryStrategy, String> getCategoryStrategies() {
        return getEnumAsMap(CategoryStrategy.values());
    }
    
    public Map<StrategyType, String> getStrategyTypes() {
        return getEnumAsMap(StrategyType.values());
    }

    public List<String> getBioAgressorTypes() {
        return bioAgressorTypes;
    }

    public List<SectionDto> getSections() {
        return sections;
    }
    
    public void setSections(String json) {
        Type type = new TypeToken<List<SectionDto>>() {
        }.getType();
        this.sections = getGson().fromJson(json, type);
    }

    public List<CroppingPlanEntry> getCroppingPlanEntries() {
        return croppingPlanEntries;
    }

    public List<DecisionRule> getDecisionRules() {
        return decisionRules;
    }
}
