/*
 * #%L
 * Agrosyst :: Web
 * $Id: DecisionRulesEdit.java 2092 2013-11-10 09:13:41Z echatellier $
 * $HeadURL: https://forge.codelutin.com/svn/agrosyst/tags/agrosyst-0.7.1/agrosyst-web/src/main/java/fr/inra/agrosyst/web/actions/managementmodes/DecisionRulesEdit.java $
 * %%
 * Copyright (C) 2013 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */
package fr.inra.agrosyst.web.actions.managementmodes;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.apache.struts2.convention.annotation.Action;
import org.apache.struts2.convention.annotation.Result;

import com.google.common.base.Strings;
import com.google.common.collect.Sets;
import com.opensymphony.xwork2.Preparable;

import fr.inra.agrosyst.api.NavigationContext;
import fr.inra.agrosyst.api.entities.CroppingPlanEntry;
import fr.inra.agrosyst.api.entities.DecisionRule;
import fr.inra.agrosyst.api.entities.DecisionRuleImpl;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.InterventionType;
import fr.inra.agrosyst.api.entities.referentiels.RefBioAgressor;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.growingplan.GrowingPlanFilter;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemFilter;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemService;
import fr.inra.agrosyst.api.services.managementmode.ManagementModeService;
import fr.inra.agrosyst.web.actions.AbstractAgrosystAction;

/**
 * Decision rule edit action.
 * 
 * @author Eric Chatellier
 */
public class DecisionRulesEdit extends AbstractAgrosystAction implements Preparable {

    /** serialVersionUID. */
    private static final long serialVersionUID = -6937262808324248162L;

    protected ManagementModeService managementModeService;
    protected GrowingSystemService growingSystemService;

    /** Decision rule id to edit. */
    protected String decisionRuleTopiaId;

    /** Decision rule entity to edit. */
    protected DecisionRule decisionRule;

    /** Growing system list (only for edit mode). */
    protected Set<GrowingSystem> growingSystems;

    /** Growing system id (edit mode). */
    protected String growingSystemTopiaId;

    /** Liste des cultures associées au système de culture sélectionné. */
    protected List<CroppingPlanEntry> croppingPlanEntries;

    /** Selected cropping plan entry topia Id. */
    protected String croppingPlanEntryTopiaId;

    /** Type de bio agresseurs. */
    protected List<String> bioAgressorTypes;

    /** Sous ensemble de bioagresseur pour le {@code bioAgressorType} sélectionné. */
    protected List<? extends RefBioAgressor> bioAgressors;

    /** Selected bio agressor topiaId. */
    protected String bioAgressorTopiaId;

    /** Code of the decision rule*/
    protected String decisionRuleCode;

    protected boolean newVersion;

    /** Related Decision Rules*/
    protected LinkedHashMap<Integer, String> relatedDecisionRules;

    public void setManagementModeService(ManagementModeService managementModeService) {
        this.managementModeService = managementModeService;
    }

    public void setGrowingSystemService(GrowingSystemService growingSystemService) {
        this.growingSystemService = growingSystemService;
    }

    public DecisionRule getDecisionRule() {
        if (decisionRule == null) {
            // EChatellier 27/06/2013 Fais chier de devoir écrire ça, mais c'est la seule option pour ne pas avoir une grosse dose d'exceptions avec du paramsPrepareParams
            return new DecisionRuleImpl();
        }
        return decisionRule;
    }

    @Override
    public void prepare() throws Exception {
        if (StringUtils.isBlank(decisionRuleTopiaId)) {
            if (StringUtils.isNotBlank(decisionRuleCode)) {
                // Cas ou l'on vien de la liste des règles de décision
                decisionRule = managementModeService.getCopiedDecisionRuleVersion(decisionRuleCode);
                decisionRuleTopiaId = decisionRule.getTopiaId();
            } else {
                // Cas de création d'une règle de décision
                decisionRule = managementModeService.newDecisionRule();
            }
        } else {
            // Cas d'une mise à jour d'une règle de décision, on récupère une copie afin de ne pas modifier l'original
            decisionRule = managementModeService.getCopiedDecisionRule(decisionRuleTopiaId);
        }
    }

    /**
     * Initialisation de certaines variables pour le premier appel de la page.
     */
    @Override
    @Action("decision-rules-edit-input")
    public String input() throws Exception {

        if (!Strings.isNullOrEmpty(decisionRuleTopiaId)) {
            authorizationService.checkDecisionRuleReadable(decisionRuleTopiaId);
            readOnly = !authorizationService.isDecisionRuleWritable(decisionRuleTopiaId);
        }
        if (readOnly) {
            notificationSupport.decisionRuleNotWritable();
        }

        if (getDecisionRule().getGrowingSystem() != null) {
            growingSystemTopiaId = decisionRule.getGrowingSystem().getTopiaId();
        }
        if (decisionRule.getCroppingPlanEntry() != null) {
            croppingPlanEntryTopiaId = decisionRule.getCroppingPlanEntry().getTopiaId();
        }
        if (decisionRule.getBioAgressor() != null) {
            bioAgressorTopiaId = decisionRule.getBioAgressor().getTopiaId();
        }

        initForInput();

        return INPUT;
    }

    /**
     * Initialisation des listes ou autres données à chaque affichage (premier/erreurs).
     */
    @Override
    protected void initForInput() {
        
        // select combo box
        if (!getDecisionRule().isPersisted()) {
            GrowingSystemFilter growingSystemFilter = new GrowingSystemFilter();
            NavigationContext navigationContext = getNavigationContext();
            growingSystemFilter.setNavigationContext(navigationContext);
            growingSystemFilter.setActive(true);
            growingSystemFilter.setPageSize(GrowingPlanFilter.ALL_PAGE_SIZE);
            ResultList<GrowingSystem> growingSystemListResult = growingSystemService.getFilteredGrowingSystems(growingSystemFilter);
            growingSystems = Sets.newLinkedHashSet(growingSystemListResult.getElements());
            
            croppingPlanEntries = Collections.emptyList();
        }

        if (StringUtils.isNotBlank(growingSystemTopiaId)) {
            croppingPlanEntries = managementModeService.getGrowingSystemCroppingPlanEntries(growingSystemTopiaId);
            relatedDecisionRules = managementModeService.getRelatedDecisionRules(decisionRule.getCode());
        }
        
        bioAgressorTypes = managementModeService.getBioAgressorTypes();
        String bioAgressorType = decisionRule.getBioAgressorType();
        if (StringUtils.isNotBlank(bioAgressorType)) {
            bioAgressors = managementModeService.getBioAgressor(bioAgressorType);
        }
    }

    @Override
    public void validate() {

        // growing system mandatory only during edit mode
        if (!getDecisionRule().isPersisted() && StringUtils.isBlank(growingSystemTopiaId)) {
            addFieldError("growingSystemTopiaId", "Le système de culture est obligatoire !");
        }

        if (StringUtils.isBlank(decisionRule.getName())) {
            addFieldError("decisionRule.name", "Le nom est obligatoire !");
        }

        if (decisionRule.getInterventionType() == null) {
            addFieldError("decisionRule.interventionType", "Le type d'intervention EDI est obligatoire !");
        }

        if (hasErrors()) {
            initForInput();
        }
    }

    @Override
    @Action(results = {@Result(type = "redirectAction", params = {"actionName", "decision-rules-edit-input", "decisionRuleTopiaId", "${decisionRule.topiaId}"})})
    public String execute() throws Exception {

        decisionRule = managementModeService.updateDecisionRule(decisionRule, growingSystemTopiaId, croppingPlanEntryTopiaId, bioAgressorTopiaId, newVersion);
        decisionRuleTopiaId = decisionRule.getTopiaId();
        notificationSupport.decisionRuleSaved(decisionRule);

        return super.execute();
    }

    public Map<InterventionType, String> getInterventionTypes() {
        return getEnumAsMap(InterventionType.values());
    }

    public String getGrowingSystemTopiaId() {
        return growingSystemTopiaId;
    }

    public void setGrowingSystemTopiaId(String growingSystemTopiaId) {
        this.growingSystemTopiaId = growingSystemTopiaId;
    }

    public Set<GrowingSystem> getGrowingSystems() {
        return growingSystems;
    }

    public String getCroppingPlanEntryTopiaId() {
        return croppingPlanEntryTopiaId;
    }

    public void setCroppingPlanEntryTopiaId(String croppingPlanEntryTopiaId) {
        this.croppingPlanEntryTopiaId = croppingPlanEntryTopiaId;
    }

    public void setDecisionRuleTopiaId(String decisionRuleTopiaId) {
        this.decisionRuleTopiaId = decisionRuleTopiaId;
    }

    public String getDecisionRuleTopiaId() {
        return decisionRuleTopiaId;
    }

    public List<String> getBioAgressorTypes() {
        return bioAgressorTypes;
    }

    public List<? extends RefBioAgressor> getBioAgressors() {
        return bioAgressors;
    }

    public List<CroppingPlanEntry> getCroppingPlanEntries() {
        return croppingPlanEntries;
    }

    public String getBioAgressorTopiaId() {
        return bioAgressorTopiaId;
    }

    public void setBioAgressorTopiaId(String bioAgressorTopiaId) {
        this.bioAgressorTopiaId = bioAgressorTopiaId;
    }

    public Map<Integer, String> getRelatedDecisionRules() {
        return relatedDecisionRules;
    }

    public void setNewVersion(boolean newVersion) {
        this.newVersion = newVersion;
    }

    public void setDecisionRuleCode(String decisionRuleCode) {
        this.decisionRuleCode = decisionRuleCode;
    }
}
