package fr.inra.agrosyst.web.actions;

/*
 * #%L
 * Agrosyst :: Web
 * $Id: AbstractAgrosystAction.java 2091 2013-11-09 20:06:21Z echatellier $
 * $HeadURL: https://forge.codelutin.com/svn/agrosyst/tags/agrosyst-0.7.1/agrosyst-web/src/main/java/fr/inra/agrosyst/web/actions/AbstractAgrosystAction.java $
 * %%
 * Copyright (C) 2013 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.Calendar;
import java.util.Map;
import java.util.Set;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts2.interceptor.ParameterAware;
import org.apache.struts2.interceptor.ServletRequestAware;
import org.apache.struts2.interceptor.ServletResponseAware;
import org.nuiton.topia.persistence.TopiaEntity;

import com.google.common.base.Strings;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.gson.Gson;
import com.opensymphony.xwork2.ActionSupport;

import fr.inra.agrosyst.api.NavigationContext;
import fr.inra.agrosyst.api.services.context.NavigationContextService;
import fr.inra.agrosyst.api.services.security.BusinessAuthorizationService;
import fr.inra.agrosyst.commons.gson.AgrosystGsonSupplier;
import fr.inra.agrosyst.web.AgrosystWebConfig;
import fr.inra.agrosyst.web.AgrosystWebNotification;
import fr.inra.agrosyst.web.AgrosystWebNotificationSupport;
import fr.inra.agrosyst.web.AgrosystWebSession;

/**
 * Toutes les actions Struts doivent hériter de cette classe.
 *
 * @author Arnaud Thimel <thimel@codelutin.com>
 */
public abstract class AbstractAgrosystAction extends ActionSupport implements ServletRequestAware, ServletResponseAware,
        ParameterAware {

    private static final long serialVersionUID = 4829352350362628085L;

    private static final Log log = LogFactory.getLog(AbstractAgrosystAction.class);

    // Injectés par fr.inra.agrosyst.web.AgrosystWebInterceptor
    protected AgrosystWebConfig config;
    protected AgrosystWebSession session;
    protected AgrosystWebNotificationSupport notificationSupport;
    protected NavigationContextService navigationContextService;
    protected BusinessAuthorizationService authorizationService;

    protected HttpServletRequest servletRequest;
    protected HttpServletResponse servletResponse;

    protected Map<String, String[]> parameters;

    protected RichNavigationContext richNavigationContext;

    protected boolean readOnly = false;

    protected Boolean currentUserAnAdmin;

    /** Action name to redirect to after current action. */
    protected String nextActionName;
    /** Action param value to redirect to after current action. */
    protected String nextParamValue;

    /**
     * gson (can be used into non json action to initialize json data).
     */
    protected Gson gson;

    public Gson getGson() {
        if (gson == null) {
            gson = new AgrosystGsonSupplier() {
                @Override
                protected Map<Class<? extends TopiaEntity>, Class<? extends TopiaEntity>> getAllTypesAndImplementation() {
                    Map<String, String> stringMap = navigationContextService.getAllTypesAndImplementationAsString();

                    Map<Class<? extends TopiaEntity>, Class<? extends TopiaEntity>> result = Maps.newHashMap();
                    try {
                        for (Map.Entry<String, String> entry : stringMap.entrySet()) {
                            Class<? extends TopiaEntity> key = (Class<? extends TopiaEntity>) Class.forName(entry.getKey());
                            Class<? extends TopiaEntity> value = (Class<? extends TopiaEntity>) Class.forName(entry.getValue());
                            result.put(key, value);
                        }
                    } catch (ClassNotFoundException e) {
                        if (log.isErrorEnabled()) {
                            log.error("Un exception occurred", e);
                        }
                    }
                    return result;
                }
            }.get();
        }
        return gson;
    }

    public String toJson(Object element) {
        return getGson().toJson(element);
    }

    @Override
    public void setServletRequest(HttpServletRequest servletRequest) {
        this.servletRequest = servletRequest;
    }

    @Override
    public void setServletResponse(HttpServletResponse servletResponse) {
        this.servletResponse = servletResponse;
    }

    @Override
    public void setParameters(Map<String, String[]> parameters) {
        this.parameters = parameters;
    }

    public void setConfig(AgrosystWebConfig config) {
        this.config = config;
    }

    public void setSession(AgrosystWebSession session) {
        this.session = session;
    }

    public final void setNavigationContextService(NavigationContextService navigationContextService) { // AThimel 11/10/13 Final to make sure it is not overridden
        this.navigationContextService = navigationContextService;
    }

    public final void setAuthorizationService(BusinessAuthorizationService authorizationService) { // AThimel 11/10/13 Final to make sure it is not overridden
        this.authorizationService = authorizationService;
    }

    public void setNotificationSupport(AgrosystWebNotificationSupport notificationSupport) {
        this.notificationSupport = notificationSupport;
    }

    // Ne pas supprimer, méthode utilisée depuis les JSP
    public AgrosystWebConfig getConfig() {
        return config;
    }

    // Ne pas supprimer, méthode utilisée depuis les JSP
    public AgrosystWebSession getSession() {
        return session;
    }

    protected NavigationContext getNavigationContext() {
        NavigationContext navigationContext = session.getNavigationContext();

        // 'dirty' permet de savoir si les données du contexte doivent être validées
        boolean dirty = false;
        if (navigationContext == null) {

            // Lecture du cookie
            navigationContext = readNavigationContextCookie();
            dirty = true;

            // Pas de cookie trouvé, création du contexte de navigation par défaut
            if (navigationContext == null) {
                navigationContext = new NavigationContext();
                dirty = false;
            }

            if (navigationContext.getCampaigns().isEmpty()) {
                int campaign = Calendar.getInstance().get(Calendar.YEAR);
                navigationContext.setCampaigns(Sets.newHashSet(campaign));
            }
        }

        if (dirty) {
            navigationContext = verifyAndSaveNavigationContext(navigationContext);
        }
        return navigationContext;
    }

    protected NavigationContext verifyAndSaveNavigationContext(NavigationContext navigationContext) {
        NavigationContext newNavigationContext = navigationContextService.verify(navigationContext);
        session.setNavigationContext(newNavigationContext);
        writeNavigationContextCookie(newNavigationContext);
        return newNavigationContext;
    }

    protected NavigationContext navigationContextEntityCreated(TopiaEntity newEntity) {
        NavigationContext navigationContext = getNavigationContext();
        NavigationContext newNavigationContext = navigationContextService.verify(navigationContext, newEntity);
        session.setNavigationContext(newNavigationContext);
        writeNavigationContextCookie(newNavigationContext);
        return newNavigationContext;
    }

    protected NavigationContext readNavigationContextCookie() {
        NavigationContext result = null;
        if (servletRequest != null) {
            Cookie[] cookies = servletRequest.getCookies();
            if (cookies != null) {
                for (Cookie cookie : cookies) {
                    if (AgrosystWebConfig.NAVIGATION_CONTEXT_COOKIE_NAME.equals(cookie.getName())) {
                        result = new Gson().fromJson(cookie.getValue(), NavigationContext.class);
                        break;
                    }
                }
            }
        }
        return result;
    }

    protected void writeNavigationContextCookie(NavigationContext navigationContext) {
        Cookie cookie = new Cookie(AgrosystWebConfig.NAVIGATION_CONTEXT_COOKIE_NAME, getGson().toJson(navigationContext));
        cookie.setPath(servletRequest.getContextPath());
        servletResponse.addCookie(cookie);
    }

    public RichNavigationContext getRichNavigationContext() {
        if (richNavigationContext == null) {
            NavigationContext navigationContext = getNavigationContext();
            richNavigationContext = new RichNavigationContext(navigationContext, navigationContextService);
        }
        return richNavigationContext;
    }

    /**
     * Transform enumeration values into map with i18n value for each enum value.
     * <p/>
     * i18n key is fqn.NAME
     *
     * @param values values to transform
     * @return map (enum value > i18n text)
     */
    protected <T> Map<T, String> getEnumAsMap(T... values) {
        Map<T, String> valuesMap = Maps.newLinkedHashMap();
        for (T value : values) {
            String i18n = value.getClass().getName() + "." + value.toString();
            String trans = getText(i18n);
            valuesMap.put(value, trans);
        }
        return valuesMap;
    }

    protected void initForInput() {

    }

    /**
     * Return session info notification.
     * <p/>
     * Warning : this get method clear resulting collection
     *
     * @return session info notifications
     */
    public Set<AgrosystWebNotification> getInfoNotifications() {
        Set<AgrosystWebNotification> messages = Sets.newHashSet(session.getInfoNotifications());
        session.getInfoNotifications().clear();
        return messages;
    }

    public boolean isInfoNotificationsEmpty() {
        return session.getInfoNotifications().isEmpty();
    }

    /**
     * Return session warning notifications.
     * <p/>
     * Warning : this get method clear resulting collection
     *
     * @return session warning notifications
     */
    public Set<AgrosystWebNotification> getWarningNotifications() {
        Set<AgrosystWebNotification> messages = Sets.newHashSet(session.getWarningNotifications());
        session.getWarningNotifications().clear();
        return messages;
    }

    public boolean isWarningNotificationsEmpty() {
        return session.getWarningNotifications().isEmpty();
    }

    /**
     * Return session error notifications.
     * <p/>
     * Warning : this get method clear resulting collection
     *
     * @return session error notifications
     */
    public Set<AgrosystWebNotification> getErrorNotifications() {
        Set<AgrosystWebNotification> messages = Sets.newHashSet(session.getErrorNotifications());
        session.getErrorNotifications().clear();
        return messages;
    }

    public boolean isErrorNotificationsEmpty() {
        return session.getErrorNotifications().isEmpty();
    }

    public boolean isReadOnly() {
        return readOnly;
    }

    public boolean isCurrentUserAnAdmin() {
        if (currentUserAnAdmin == null) {
            if (Strings.isNullOrEmpty(getSession().getAuthenticationToken())) {
                currentUserAnAdmin = Boolean.FALSE;
            } else {
                currentUserAnAdmin = authorizationService.isAdmin();
            }
        }
        return currentUserAnAdmin;
    }

    public String getNextActionName() {
        return nextActionName;
    }

    public void setNextActionName(String nextActionName) {
        this.nextActionName = nextActionName;
    }

    public String getNextParamValue() {
        return nextParamValue;
    }

    public void setNextParamValue(String nextParamValue) {
        this.nextParamValue = nextParamValue;
    }
}
