package fr.inra.agrosyst.web;

/*
 * #%L
 * Agrosyst :: Web
 * $Id: AgrosystWebInterceptor.java 1849 2013-10-23 06:12:20Z athimel $
 * $HeadURL: https://forge.codelutin.com/svn/agrosyst/tags/agrosyst-0.7.1/agrosyst-web/src/main/java/fr/inra/agrosyst/web/AgrosystWebInterceptor.java $
 * %%
 * Copyright (C) 2013 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.beans.PropertyDescriptor;
import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts2.StrutsStatics;
import org.nuiton.topia.TopiaContext;
import org.nuiton.util.beans.BeanUtil;
import org.nuiton.web.filter.TopiaTransactionFilter;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.base.Supplier;
import com.opensymphony.xwork2.ActionInvocation;
import com.opensymphony.xwork2.interceptor.Interceptor;

import fr.inra.agrosyst.api.services.AgrosystService;
import fr.inra.agrosyst.api.services.ServiceFactory;
import fr.inra.agrosyst.api.services.security.AuthenticationService;
import fr.inra.agrosyst.api.services.users.UserDto;
import fr.inra.agrosyst.web.actions.AbstractAgrosystAction;

/**
 * Interceptor struts qui inject les parametres de type connu (Session, Service, Config) dans
 * l'action struts avant son execution.
 * 
 * @author Arnaud Thimel <thimel@codelutin.com>
 */
public class AgrosystWebInterceptor implements Interceptor {

    private static final long serialVersionUID = 2035088485913506328L;

    private static final Log log = LogFactory.getLog(AgrosystWebInterceptor.class);

    @Override
    public void init() {
        if (log.isInfoEnabled()) {
            log.info("Initializing Agrosyst Web Interceptor");
        }
    }

    @Override
    public String intercept(ActionInvocation invocation) throws Exception {

        Object action = invocation.getAction();

        if (action instanceof AbstractAgrosystAction) {


            AgrosystWebSession agrosystSession = getAgrosystSession(invocation);

            ServiceFactory serviceFactory = newServiceFactory(invocation, agrosystSession.getAuthenticationToken());

            populateAuthenticatedUser(agrosystSession, serviceFactory);

            Set<PropertyDescriptor> descriptors =
                    BeanUtil.getDescriptors(
                            action.getClass(),
                            BeanUtil.IS_WRITE_DESCRIPTOR);

            
            for (PropertyDescriptor propertyDescriptor : descriptors) {

                Class<?> propertyType = propertyDescriptor.getPropertyType();

                Object toInject = null;

                if (AgrosystService.class.isAssignableFrom(propertyType)) {
                    Class<? extends AgrosystService> serviceClass =
                            (Class<? extends AgrosystService>) propertyType;
                    toInject = serviceFactory.newService(serviceClass);
                } else if (AgrosystWebSession.class.isAssignableFrom(propertyType)) {
                    toInject = agrosystSession;
                } else if (AgrosystWebConfig.class.isAssignableFrom(propertyType)) {
                    toInject = getAgrosystApplicationContext(invocation).getWebConfig();
                } else if (AgrosystWebNotificationSupport.class.isAssignableFrom(propertyType)) {
                    toInject = new AgrosystWebNotificationSupport(agrosystSession);
                }

                if (toInject != null) {

                    if (log.isTraceEnabled()) {
                        log.trace("injecting " + toInject + " in action " + action);
                    }

                    propertyDescriptor.getWriteMethod().invoke(action, toInject);

                }
            }

            return invocation.invoke();

        } else {

            // not an action, just process
            return invocation.invoke();

        }

    }

    private void populateAuthenticatedUser(AgrosystWebSession agrosystSession, ServiceFactory serviceFactory) {
        if (!Strings.isNullOrEmpty(agrosystSession.getAuthenticationToken()) && agrosystSession.getAuthenticatedUser() == null) {
            if (log.isDebugEnabled()) {
                log.debug("Authenticated user DTO is dirty, reloading from service...");
            }
            AuthenticationService authenticationService = serviceFactory.newService(AuthenticationService.class);
            UserDto authenticatedUser = authenticationService.getAuthenticatedUser(agrosystSession.getAuthenticationToken());
            agrosystSession.setAuthenticatedUser(authenticatedUser);
        }
    }

    protected AgrosystWebSession getAgrosystSession(ActionInvocation invocation) {

        AgrosystWebSession result = (AgrosystWebSession) invocation.getInvocationContext().getSession().get(AgrosystWebSession.SESSION_PARAMETER);

        if (result == null) {
            result = new AgrosystWebSession();
            invocation.getInvocationContext().getSession().put(AgrosystWebSession.SESSION_PARAMETER, result);
        }

        return result;

    }

    protected AgrosystWebApplicationContext getAgrosystApplicationContext(ActionInvocation invocation) {

        AgrosystWebApplicationContext applicationContext =
                (AgrosystWebApplicationContext) invocation
                        .getInvocationContext()
                        .getApplication()
                        .get(AgrosystWebApplicationContext.APPLICATION_CONTEXT_PARAMETER);

        Preconditions.checkState(applicationContext != null, AgrosystWebTransactionFilter.MISSING_APPLICATION_CONTEXT);

        return applicationContext;

    }

    protected ServiceFactory newServiceFactory(ActionInvocation invocation, String authenticationToken) {

        final TopiaContext topiaContext = (TopiaContext)
                ((HttpServletRequest) invocation
                        .getInvocationContext()
                        .get(StrutsStatics.HTTP_REQUEST))
                        .getAttribute(TopiaTransactionFilter.TOPIA_TRANSACTION_REQUEST_ATTRIBUTE);

        AgrosystWebApplicationContext applicationContext = getAgrosystApplicationContext(invocation);

        ServiceFactory result = null;
        AgrosystWebConfig config = applicationContext.getWebConfig();
        if (config.isServicesRemoteEnabled()) {
            result = new RemoteServiceFactory(config, authenticationToken);
        } else {
            try {
                Supplier<TopiaContext> topiaContextSupplier = new Supplier<TopiaContext>() {
                    @Override
                    public TopiaContext get() {
                        return topiaContext;
                    }
                };
                Class<?> serviceContextClass = Class.forName("fr.inra.agrosyst.services.DefaultServiceContext");
                Class<?> serviceConfigClass = Class.forName("fr.inra.agrosyst.services.AgrosystServiceConfig");
                Constructor<?> constructor = serviceContextClass.getConstructor(serviceConfigClass, Supplier.class, String.class);

                Object serviceContext = constructor.newInstance(applicationContext.getServiceConfig(), topiaContextSupplier, authenticationToken);

                Method getServiceFactoryMethod = serviceContextClass.getMethod("getServiceFactory");
                result = (ServiceFactory)getServiceFactoryMethod.invoke(serviceContext);
            } catch (Exception eee) {
                if (log.isErrorEnabled()) {
                    log.error("Unable to create local serviceFactory", eee);
                }
            }
        }
        Preconditions.checkState(result != null, "ServiceFactory not instantiated, check configuration");
        return result;
    }

    @Override
    public void destroy() {
        if (log.isInfoEnabled()) {
            log.info("Destroy Interceptor");
        }
    }

}
