/*
 * #%L
 * Agrosyst :: Web
 * $Id: PracticedSystemsEdit.java 1873 2013-10-24 16:34:48Z dcosse $
 * $HeadURL: https://forge.codelutin.com/svn/agrosyst/tags/agrosyst-0.6.1/agrosyst-web/src/main/java/fr/inra/agrosyst/web/actions/practiced/PracticedSystemsEdit.java $
 * %%
 * Copyright (C) 2013 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

package fr.inra.agrosyst.web.actions.practiced;

import java.lang.reflect.Type;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.struts2.convention.annotation.Action;
import org.apache.struts2.convention.annotation.Result;

import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.gson.reflect.TypeToken;
import com.opensymphony.xwork2.Preparable;

import fr.inra.agrosyst.api.NavigationContext;
import fr.inra.agrosyst.api.entities.ActionType;
import fr.inra.agrosyst.api.entities.CapacityUnit;
import fr.inra.agrosyst.api.entities.CropCyclePhase;
import fr.inra.agrosyst.api.entities.CropCyclePhaseType;
import fr.inra.agrosyst.api.entities.FertilizerQuantityUnit;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.Intervention;
import fr.inra.agrosyst.api.entities.InterventionType;
import fr.inra.agrosyst.api.entities.OrchardFrutalForm;
import fr.inra.agrosyst.api.entities.PollinatorSpreadMode;
import fr.inra.agrosyst.api.entities.PracticedPerennialCropCycle;
import fr.inra.agrosyst.api.entities.PracticedSeasonalCropCycle;
import fr.inra.agrosyst.api.entities.PracticedSystem;
import fr.inra.agrosyst.api.entities.PracticedSystemImpl;
import fr.inra.agrosyst.api.entities.PracticedSystemSource;
import fr.inra.agrosyst.api.entities.SeedType;
import fr.inra.agrosyst.api.entities.UnitPlantSeeds;
import fr.inra.agrosyst.api.entities.VineFrutalForm;
import fr.inra.agrosyst.api.entities.WeedType;
import fr.inra.agrosyst.api.entities.YealdCategory;
import fr.inra.agrosyst.api.entities.YealdUnit;
import fr.inra.agrosyst.api.entities.referentiels.RefActionAgrosystTravailEDI;
import fr.inra.agrosyst.api.entities.referentiels.RefOrientationEDI;
import fr.inra.agrosyst.api.entities.referentiels.RefUnitesEDI;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.domain.CroppingPlanSpeciesDto;
import fr.inra.agrosyst.api.services.domain.CroppingPlans;
import fr.inra.agrosyst.api.services.domain.ToolsCouplingDto;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemFilter;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemService;
import fr.inra.agrosyst.api.services.practiced.ActionDto;
import fr.inra.agrosyst.api.services.practiced.CropCycleConnectionDto;
import fr.inra.agrosyst.api.services.practiced.CropCycleModelDto;
import fr.inra.agrosyst.api.services.practiced.CropCycleNodeDto;
import fr.inra.agrosyst.api.services.practiced.CropCyclePerennialSpeciesDto;
import fr.inra.agrosyst.api.services.practiced.CropCyclePhaseDto;
import fr.inra.agrosyst.api.services.practiced.InterventionDto;
import fr.inra.agrosyst.api.services.practiced.PracticedCropCycleService;
import fr.inra.agrosyst.api.services.practiced.PracticedPerennialCropCycleDto;
import fr.inra.agrosyst.api.services.practiced.PracticedSeasonalCropCycleDto;
import fr.inra.agrosyst.api.services.practiced.PracticedSystemService;
import fr.inra.agrosyst.api.services.practiced.PracticedSystems;
import fr.inra.agrosyst.api.services.referentiels.ReferentielService;
import fr.inra.agrosyst.web.actions.AbstractAgrosystAction;

/**
 * Action d'édition d'un système pratiqué.
 * 
 * Avec:
 * <ul>
 * <li>l'onglet géneralité
 * <li>la liste des cycles pluriannuels assolés
 * <li>la liste des cycles pluriannuels pérennes
 * <li>l'itinéraire technique
 * </ul>
 * 
 * @author Eric Chatellier
 */
public class PracticedSystemsEdit extends AbstractAgrosystAction implements Preparable {

    /** serialVersionUID. */
    private static final long serialVersionUID = -5696853256365484417L;

    protected PracticedSystemService practicedSystemService;

    protected GrowingSystemService growingSystemService;
    
    protected PracticedCropCycleService practicedCropCycleService;
    
    protected ReferentielService referentielService;

    /** L'id du systeme pratiqué en cours d'edition. */
    protected String practicedSystemId;

    /** L'instance du systeme pratiqué en cours d'edition. */
    protected PracticedSystem practicedSystem;

    /** La liste de systeme de culture auquel peut être ratachés le SP (en creation). */
    protected List<GrowingSystem> growingSystems;

    /** Le systeme de culture sélectionné. */
    protected String growingSystemId;

    /** La liste des campaignes. */
    protected String campaigns;

    protected List<PracticedPerennialCropCycleDto> practicedPerennialCropCycleDtos;

    /** @deprecated pas de necessité d'être en globale */
    @Deprecated
    protected List<Intervention> interventions;

    protected List<RefOrientationEDI> refOrientationEDIs;

    protected List<RefActionAgrosystTravailEDI> refActionAgrosystTravailEDIs;

    protected Map<Integer, String> allActiveFertiMinProductType;

    /** Liste des cycles pluriannuels assolé. */
    protected List<PracticedSeasonalCropCycleDto> practicedSeasonalCropCycleDtos;

    /** Les combinaisons d'outils concernant le système de cultures pour les années sélectionnées. */
    protected List<ToolsCouplingDto> practicedSystemToolsCouplings;

    /**
     * Contient une map <culture, List<Species>> pour toutes les cultures des systemes de cultures
     * concernés sur les campagnes du systeme pratiqué.
     */
    protected Map<String, List<CroppingPlanSpeciesDto>> practicedSystemCroppingEntryPlanToSpecies;

    /** Les cultures principales disponibles dans le graphique. */
    protected List<CropCycleModelDto> practicedSystemMainCropCycleModels;

    /** Les cultures intermédiaires disponible dans le graphique. */
    protected List<CropCycleModelDto> practicedSystemIntermediateCropCycleModels;

    protected List<RefUnitesEDI> refUnitesEDIs;

    public void setPracticedSystemService(PracticedSystemService practicedSystemService) {
        this.practicedSystemService = practicedSystemService;
    }
    
    public void setPracticedCropCycleService(PracticedCropCycleService practicedCropCycleService) {
        this.practicedCropCycleService = practicedCropCycleService;
    }

    public void setGrowingSystemService(GrowingSystemService growingSystemService) {
        this.growingSystemService = growingSystemService;
    }
    
    public void setReferentielService(ReferentielService referentielService) {
        this.referentielService = referentielService;
    }

    public PracticedSystem getPracticedSystem() {
        if (practicedSystem == null) {
            // EChatellier 27/06/2013 Fais chier de devoir écrire ça, mais c'est la seule option pour ne pas avoir une grosse dose d'exceptions avec du paramsPrepareParams
            return new PracticedSystemImpl();
        }
        return practicedSystem;
    }

    @Override
    public void prepare() {
        if (StringUtils.isNotBlank(practicedSystemId)) {
            practicedSystem = practicedSystemService.getPracticedSystem(practicedSystemId);
            
        } else {
            // j'offre la possibilité de crééer un PracticedCropCycle
            practicedSystem = new PracticedSystemImpl();
        }
    }

    @Override
    @Action("practiced-systems-edit-input")
    public String input() {
        initForInput();

        // default to empty lists
        practicedSeasonalCropCycleDtos = Lists.newArrayList();
        practicedPerennialCropCycleDtos = Lists.newArrayList();

        if (practicedSystem.isPersisted()) {
            authorizationService.checkPracticedSystemReadable(practicedSystemId);

            readOnly = !authorizationService.isPracticedSystemWritable(practicedSystemId);
            if (readOnly) {
                notificationSupport.practicedSystemNotWritable();
            }

            List<PracticedPerennialCropCycle> practicedPerennialCropCycles =  practicedCropCycleService.getAllPracticedPerennialCropCycle(practicedSystem);
            practicedPerennialCyclesInput(practicedPerennialCropCycles);
            List<PracticedSeasonalCropCycle> practicedSeasonalCropCycles = practicedCropCycleService.getAllPracticedSeasonalCropCycle(practicedSystem);
            practicedSeasonalCyclesInput(practicedSeasonalCropCycles);
        }

        return INPUT;
    }

    protected void practicedPerennialCyclesInput(List<PracticedPerennialCropCycle> practicedPerennialCropCycles) {
        for (PracticedPerennialCropCycle practicedPerennialCropCycle : practicedPerennialCropCycles) {
            PracticedPerennialCropCycleDto practicedPerennialCropCycleDto = new PracticedPerennialCropCycleDto();
            String cycleId = practicedPerennialCropCycle.getTopiaId();
            practicedPerennialCropCycleDto.setTopiaId(cycleId);
            practicedPerennialCropCycleDto.setPlantingYear(practicedPerennialCropCycle.getPlantingYear());
            practicedPerennialCropCycleDto.setPlantingDensity(practicedPerennialCropCycle.getPlantingDensity());
            practicedPerennialCropCycleDto.setPlantingInterFurrow(practicedPerennialCropCycle.getPlantingInterFurrow());
            practicedPerennialCropCycleDto.setPlantingSpacing(practicedPerennialCropCycle.getPlantingSpacing());
            practicedPerennialCropCycleDto.setPollinator(practicedPerennialCropCycle.isPollinator());
            practicedPerennialCropCycleDto.setPlantingDeathRate(practicedPerennialCropCycle.getPlantingDeathRate());
            practicedPerennialCropCycleDto.setPlantingDeathRateMeasureYear(practicedPerennialCropCycle.getPlantingDeathRateMeasureYear());
            practicedPerennialCropCycleDto.setVineFrutalForm(practicedPerennialCropCycle.getVineFrutalForm());
            practicedPerennialCropCycleDto.setOrchardFrutalForm(practicedPerennialCropCycle.getOrchardFrutalForm());
            practicedPerennialCropCycleDto.setPollinatorSpreadMode(practicedPerennialCropCycle.getPollinatorSpreadMode());
            practicedPerennialCropCycleDto.setWeedType(practicedPerennialCropCycle.getWeedType());
            practicedPerennialCropCycleDto.setPollinatorPercent(practicedPerennialCropCycle.getPollinatorPercent());
            practicedPerennialCropCycleDto.setOtherCaracteristics(practicedPerennialCropCycle.getOtherCaracteristics());

            RefOrientationEDI orientation = practicedPerennialCropCycle.getOrientation();
            if (orientation != null) {
                practicedPerennialCropCycleDto.setOrientationTopiaId(orientation.getTopiaId());
            }

            String croppingPlanEntryCode = practicedPerennialCropCycle.getCroppingPlanEntryCode();
            practicedPerennialCropCycleDto.setCroppingPlanEntryCode(croppingPlanEntryCode);

            List<CropCyclePhaseDto> cropCyclePhaseDtos = convertCropCyclePhases(practicedPerennialCropCycle);
            practicedPerennialCropCycleDto.setCropCyclePhaseDtos(cropCyclePhaseDtos);

            List<CropCyclePerennialSpeciesDto> cropCyclePerennialSpeciesDtos = practicedCropCycleService.getCropCyclePerennialSpecies(croppingPlanEntryCode,
                    practicedPerennialCropCycle, practicedSystem.getCampaigns());
            practicedPerennialCropCycleDto.setCropCyclePerennialSpeciesDto(cropCyclePerennialSpeciesDtos);

            practicedPerennialCropCycleDtos.add(practicedPerennialCropCycleDto);
        }
    }

    /**
     * Convert database seasonal entity to dto.
     */
    protected void practicedSeasonalCyclesInput(List<PracticedSeasonalCropCycle> practicedSeasonalCropCycles) {

        for (PracticedSeasonalCropCycle practicedSeasonalCropCycle : practicedSeasonalCropCycles) {
            PracticedSeasonalCropCycleDto dto = new PracticedSeasonalCropCycleDto();
            dto.setTopiaId(practicedSeasonalCropCycle.getTopiaId());

            // graph data for each cycle
            List<CropCycleNodeDto> nodes = Lists.newArrayList();
            Map<CropCycleNodeDto, List<CroppingPlanSpeciesDto>> nodesToSpecies = practicedCropCycleService.getNodes(practicedSeasonalCropCycle.getTopiaId());
            for (Map.Entry<CropCycleNodeDto, List<CroppingPlanSpeciesDto>> entry : nodesToSpecies.entrySet()) {
                CropCycleNodeDto cycleNodeDto = entry.getKey();
                nodes.add(cycleNodeDto);
            }
            dto.setCropCycleNodeDtos(nodes);
            List<CropCycleConnectionDto> connections = practicedCropCycleService.getConnections(practicedSeasonalCropCycle.getTopiaId());
            dto.setCropCycleConnectionDtos(connections);

            // valuation du label des connexions
            if (connections != null && practicedSystemIntermediateCropCycleModels != null) {
                for (CropCycleConnectionDto connection : connections) {
                    final String expectedCode = connection.getIntermediateCroppingPlanEntryCode();
                    if (!Strings.isNullOrEmpty(expectedCode)) {
                        Optional<CropCycleModelDto> optional = Iterables.tryFind(practicedSystemIntermediateCropCycleModels, new Predicate<CropCycleModelDto>() {
                            @Override
                            public boolean apply(CropCycleModelDto input) {
                                return expectedCode.equals(input.getCroppingPlanEntryCode());
                            }
                        });
                        if (optional.isPresent()) {
                            connection.setLabel(optional.get().getLabel());
                        }
                    }
                }
            }

            // add dto
            practicedSeasonalCropCycleDtos.add(dto);
        }
    }

    @Override
    protected void initForInput() {
        if (getPracticedSystem().isPersisted()) {
            // warning, practicedSystem's growingSystem must always be part of growingSystems set
            // even not selected be navigation context
            GrowingSystem growingSystem = getPracticedSystem().getGrowingSystem();
            growingSystemId = growingSystem.getTopiaId();
            growingSystems = Collections.singletonList(growingSystem);
            campaigns = practicedSystem.getCampaigns();
        } else {
            GrowingSystemFilter growingSystemFilter = new GrowingSystemFilter();
            NavigationContext navigationContext = getNavigationContext();
            growingSystemFilter.setNavigationContext(navigationContext);
            growingSystemFilter.setActive(Boolean.TRUE);
            growingSystemFilter.setPageSize(GrowingSystemFilter.ALL_PAGE_SIZE);
            ResultList<GrowingSystem> resultList = growingSystemService.getFilteredGrowingSystems(growingSystemFilter);
            growingSystems = resultList.getElements();
        }
        
        // chargement des référentiels
        refOrientationEDIs = referentielService.findAllReferentielEDI();
        refActionAgrosystTravailEDIs = referentielService.getRefActionAgrosystTravailEDIs();

        allActiveFertiMinProductType = referentielService.findAllActiveFertiMinProductType();

        refUnitesEDIs = referentielService.findAllActiveRefUnitesEDI();

        // initialisation des données chargés en json autrement sur la page
        if (!Strings.isNullOrEmpty(growingSystemId) && !Strings.isNullOrEmpty(campaigns)) {
            // chargement des toutes les cultures pour les systems de cultures et les années
            Map<CropCycleModelDto, List<CroppingPlanSpeciesDto>> modelToSpecies =
                    practicedSystemService.getCropCycleModelMap(growingSystemId, campaigns, true);
            Set<CropCycleModelDto> croppingPlanEntryDtos = modelToSpecies.keySet();

            // definition de la liste de culture principale
            Iterable<CropCycleModelDto> modelMain = Iterables.filter(croppingPlanEntryDtos, CroppingPlans.IS_NOT_INTERMEDIATE);
            practicedSystemMainCropCycleModels = Lists.newArrayList(modelMain); // force no lazy

            // définition de la liste de culture intermédiaire
            Iterable<CropCycleModelDto> modelIntermediate = Iterables.filter(croppingPlanEntryDtos, CroppingPlans.IS_INTERMEDIATE);
            practicedSystemIntermediateCropCycleModels = Lists.newArrayList(modelIntermediate); // force no lazy

            // chargement de la map 'code culture' > liste d'espece
            practicedSystemCroppingEntryPlanToSpecies = Maps.newHashMap();
            for (Map.Entry<CropCycleModelDto, List<CroppingPlanSpeciesDto>> entry : modelToSpecies.entrySet()) {
                practicedSystemCroppingEntryPlanToSpecies.put(entry.getKey().getCroppingPlanEntryCode(), entry.getValue());
            }

            // chargement des toolsCouplings
            List<ToolsCouplingDto> toolsCouplingsDtos = practicedSystemService.getToolsCouplingModel(growingSystemId, campaigns);
            practicedSystemToolsCouplings = Lists.newArrayList(toolsCouplingsDtos); // force no lazy
        }
    }

    @Override
    public void validate() {
        if (StringUtils.isBlank(getPracticedSystem().getTopiaId())) {
            if (StringUtils.isBlank(growingSystemId)) {
                addFieldError("growingSystemId", "Le système de culture est requis !");
            }
        }

        if (StringUtils.isBlank(practicedSystem.getName())) {
            addFieldError("practicedSystem.name", "Le nom du système pratiqué est requis !");
        }
        String campaigns = practicedSystem.getCampaigns();
        if (StringUtils.isBlank(campaigns)) {
            addFieldError("practicedSystem.campaigns", "La série de campagne est obligatoire");
        }
        if (StringUtils.isNotBlank(campaigns)) {
            if (!practicedSystemService.areCampaignsValid(campaigns)) {
                String format = "La série de campagne doit être composée d'années séparées par des virgules (,), " +
                        "espaces ( ) ou point-virgules (;). Les campagnes doivent être contenues entre %d et %d.";
                Pair<Integer,Integer> bounds = practicedSystemService.getCampaignsBounds();
                addFieldError("practicedSystem.campaigns", String.format(format, bounds.getLeft(), bounds.getRight()));
            }
        }

        if (practicedSeasonalCropCycleDtos != null) {
            practicedSeasonalCropCycleValidate();
        }
        if (practicedPerennialCropCycleDtos != null){
            practicedPerennialCycleValidate();
        }
        
        if (hasErrors()) {
            initForInput();
        }
    }

    protected void practicedSeasonalCropCycleValidate() {

        for (PracticedSeasonalCropCycleDto practicedSeasonalCropCycleDto : practicedSeasonalCropCycleDtos) {
            List<CropCycleConnectionDto> cropCycleConnectionDtos = practicedSeasonalCropCycleDto.getCropCycleConnectionDtos();
            if (cropCycleConnectionDtos != null) {
                for (CropCycleConnectionDto cropCycleConnectionDto: cropCycleConnectionDtos) {
                    List<InterventionDto> interventionDtos = cropCycleConnectionDto.getInterventions();
                    if (interventionDtos != null) {

                        for (InterventionDto interventionDto : interventionDtos) {
                            if (Strings.isNullOrEmpty(interventionDto.getIntervention().getName())) {
                                addFieldError("intervention_name", "Le nom est obligatoire");
                            }
                            if (interventionDto.getIntervention().getInterventionType() == null) {
                                addActionError("Le type d'intervention est obligatoire");
                            }
                            List<ActionDto> actionDtos = interventionDto.getActionDtos();
                            if (actionDtos == null || actionDtos.size() == 0) {
                                addActionError("Une intervention a au moins une action");
                            } else {
                                for (ActionDto actionDto : actionDtos) {
                                    if (StringUtils.isBlank(actionDto.getRefActionAgrosystTravailEdiId())) {
                                        addActionError("Le type de travail EDI est obligatoire sur une action");
                                    }
                                    if (actionDto.getActionType()==null) {
                                        addActionError("le type d'action est obligatoire sur une action");
                                    }
                                }
                            }
                        }

                    }
                }
            }
        }
    }

    protected void practicedPerennialCycleValidate() {
        // validate perennial cycle
        for (PracticedPerennialCropCycleDto practicedPerennialCropCycleDto : practicedPerennialCropCycleDtos){
            if (Strings.isNullOrEmpty(practicedPerennialCropCycleDto.getCroppingPlanEntryCode())) {
                addFieldError("croppingPlanEntry", "Champ obligatoire");
            }

            List<CropCyclePhaseDto> cropCyclePhaseDtos = practicedPerennialCropCycleDto.getCropCyclePhaseDtos();
            if (cropCyclePhaseDtos == null) {
                addActionError("Un cycle pluriannuel doit avoir une phase de pleine production");
            } else {
                int instCount = 0, montCount = 0, pleiCount = 0, declCount = 0;
                for (CropCyclePhaseDto cropCyclePhaseDto : cropCyclePhaseDtos) {
                    if (cropCyclePhaseDto.getCropCyclePhaseType() != null) {
                        switch (cropCyclePhaseDto.getCropCyclePhaseType()) {
                            case INSTALLATION:
                                instCount++;
                                break;
                            case MONTEE_EN_PRODUCTION:
                                montCount++;
                                break;
                            case PLEINE_PRODUCTION:
                                pleiCount++;
                                break;
                            case DECLIN_DE_PRODUCTION:
                                declCount++;
                                break;
                            default:
                                break;
                        }

                    }
                    List<InterventionDto> interventionDtos = cropCyclePhaseDto.getInterventions();
                    if (interventionDtos != null) {
                        for (InterventionDto interventionDto : interventionDtos) {
                            if (Strings.isNullOrEmpty(interventionDto.getIntervention().getName())) {
                                addFieldError("intervention_name", "Le nom est obligatoire");
                            }
                            if (interventionDto.getIntervention().getInterventionType() == null) {
                                addActionError("Le type d'intervention est obligatoire");
                            }
                            List<ActionDto> actionDtos = interventionDto.getActionDtos();
                            if (actionDtos == null || actionDtos.size() == 0) {
                                addActionError("Une intervention a au moins une action");
                            } else {
                                for (ActionDto actionDto : actionDtos) {
                                    if (StringUtils.isBlank(actionDto.getRefActionAgrosystTravailEdiId())) {
                                        addActionError("Le type de travail EDI est obligatoire sur une action");
                                    }
                                    if (actionDto.getActionType()==null) {
                                        addActionError("le type d'action est obligatoire sur une action");
                                    }
                                }
                            }
                        }
                    }
                }
                if (instCount > 1) {
                    addActionError("Un cycle pluriannuel ne peut avoir qu'une seul phase d'installation");
                }
                if (montCount > 1) {
                    addActionError("Un cycle pluriannuel ne peut avoir qu'une seul phase de montée en production");
                }
                if (pleiCount == 0) {
                    addActionError("Un cycle pluriannuel doit avoir une phase de pleine production");
                }
                else if (pleiCount > 1) {
                    addActionError("Un cycle pluriannuel ne peut avoir qu'une seul phase de pleine production");
                }
                if (declCount > 1) {
                    addActionError("Un cycle pluriannuel ne peut avoir qu'une seul phase de déclin de production");
                }
            }
        }
        // / validate perennial
        
        // validate 
    }

    @Action(results = {@Result(type = "redirectAction", params = {"actionName", "practiced-systems-edit-input", "practicedSystemTopiaId", "${practicedSystem.topiaId}"})})
    @Override
    public String execute() throws Exception {
        PracticedSystem practicedSystem = getPracticedSystem();
        // can define domain only during create action
        if (StringUtils.isBlank(practicedSystem.getTopiaId())) {
            GrowingSystem growingSystem = growingSystemService.getGrowingSystem(growingSystemId);
            getPracticedSystem().setGrowingSystem(growingSystem);
        }

        practicedSystem = practicedSystemService.createOrUpdatePracticedSystem(practicedSystem, practicedPerennialCropCycleDtos,
                practicedSeasonalCropCycleDtos);

        notificationSupport.practicedSystemSaved(practicedSystem);

        return SUCCESS;
    }

    protected List<CropCyclePhaseDto> convertCropCyclePhases(PracticedPerennialCropCycle cycle) {
        List<CropCyclePhase> currentPhases = cycle.getCropCyclePhases();
        List<CropCyclePhaseDto> currentPhasesDtos = Lists.newArrayListWithCapacity(currentPhases.size());
        for (CropCyclePhase currentPhase : currentPhases) {
            CropCyclePhaseDto currentPhaseDto = new CropCyclePhaseDto();
            currentPhaseDto.setCropCyclePhaseType(currentPhase.getCropCyclePhaseType());
            currentPhaseDto.setDuration(currentPhase.getDuration());
            currentPhaseDto.setTopiaId(currentPhase.getTopiaId());
            List<Intervention> interventions = practicedCropCycleService.getCropCyclePhaseInterventions(currentPhase);
            if (interventions != null) {
                List<InterventionDto> interventionDtos = Lists.newArrayList(Iterables.transform(interventions, PracticedSystems.INTERVENTION_TO_INTERVENTION_DTO));
                currentPhaseDto.setInterventions(interventionDtos);
            }
            currentPhasesDtos.add(currentPhaseDto);

        }
        return currentPhasesDtos;
    }

    public List<GrowingSystem> getGrowingSystems() {
        return growingSystems;
    }
    
    public String getGrowingSystemsJson() {
        return getGson().toJson(growingSystems);
    }

    public void setPracticedSystemTopiaId(String practicedSystemTopiaId) {
        this.practicedSystemId = practicedSystemTopiaId;
    }

    public String getGrowingSystemId() {
        return growingSystemId;
    }

    public void setGrowingSystemId(String growingSystemId) {
        this.growingSystemId = growingSystemId;
    }

    public Map<PracticedSystemSource, String> getSources() {
        return getEnumAsMap(PracticedSystemSource.values());
    }
    
    public List<RefOrientationEDI> getRefOrientationEDIs() {
        return refOrientationEDIs;
    }

    public void setPracticedSystemId(String practicedSystemId) {
        this.practicedSystemId = practicedSystemId;
    }

    public String getPracticedSystemId() {
        return practicedSystemId;
    }

    public Map<VineFrutalForm, String> getVineFrutalForms() {
        return getEnumAsMap(VineFrutalForm.values());
    }

    public Map<OrchardFrutalForm, String> getOrchardFrutalForms() {
        return getEnumAsMap(OrchardFrutalForm.values());
    }

    public Map<PollinatorSpreadMode, String> getPollinatorSpreadModes() {
        return getEnumAsMap(PollinatorSpreadMode.values());
    }

    public Map<CropCyclePhaseType, String> getCropCyclePhaseTypes() {
        return getEnumAsMap(CropCyclePhaseType.values());
    }
    
    public Map<WeedType, String> getWeedTypes() {
        return getEnumAsMap(WeedType.values());
    }

    public Map<FertilizerQuantityUnit, String> getFertilizerQuantityUnit() {
        return getEnumAsMap(FertilizerQuantityUnit.values());
    }

    public String getCampaigns() {
        String result = "";
        if (practicedSystem != null) {
            result = practicedSystem.getCampaigns();
        }
        return getGson().toJson(result);
    }
    
    public String getGrowingSystemIdJson(){
        return getGson().toJson(growingSystemId);
    }
    
    public Map<ActionType, String> getActionTypes(){
        return getEnumAsMap(ActionType.values());
    }

    public Map<SeedType, String> getSeedTypes() {
        return getEnumAsMap(SeedType.values());
    }

    public Map<UnitPlantSeeds, String> getUnitPlantSeeds() {
        return getEnumAsMap(UnitPlantSeeds.values());
    }

    public Map<YealdUnit, String>  getYealdUnit() {
        return getEnumAsMap(YealdUnit.values());
    }

    public Map<InterventionType, String> getInterventionType() {
        return getEnumAsMap(InterventionType.values());
    }

    public void setPracticedPerennialCropCycleDtosJson(String json) {
        Type type = new TypeToken<List<PracticedPerennialCropCycleDto>>() {}.getType();
        this.practicedPerennialCropCycleDtos = getGson().fromJson(json, type);
    }

    public void setPracticedSeasonalCropCycleDtosJson(String json) {
        Type type = new TypeToken<List<PracticedSeasonalCropCycleDto>>() {}.getType();
        this.practicedSeasonalCropCycleDtos = getGson().fromJson(json, type);
    }

    public List<PracticedPerennialCropCycleDto> getPracticedPerennialCropCycleDtos() {
        return practicedPerennialCropCycleDtos;
    }

    public List<RefActionAgrosystTravailEDI> getRefActionAgrosystTravailEDIs(){
        return refActionAgrosystTravailEDIs;
    }

    public List<PracticedSeasonalCropCycleDto> getPracticedSeasonalCropCycleDtos() {
        return practicedSeasonalCropCycleDtos;
    }

    public List<ToolsCouplingDto> getPracticedSystemToolsCouplings() {
        return practicedSystemToolsCouplings;
    }

    public Map<String, List<CroppingPlanSpeciesDto>> getPracticedSystemCroppingEntryPlanToSpecies() {
        return practicedSystemCroppingEntryPlanToSpecies;
    }

    public List<CropCycleModelDto> getPracticedSystemMainCropCycleModels() {
        return practicedSystemMainCropCycleModels;
    }

    public List<CropCycleModelDto> getPracticedSystemIntermediateCropCycleModels() {
        return practicedSystemIntermediateCropCycleModels;
    }

    public Map<YealdCategory, String> getYealdCategorys(){
        return getEnumAsMap(YealdCategory.values());
    }

    public Map<CapacityUnit, String> getCapacityUnits(){
        return getEnumAsMap(CapacityUnit.values());
    }

    public Map<Integer, String> getAllActiveFertiMinProductType() {
        return allActiveFertiMinProductType;
    }

    public List<RefUnitesEDI> getRefUnitesEDIs() {
        return refUnitesEDIs;
    }

}
