/*
 * #%L
 * Agrosyst :: Web
 * $Id: GrowingSystemsEdit.java 1916 2013-10-28 16:52:52Z athimel $
 * $HeadURL: https://forge.codelutin.com/svn/agrosyst/tags/agrosyst-0.6.1/agrosyst-web/src/main/java/fr/inra/agrosyst/web/actions/growingsystems/GrowingSystemsEdit.java $
 * %%
 * Copyright (C) 2013 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */
package fr.inra.agrosyst.web.actions.growingsystems;

import java.lang.reflect.Type;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.apache.struts2.convention.annotation.Action;
import org.apache.struts2.convention.annotation.Result;

import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.gson.reflect.TypeToken;
import com.opensymphony.xwork2.Preparable;

import fr.inra.agrosyst.api.NavigationContext;
import fr.inra.agrosyst.api.entities.GrowingPlan;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.GrowingSystemImpl;
import fr.inra.agrosyst.api.entities.Plot;
import fr.inra.agrosyst.api.entities.Sector;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.growingplan.GrowingPlanFilter;
import fr.inra.agrosyst.api.services.growingplan.GrowingPlanService;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemService;
import fr.inra.agrosyst.api.services.network.NetworkService;
import fr.inra.agrosyst.api.services.plot.PlotService;
import fr.inra.agrosyst.web.actions.AbstractAgrosystAction;
import fr.inra.agrosyst.web.actions.AbstractJsonAction;

/**
 * Action d'edition d'un système de culture.
 * 
 * @author Eric Chatellier
 */
public class GrowingSystemsEdit extends AbstractAgrosystAction implements Preparable {

    private static final long serialVersionUID = -7030514291314044857L;

    protected GrowingSystemService growingSystemService;

    protected String growingSystemTopiaId;

    /** Edited instance. */
    protected GrowingSystem growingSystem;

    /** To display timeline. */
    protected LinkedHashMap<Integer, String> relatedGrowingSystems;

    protected PlotService plotService;
    
    protected NetworkService networkService;

    protected List<Plot> availablePlots;
    
    protected GrowingPlanService growingPlanService;

    protected Set<GrowingPlan> growingPlans;

    protected GrowingPlan growingPlan;
    
    protected String growingPlanTopiaId;
    
    protected List<String> selectedPlotsIds;
    
    protected List<String> growingSystemNetworkIds;

    protected static final String REQUIRED_FIELD = "Champ obligatoire";
    protected static final String PERCENT_FIELD = "La valeur doit être comprise entre 0 et 100";
    protected static final String INCOHERENT_STARTING_AND_ENDING_DATES = "Les dates de début et de fin sont incohérentes";

    public void setGrowingSystemService(GrowingSystemService growingSystemService) {
        this.growingSystemService = growingSystemService;
    }

    public void setPlotService(PlotService plotService) {
        this.plotService = plotService;
    }

    public void setGrowingPlanService(GrowingPlanService growingPlanService) {
        this.growingPlanService = growingPlanService;
    }

    public void setNetworkService(NetworkService networkService) {
        this.networkService = networkService;
    }

    @Override
    public void prepare() throws Exception {

        if (StringUtils.isEmpty(growingSystemTopiaId)) {
            // Cas de création d'un growingSystem
            growingSystem = growingSystemService.newGrowingSystem();
        } else {
            // TODO AThimel 07/10/13 May be included directly in the service ?
            authorizationService.checkGrowingSystemReadable(growingSystemTopiaId);

            // Cas d'une mise à jour de growingSystem
            readOnly = !authorizationService.isGrowingSystemWritable(growingSystemTopiaId);
            if (readOnly) {
                notificationSupport.growingSystemNotWritable();
            }

            growingSystem = growingSystemService.getGrowingSystem(growingSystemTopiaId);
            growingPlanTopiaId = growingSystem.getGrowingPlan().getTopiaId();
        }
    }

    public GrowingSystem getGrowingSystem() {
        if (growingSystem == null) {
            // AThimel 26/06/2013 Fais chier de devoir écrire ça, mais c'est la seule option pour ne pas avoir une grosse dose d'exceptions avec du paramsPrepareParams
            GrowingSystem growingSystem = new GrowingSystemImpl();
            return growingSystem;
        }
        return growingSystem;
    }

    @Override
    @Action("growing-systems-edit-input")
    public String input() throws Exception {

        GrowingSystem growingSystem = getGrowingSystem();
        if (growingSystem.isPersisted()) {
            growingPlanTopiaId = growingSystem.getGrowingPlan().getTopiaId();
        }

        initForInput();

        // initialise la saisie des checkbox de selection
        if (growingSystem.isPersisted()) {
            selectedPlotsIds = Lists.newArrayList();
            for (Plot plot : availablePlots) {
                if (growingSystem.equals(plot.getGrowingSystem())) {
                    selectedPlotsIds.add(plot.getTopiaId());
                }
            }
        }

        return INPUT;
    }

    @Override
    protected void initForInput() {
        // warning, growingSystem's must always be part of growingPlan set
        // even not selected be navigation context
        GrowingSystem growingSystem = getGrowingSystem();
        if (growingSystem.isPersisted()) {
            // select combo box
            growingPlans = Collections.singleton(growingSystem.getGrowingPlan());
            // timeline
            relatedGrowingSystems = growingSystemService.getRelatedGrowingSystems(growingSystem.getCode());
            // plots tab
            availablePlots = plotService.findAllFreeAndGrowingSystemPlots(growingSystem, growingPlanTopiaId);
        } else {
            // select combo box
            GrowingPlanFilter growingPlanFilter = new GrowingPlanFilter();
            NavigationContext navigationContext = getNavigationContext();
            growingPlanFilter.setNavigationContext(navigationContext);
            growingPlanFilter.setActive(true);
            growingPlanFilter.setPageSize(GrowingPlanFilter.ALL_PAGE_SIZE);
            ResultList<GrowingPlan> growingPlanListResult = growingPlanService.getFilteredGrowingPlans(growingPlanFilter);
            growingPlans = Sets.newLinkedHashSet(growingPlanListResult.getElements());

            // plots tab
            if (growingPlanTopiaId != null) {
                availablePlots = plotService.findAllFreeAndGrowingSystemPlots(null, growingPlanTopiaId);
            }
        }
    }

    @Override
    public void validate() {
        // Valid if the growing system :
        //  is attached to a growingPlan
        if (StringUtils.isBlank(growingPlanTopiaId)) {
            addFieldError("growingPlanTopiaId", REQUIRED_FIELD);
        }
        //  has a name
        if (StringUtils.isBlank(growingSystem.getName())) {
            addFieldError("growingSystem.name", REQUIRED_FIELD);
        }
        //  has startingDate
        Date startingDate = growingSystem.getStartingDate();
        if (startingDate == null) {
            addFieldError("growingSystem.startingDate", REQUIRED_FIELD);
        }

        // Filière
        if (growingSystem.getSector() == null) {
            addFieldError("growingSystem.sector", REQUIRED_FIELD);
        }

        //  has endingDate
        Date endingDate = growingSystem.getEndingDate();
        if (endingDate != null) {
            if (endingDate.before(startingDate)) {
                addFieldError("growingSystem.startingDate", INCOHERENT_STARTING_AND_ENDING_DATES);
                addFieldError("growingSystem.endingDate", INCOHERENT_STARTING_AND_ENDING_DATES);
            }
        }
        // affectedAreaRate
        Integer affectedAreaRate = growingSystem.getAffectedAreaRate();
        if (affectedAreaRate != null) {
            if (affectedAreaRate < 0 || affectedAreaRate > 100) {
                addFieldError("growingSystem.affectedAreaRate", PERCENT_FIELD);

            }
        }
        // affectedWorkForceRate
        Integer affectedWorkForceRate = growingSystem.getAffectedWorkForceRate();
        if (affectedWorkForceRate != null) {
            if (affectedWorkForceRate < 0 || affectedWorkForceRate > 100) {
                addFieldError("growingSystem.affectedWorkForceRate", PERCENT_FIELD);

            }
        }
        // domainsToolsUsageRate
        Integer domainsToolsUsageRate = growingSystem.getDomainsToolsUsageRate();
        if (domainsToolsUsageRate != null) {
            if (domainsToolsUsageRate < 0 || domainsToolsUsageRate > 100) {
                addFieldError("growingSystem.domainsToolsUsageRate", PERCENT_FIELD);

            }
        }
        
        if (hasErrors()) {
            initForInput();
        }
    }

    @Override
    @Action(results = {@Result(type = "redirectAction", params = {"actionName", "growing-systems-edit-input", "growingSystemTopiaId", "${growingSystem.topiaId}"})})
    public String execute() throws Exception {
        if (StringUtils.isBlank(getGrowingSystem().getTopiaId())) {
            GrowingPlan growingPlan = growingPlanService.getGrowingPlan(growingPlanTopiaId);
            growingSystem.setGrowingPlan(growingPlan);
        }

        growingSystem = growingSystemService.createOrUpdateGrowingSystem(growingSystem, growingSystemNetworkIds, selectedPlotsIds);
        notificationSupport.growingSystemSaved(growingSystem);

        if (Strings.isNullOrEmpty(growingSystemTopiaId)) {
            navigationContextEntityCreated(growingSystem);
        }

        return SUCCESS;
    }

    public String getStartingDateFormatted() {
        Date startingDate = growingSystem.getStartingDate();
        if (startingDate == null) {
            return "";
        }
        return new SimpleDateFormat("dd/MM/yyyy").format(startingDate); // TODO AThimel 27/06/13 Better way ?
    }

    public String getEndingDateFormatted() {
        String result = null;
        Date endingDate = growingSystem.getEndingDate();
        if (endingDate != null) {
            result = new SimpleDateFormat("dd/MM/yyyy").format(endingDate); // TODO AThimel 27/06/13 Better way ?
        }
        return result;
    }

    public void setGrowingSystem(GrowingSystem growingSystem) {
        this.growingSystem = growingSystem;
    }

    public  Map<String, Boolean> getSelectedPlotsIds() {
        Map<String, Boolean> result;
        if (selectedPlotsIds == null) {
            result = Maps.newHashMap();
        } else {
            result = Maps.toMap(selectedPlotsIds, AbstractJsonAction.GET_TRUE);
        }
        return result;
    }

    public List<Plot> getAvailablePlots() {
        return availablePlots;
    }

    public String getGrowingSystemTopiaId() {
        return growingSystemTopiaId;
    }

    public void setGrowingSystemTopiaId(String growingSystemTopiaId) {
        this.growingSystemTopiaId = growingSystemTopiaId;
    }

    public String getGrowingPlanTopiaId() {
        return growingPlanTopiaId;
    }

    public void setGrowingPlanTopiaId(String growingPlanTopiaId) {
        this.growingPlanTopiaId = growingPlanTopiaId;
    }

    public Collection<GrowingPlan> getGrowingPlans() {
        return growingPlans;
    }

    public void setSelectedPlotsIds(List<String> selectedPlotsIds) {
        this.selectedPlotsIds = selectedPlotsIds;
    }

    public Map<Sector, String> getSectors() {
        return getEnumAsMap(Sector.values());
    }

    public LinkedHashMap<Integer, String> getRelatedGrowingSystems() {
        return relatedGrowingSystems;
    }
    
    public void setGrowingSystemNetworkIdsJson(String json) {
        Type type = new TypeToken<List<String>>() {}.getType();
        this.growingSystemNetworkIds = getGson().fromJson(json, type);
    }
}
