package fr.inra.agrosyst.web.actions.effective;

/*
 * #%L
 * Agrosyst :: Web
 * $Id: EffectiveCropCyclesEdit.java 1897 2013-10-28 10:18:31Z echatellier $
 * $HeadURL: https://forge.codelutin.com/svn/agrosyst/tags/agrosyst-0.6.1/agrosyst-web/src/main/java/fr/inra/agrosyst/web/actions/effective/EffectiveCropCyclesEdit.java $
 * %%
 * Copyright (C) 2013 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.lang.reflect.Type;
import java.util.List;
import java.util.Map;

import org.apache.struts2.convention.annotation.Action;
import org.apache.struts2.convention.annotation.Result;

import com.google.common.collect.Lists;
import com.google.gson.reflect.TypeToken;
import com.opensymphony.xwork2.Preparable;

import fr.inra.agrosyst.api.entities.CropCyclePhaseType;
import fr.inra.agrosyst.api.entities.CroppingPlanEntry;
import fr.inra.agrosyst.api.entities.InterventionType;
import fr.inra.agrosyst.api.entities.OrchardFrutalForm;
import fr.inra.agrosyst.api.entities.PollinatorSpreadMode;
import fr.inra.agrosyst.api.entities.ToolsCoupling;
import fr.inra.agrosyst.api.entities.VineFrutalForm;
import fr.inra.agrosyst.api.entities.WeedType;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.entities.referentiels.RefOrientationEDI;
import fr.inra.agrosyst.api.services.effective.EffectiveCropCycleModelDto;
import fr.inra.agrosyst.api.services.effective.EffectiveCropCycleService;
import fr.inra.agrosyst.api.services.effective.EffectiveCropCycles;
import fr.inra.agrosyst.api.services.effective.EffectivePerennialCropCycleDto;
import fr.inra.agrosyst.api.services.effective.EffectiveSeasonalCropCycleDto;
import fr.inra.agrosyst.api.services.referentiels.ReferentielService;
import fr.inra.agrosyst.web.actions.AbstractAgrosystAction;

public class EffectiveCropCyclesEdit extends AbstractAgrosystAction implements Preparable {

    /** serialVersionUID. */
    private static final long serialVersionUID = -1526208592874558224L;

    protected EffectiveCropCycleService effectiveCropCycleService;

    protected ReferentielService referentielService;

    /** Topia id de la zone à laquelle sont liés les cycles. */
    protected String zoneTopiaId;

    /** Zone à laquelle sont liés les cycles. */
    protected Zone zone;

    /** Cycles assolés liés à la parcelle. */
    protected List<EffectiveSeasonalCropCycleDto> effectiveSeasonalCropCycles;

    /** Cycles pérennes lié à la parcelle. */
    protected List<EffectivePerennialCropCycleDto> effectivePerennialCropCycles;

    /** Cultures associées à la zone. */
    protected List<CroppingPlanEntry> croppingPlanEntries;

    /** Main cropping plan model entries. */
    protected List<EffectiveCropCycleModelDto> effectiveCropCyclesMainModels;

    /** Intermediate cropping plan model entries. */
    protected List<EffectiveCropCycleModelDto> effectiveCropCyclesIntermediateModels;

    /** Orientation EDI. */
    protected List<RefOrientationEDI> refOrientationEDIs;

    protected List<ToolsCoupling> toolsCouplings;

    public void setEffectiveCropCycleService(EffectiveCropCycleService effectiveCropCycleService) {
        this.effectiveCropCycleService = effectiveCropCycleService;
    }

    public void setReferentielService(ReferentielService referentielService) {
        this.referentielService = referentielService;
    }

    @Override
    public void prepare() throws Exception {
        zone = effectiveCropCycleService.getZone(zoneTopiaId);
    }

    @Override
    @Action("effective-crop-cycles-edit-input")
    public String input() throws Exception {

        authorizationService.checkEffectiveCropCyclesReadable(zoneTopiaId);
        readOnly = !authorizationService.areEffectiveCropCyclesWritable(zoneTopiaId);
        if (readOnly) {
            notificationSupport.effectiveCropCyclesNotWritable();
        }

        effectivePerennialCropCycles = effectiveCropCycleService.getPlotEffectivePerennialCropCycles(zone);
        effectiveSeasonalCropCycles = effectiveCropCycleService.getPlotEffectiveSeasonnalCropCycles(zone);

        initForInput();
        return INPUT;
    }

    @Override
    protected void initForInput() {

        croppingPlanEntries = effectiveCropCycleService.getZoneCroppingPlanEntries(zone);

        // build main and intermediate cropping plan entri map
        effectiveCropCyclesMainModels = Lists.newArrayList();
        effectiveCropCyclesIntermediateModels = Lists.newArrayList();
        for (CroppingPlanEntry croppingPlanEntry : croppingPlanEntries) {
            EffectiveCropCycleModelDto effectiveCropCycleModelDto = EffectiveCropCycles.CROPPING_PLAN_ENTRY_TO_DTO.apply(croppingPlanEntry);
            if (effectiveCropCycleModelDto.isIntermediate()) {
                effectiveCropCyclesIntermediateModels.add(effectiveCropCycleModelDto);
            } else {
                effectiveCropCyclesMainModels.add(effectiveCropCycleModelDto);
            }
        }

        refOrientationEDIs = referentielService.findAllReferentielEDI();
        toolsCouplings = zone.getPlot().getDomain().getToolsCoupling();
    }

    @Override
    public void validate() {
        if (zone == null) {
            addFieldError("zoneTopiaId", "Zone can't be null");
        }
        if (effectiveSeasonalCropCycles == null) {
            addFieldError("effectiveSeasonalCropCycles", "Seasonal crop cycles list can't be null");
        }
        if (effectivePerennialCropCycles == null) {
            addFieldError("effectivePerennialCropCycles", "Perennial crop cycles list can't be null");
        }
        if (hasErrors()) {
            initForInput();
        }
    }

    @Action(results = {@Result(type = "redirectAction", params = {"actionName", "effective-crop-cycles-edit-input", "zoneTopiaId", "${zoneTopiaId}"})})
    @Override
    public String execute() throws Exception {
        effectiveCropCycleService.udpateEffectiveCropCycles(zoneTopiaId, effectiveSeasonalCropCycles, effectivePerennialCropCycles);
        notificationSupport.culturalInterventionCreated();
        return SUCCESS;
    }
    
    public String getZoneTopiaId() {
        return zoneTopiaId;
    }

    public void setZoneTopiaId(String plotTopiaId) {
        this.zoneTopiaId = plotTopiaId;
    }

    public Zone getZone() {
        return zone;
    }

    public void setEffectiveSeasonalCropCycles(String json) {
        Type type = new TypeToken<List<EffectiveSeasonalCropCycleDto>>() {}.getType();
        this.effectiveSeasonalCropCycles = getGson().fromJson(json, type);
    }

    public void setEffectivePerennialCropCycles(String json) {
        Type type = new TypeToken<List<EffectivePerennialCropCycleDto>>() {}.getType();
        this.effectivePerennialCropCycles = getGson().fromJson(json, type);
    }

    public List<EffectiveSeasonalCropCycleDto> getEffectiveSeasonalCropCycles() {
        return effectiveSeasonalCropCycles;
    }

    public List<EffectivePerennialCropCycleDto> getEffectivePerennialCropCycles() {
        return effectivePerennialCropCycles;
    }

    public List<EffectiveCropCycleModelDto> getEffectiveCropCyclesMainModels() {
        return effectiveCropCyclesMainModels;
    }

    public List<EffectiveCropCycleModelDto> getEffectiveCropCyclesIntermediateModels() {
        return effectiveCropCyclesIntermediateModels;
    }

    public List<CroppingPlanEntry> getCroppingPlanEntries() {
        return croppingPlanEntries;
    }

    public List<RefOrientationEDI> getRefOrientationEDIs() {
        return refOrientationEDIs;
    }

    public List<ToolsCoupling> getToolsCouplings() {
        return toolsCouplings;
    }

    public Map<VineFrutalForm, String> getVineFrutalForms() {
        return getEnumAsMap(VineFrutalForm.values());
    }

    public Map<OrchardFrutalForm, String> getOrchardFrutalForms() {
        return getEnumAsMap(OrchardFrutalForm.values());
    }

    public Map<PollinatorSpreadMode, String> getPollinatorSpreadModes() {
        return getEnumAsMap(PollinatorSpreadMode.values());
    }

    public Map<CropCyclePhaseType, String> getCropCyclePhaseTypes() {
        return getEnumAsMap(CropCyclePhaseType.values());
    }
    
    public Map<WeedType, String> getWeedTypes() {
        return getEnumAsMap(WeedType.values());
    }
    
    public Map<InterventionType, String> getInterventionTypes() {
        return getEnumAsMap(InterventionType.values());
    }
}
