package fr.inra.agrosyst.web.actions.commons;

/*
 * #%L
 * Agrosyst :: Web
 * $Id: AttachmentsRaw.java 1905 2013-10-28 13:48:31Z athimel $
 * $HeadURL: https://forge.codelutin.com/svn/agrosyst/tags/agrosyst-0.6.1/agrosyst-web/src/main/java/fr/inra/agrosyst/web/actions/commons/AttachmentsRaw.java $
 * %%
 * Copyright (C) 2013 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.sql.SQLException;
import java.util.List;

import javax.servlet.ServletOutputStream;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts2.convention.annotation.Action;
import org.apache.struts2.convention.annotation.Result;

import com.google.common.base.Charsets;
import com.google.common.collect.Lists;
import com.opensymphony.xwork2.ActionContext;
import com.opensymphony.xwork2.Preparable;

import fr.inra.agrosyst.api.entities.AttachmentContent;
import fr.inra.agrosyst.api.entities.AttachmentMetadata;
import fr.inra.agrosyst.api.services.common.AttachmentService;
import fr.inra.agrosyst.web.actions.AbstractAgrosystAction;

/**
 * This is the default action used to upload a file attached to a given object reference ID.
 * 
 * @author <a href="mailto:sebastien.grimault@makina-corpus.com">S. Grimault</a>
 */
public class AttachmentsRaw extends AbstractAgrosystAction implements Preparable {

    private static final long serialVersionUID = 1864154104471757665L;

    private static final Log log = LogFactory.getLog(AttachmentsRaw.class);
    
    private static final String SESSION_OBJECT_REFERENCE_ID = "session_object_reference_id";
    private static final String SESSION_UPLOAD_STATUS = "session_upload_status";
    private static final String SESSION_UPLOAD_MESSAGE = "session_upload_message";
    
    protected AttachmentService attachmentService;
    
    private String objectReferenceId;
    private List<AttachmentMetadata> attachmentMetadatas = Lists.newArrayList();
    private File upload;
    private String uploadContentType;
    private String uploadFileName;
    private InputStream inputStream;
    private String selectedAttachmentMetadataId = null;
    
    public void setAttachmentService(AttachmentService attachmentService) {
        this.attachmentService = attachmentService;
    }

    public List<AttachmentMetadata> getAttachmentMetadatas() {
        return attachmentMetadatas;
    }

    public String getObjectReferenceId() {
        return (String) ActionContext.getContext().getSession().get(SESSION_OBJECT_REFERENCE_ID);
    }

    public void setObjectReferenceId(String objectReferenceId) {
        this.objectReferenceId = objectReferenceId;
    }

    @Override
    public void prepare() throws Exception {
        log.debug("prepare");

        if (StringUtils.isNotBlank(objectReferenceId)) {
            ActionContext.getContext().getSession().put(SESSION_OBJECT_REFERENCE_ID, objectReferenceId);
        }
        
        if (attachmentMetadatas.isEmpty() && StringUtils.isNotBlank(getObjectReferenceId())) {
            attachmentMetadatas = attachmentService.getAttachmentMetadatas(getObjectReferenceId());
        }
    }

    @Override
    public void validate() {
        log.debug("validate");
        
        if (hasErrors()) {
            
            if (!getActionErrors().isEmpty()) {
                ActionContext.getContext().getSession().put(SESSION_UPLOAD_STATUS, "error");
                ActionContext.getContext().getSession().put(SESSION_UPLOAD_MESSAGE, getActionErrors().iterator().next());
            }
            initForInput();
        }
        else {
            ActionContext.getContext().getSession().remove(SESSION_UPLOAD_STATUS);
            ActionContext.getContext().getSession().remove(SESSION_UPLOAD_MESSAGE);
        }
    }
    
    @Override
    @Action("attachments-raw-input")
    public String input() throws Exception {
        log.debug("input, objectReferenceId : " + getObjectReferenceId());
        
        return INPUT;
    }

    @Override
    @Action(results = {@Result(type = "stream", params = {"contentType", "text/html", "inputName", "inputStream"})})
    public String execute() throws Exception {
        log.debug("execute, objectReferenceId : " + getObjectReferenceId());
        
        if ((this.upload != null) && (this.upload.exists())) {
            AttachmentMetadata attachmentMetadata = attachmentService.newAttachmentMetadataInstance();
            attachmentMetadata.setObjectReferenceId(objectReferenceId);
            attachmentMetadata.setUserId(getSession().getAuthenticatedUser().getTopiaId());
            attachmentMetadata.setContentName(this.uploadFileName);
            attachmentMetadata.setContentType(this.uploadContentType);
            attachmentMetadata.setContentSize(this.upload.length());
            
            attachmentService.createOrUpdate(attachmentMetadata, this.upload);
            
            log.debug("execute : file uploaded");
            
            // FIXME: use i18n properties !
            inputStream = IOUtils.toInputStream("<div id='status'>success</div><div id='message'>T&eacute;l&eacute;chargement termin&eacute;</div>");
        }
        else {
            log.debug("execute : no file uploaded");
            
            // FIXME: use i18n properties !
            inputStream = IOUtils.toInputStream("<div id='status'>error</div><div id='message'>Aucun fichier t&eacute;l&eacute;charg&eacute;</div>");
        }
        
        return SUCCESS;
    }
    
    @Action("attachments-raw-attachments")
    public String doAttachments() {
        
        servletResponse.setCharacterEncoding(Charsets.UTF_8.name());
        servletResponse.setContentType("application/json");
        
        try {
            final ServletOutputStream sos = servletResponse.getOutputStream();
            
            sos.write(toJson(attachmentMetadatas).getBytes(Charsets.UTF_8));
            sos.flush();
            sos.close();
        } catch (IOException ioe) {
            log.warn(ioe);
        }
        
        return NONE;
    }

    @Action("attachments-raw-download")
    public String doDownload() {
        log.debug("doDownload");

        if (StringUtils.isNotBlank(selectedAttachmentMetadataId)) {
            AttachmentMetadata selectedAttachmentMetadata = attachmentService.getAttachmentMetadata(selectedAttachmentMetadataId);

            if (selectedAttachmentMetadata == null) {
                try {
                    servletResponse.sendError(404);
                }
                catch (IOException ioe) {
                    log.warn(ioe);
                }
            }
            else {
                AttachmentContent selectedAttachmentContent = attachmentService.getAttachmentContent(selectedAttachmentMetadata);

                if (selectedAttachmentContent == null) {
                    try {
                        servletResponse.sendError(404);
                    }
                    catch (IOException ioe) {
                        log.warn(ioe);
                    }
                }
                else {
                    try {
                        InputStream binaryStream = selectedAttachmentContent.getContent().getBinaryStream();
                        
                        servletResponse.addHeader("Content-Disposition", "attachment; filename=" + selectedAttachmentMetadata.getContentName());
                        servletResponse.setBufferSize((int) selectedAttachmentMetadata.getContentSize());
                        servletResponse.setContentType(selectedAttachmentMetadata.getContentType());
                        servletResponse.setContentLength((int) selectedAttachmentMetadata.getContentSize());

                        final ServletOutputStream sos = servletResponse.getOutputStream();
                        sos.write(IOUtils.toByteArray(binaryStream));
                        sos.flush();
                        sos.close();
                    }
                    catch (SQLException se) {
                        log.warn(se);
                    }
                    catch (IOException ioe) {
                        log.warn(ioe);
                    }
                }
            }
        }
        
        return NONE;
    }
    
    @Action("attachments-raw-delete")
    public String doDelete() {
        
        if (StringUtils.isNotBlank(selectedAttachmentMetadataId)) {
            attachmentService.delete(attachmentService.getAttachmentMetadata(selectedAttachmentMetadataId));
        }
        
        return NONE;
    }
    
    public void setUpload(File file) {
        log.debug("setUpload " + file.getName());
        
        this.upload = file;
    }

    public void setUploadContentType(String contentType) {
        log.debug("setUploadContentType " + contentType);
        
        this.uploadContentType = contentType;
    }

    public void setUploadFileName(String filename) {
        log.debug("setUploadFileName " + filename);
        
        this.uploadFileName = filename;
    }

    public InputStream getInputStream() {
        return inputStream;
    }
    
    public void setSelectedAttachmentMetadataId(String selectedAttachmentMetadataId) {
        this.selectedAttachmentMetadataId = selectedAttachmentMetadataId;
    }
    
    public String getStatus() {
        return (String) ActionContext.getContext().getSession().get(SESSION_UPLOAD_STATUS);
    }
    
    public String getMessage() {
        return (String) ActionContext.getContext().getSession().get(SESSION_UPLOAD_MESSAGE);
    }
}
