/*
 * #%L
 * Agrosyst :: Web
 * $Id: AgrosystWebTransactionFilter.java 1664 2013-10-14 10:04:47Z athimel $
 * $HeadURL: https://forge.codelutin.com/svn/agrosyst/tags/agrosyst-0.6.1/agrosyst-web/src/main/java/fr/inra/agrosyst/web/AgrosystWebTransactionFilter.java $
 * %%
 * Copyright (C) 2013 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

package fr.inra.agrosyst.web;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Properties;

import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaContextFactory;
import org.nuiton.topia.TopiaNotFoundException;
import org.nuiton.topia.TopiaRuntimeException;
import org.nuiton.web.filter.TopiaTransactionFilter;

import com.google.common.base.Preconditions;

/**
 * Filter managing unique transaction for each servlet request.
 *
 * @author Arnaud Thimel <thimel@codelutin.com>
 */
public class AgrosystWebTransactionFilter extends TopiaTransactionFilter {

    private static final Log log = LogFactory.getLog(AgrosystWebTransactionFilter.class);

    public static final String MISSING_APPLICATION_CONTEXT =
            AgrosystWebApplicationContext.class.getSimpleName() + " not found. You probably forgot to" +
                    " register " + AgrosystWebApplicationListener.class.getName() + " in your web.xml";

    protected boolean transactionFilterEnabled = false;

    // This will be used only if remote services are disabled
    protected TopiaContext rootContext;

    @Override
    public void init(FilterConfig filterConfig) throws ServletException {

        super.init(filterConfig);

        if (log.isInfoEnabled()) {
            log.info("Initializing Agrosyst Web Transaction Filter");
        }

        AgrosystWebApplicationContext applicationContext = (AgrosystWebApplicationContext)
                filterConfig.getServletContext().getAttribute(
                        AgrosystWebApplicationContext.APPLICATION_CONTEXT_PARAMETER);

        Preconditions.checkState(applicationContext != null, MISSING_APPLICATION_CONTEXT);
        transactionFilterEnabled = !applicationContext.getWebConfig().isServicesRemoteEnabled();

        if (transactionFilterEnabled) {
            try {
                Class<?> supplierClass = Class.forName("fr.inra.agrosyst.services.DefaultConfigurationSupplier");
                Method method = supplierClass.getMethod("getRootContextProperties");
                Properties rootProperties = (Properties) method.invoke(null);

                try {
                    rootContext = TopiaContextFactory.getContext(rootProperties);
                } catch (TopiaNotFoundException tnfe) {
                    throw new TopiaRuntimeException("Unable to initialize TopiaContext", tnfe);
                }
            } catch (ClassNotFoundException e) {
                if (log.isErrorEnabled()) {
                    log.error("Un exception occurred", e);
                }
            } catch (NoSuchMethodException e) {
                if (log.isErrorEnabled()) {
                    log.error("Un exception occurred", e);
                }
            } catch (InvocationTargetException e) {
                if (log.isErrorEnabled()) {
                    log.error("Un exception occurred", e);
                }
            } catch (IllegalAccessException e) {
                if (log.isErrorEnabled()) {
                    log.error("Un exception occurred", e);
                }
            }
        } else {
            if (log.isWarnEnabled()) {
                log.warn("Using remote services mode, transaction is not managed anymore on agrosyst-web side");
            }
        }
    }

    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException, ServletException {
        if (transactionFilterEnabled) {
            super.doFilter(request, response, chain);
        } else {
            // Nothing to do, just let the request process
            chain.doFilter(request, response);
        }
    }

    @Override
    protected TopiaContext beginTransaction(ServletRequest request) throws TopiaRuntimeException {
        TopiaContext tx = null;
        if (transactionFilterEnabled) {
            tx = rootContext.beginTransaction();
        }
        return tx;
    }

    @Override
    public void destroy() {
        if (log.isInfoEnabled()) {
            log.info("Destroy Filter");
        }
        if (rootContext != null) {
            rootContext.closeContext();
        }
    }

}
