/*
 * #%L
 * Agrosyst :: Web
 * $Id: PracticedSystemsEdit.java 1201 2013-09-07 12:33:56Z athimel $
 * $HeadURL: https://forge.codelutin.com/svn/agrosyst/tags/agrosyst-0.4.1/agrosyst-web/src/main/java/fr/inra/agrosyst/web/actions/practiced/PracticedSystemsEdit.java $
 * %%
 * Copyright (C) 2013 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

package fr.inra.agrosyst.web.actions.practiced;

import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import fr.inra.agrosyst.api.services.practiced.ActionDto;
import fr.inra.agrosyst.api.services.practiced.CropCyclePhaseDto;
import fr.inra.agrosyst.api.services.practiced.InterventionDto;
import fr.inra.agrosyst.api.services.practiced.PracticedPerennialCropCycleDto;
import fr.inra.agrosyst.services.practiced.PracticedCropCycleServiceImpl;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.struts2.convention.annotation.Action;
import org.apache.struts2.convention.annotation.Result;

import com.google.common.base.Function;
import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.gson.reflect.TypeToken;
import com.opensymphony.xwork2.Preparable;

import fr.inra.agrosyst.api.NavigationContext;
import fr.inra.agrosyst.api.entities.ActionType;
import fr.inra.agrosyst.api.entities.CropCyclePerennialSpecies;
import fr.inra.agrosyst.api.entities.CropCyclePhase;
import fr.inra.agrosyst.api.entities.CropCyclePhaseType;
import fr.inra.agrosyst.api.entities.CroppingPlanSpecies;
import fr.inra.agrosyst.api.entities.Entities;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.Intervention;
import fr.inra.agrosyst.api.entities.OrchardFrutalForm;
import fr.inra.agrosyst.api.entities.PollinatorSpreadMode;
import fr.inra.agrosyst.api.entities.PollinatorType;
import fr.inra.agrosyst.api.entities.PracticedPerennialCropCycle;
import fr.inra.agrosyst.api.entities.PracticedSeasonalCropCycle;
import fr.inra.agrosyst.api.entities.PracticedSystem;
import fr.inra.agrosyst.api.entities.PracticedSystemImpl;
import fr.inra.agrosyst.api.entities.PracticedSystemSource;
import fr.inra.agrosyst.api.entities.ToolsCoupling;
import fr.inra.agrosyst.api.entities.VineFrutalForm;
import fr.inra.agrosyst.api.entities.WeedType;
import fr.inra.agrosyst.api.entities.referentiels.RefActionAgrosystTravailEDI;
import fr.inra.agrosyst.api.entities.referentiels.RefClonePlantGrape;
import fr.inra.agrosyst.api.entities.referentiels.RefOrientationEDI;
import fr.inra.agrosyst.api.entities.referentiels.RefVariete;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.domain.CroppingPlanSpeciesDto;
import fr.inra.agrosyst.api.services.domain.CroppingPlans;
import fr.inra.agrosyst.api.services.domain.ToolsCouplingDto;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemFilter;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemService;
import fr.inra.agrosyst.api.services.practiced.CropCycleConnectionDto;
import fr.inra.agrosyst.api.services.practiced.CropCycleModelDto;
import fr.inra.agrosyst.api.services.practiced.CropCycleNodeDto;
import fr.inra.agrosyst.api.services.practiced.CropCyclePerennialSpeciesDto;
import fr.inra.agrosyst.api.services.practiced.PracticedCropCycleService;
import fr.inra.agrosyst.api.services.practiced.PracticedSeasonalCropCycleDto;
import fr.inra.agrosyst.api.services.practiced.PracticedSystemService;
import fr.inra.agrosyst.api.services.referentiels.ReferentielService;
import fr.inra.agrosyst.web.actions.AbstractAgrosystAction;

/**
 * Action d'édition d'un systeme pratiqué.
 * 
 * Avec:
 * <ul>
 * <li>l'onglet génralité
 * <li>la liste des cycles pluriannuels assolés
 * <li>la liste des cycles pluriannuels pérennes
 * <li>l'itinéraire technique
 * </ul>
 * 
 * @author Eric Chatellier
 */
public class PracticedSystemsEdit extends AbstractAgrosystAction implements Preparable {

    /** serialVersionUID. */
    private static final long serialVersionUID = -5696853256365484417L;

    protected PracticedSystemService practicedSystemService;

    protected GrowingSystemService growingSystemService;
    
    protected PracticedCropCycleService practicedCropCycleService;
    
    protected ReferentielService referentielService;

    /** L'id du systeme pratiqué en cours d'edition. */
    protected String practicedSystemId;

    /** L'instance du systeme pratiqué en cours d'edition. */
    protected PracticedSystem practicedSystem;

    /** La liste de systeme de culture auquel peut être ratachés le SP (en creation). */
    protected List<GrowingSystem> growingSystems;

    /** Le systeme de culture sélectionné. */
    protected String growingSystemId;

    protected List<CropCycleModelDto> croppingPlanEntryDtos;

    protected List<PracticedPerennialCropCycleDto> practicedPerennialCropCycleDtos;

    protected List<Intervention> interventions;
    
    protected List<RefOrientationEDI> refOrientationEDIs;

    protected List<RefActionAgrosystTravailEDI> refActionAgrosystTravailEDIs;
    /** Liste des cycle pluriannuels assolé. */
    protected List<PracticedSeasonalCropCycleDto> practicedSeasonalCropCycleDtos;

    protected List<ToolsCouplingDto> toolsCouplingsDtos;

    /**
     * Contient une map <culture, List<Species>> pour toutes les cultures des systemes de cultures
     * concernés sur les campagnes du systeme pratiqué.
     */
    protected Map<String, List<CroppingPlanSpeciesDto>> growingSystemsCodeToSpecies;

    /** Les cultures principales disponibles dans le graphique. */
    protected List<CropCycleModelDto> seasonalGraphMainCrops;

    /** Les cultures intermédiaires disponible dans le graphique. */
    protected List<CropCycleModelDto> seasonalGraphMainIntermediateCrops;

    public void setPracticedSystemService(PracticedSystemService practicedSystemService) {
        this.practicedSystemService = practicedSystemService;
    }
    
    public void setPracticedCropCycleService(PracticedCropCycleService practicedCropCycleService) {
        this.practicedCropCycleService = practicedCropCycleService;
    }

    public void setGrowingSystemService(GrowingSystemService growingSystemService) {
        this.growingSystemService = growingSystemService;
    }
    
    public void setReferentielService(ReferentielService referentielService) {
        this.referentielService = referentielService;
    }

    public PracticedSystem getPracticedSystem() {
        if (practicedSystem == null) {
            // EChatellier 27/06/2013 Fais chier de devoir écrire ça, mais c'est la seule option pour ne pas avoir une grosse dose d'exceptions avec du paramsPrepareParams
            return new PracticedSystemImpl();
        }
        return practicedSystem;
    }

    @Override
    public void prepare() {
        if (StringUtils.isNotBlank(practicedSystemId)) {
            practicedSystem = practicedSystemService.getPracticedSystem(practicedSystemId);
            
        } else {
            // j'offre la possibilité de crééer un PracticedCropCycle
            practicedSystem = new PracticedSystemImpl();
        }
    }

    @Override
    public String input() {
        initForInput();

        // default to empty lists
        practicedSeasonalCropCycleDtos = Lists.newArrayList();
        practicedPerennialCropCycleDtos = Lists.newArrayList();

        if (practicedSystem.isPersisted()) {
            List<PracticedPerennialCropCycle> practicedPerennialCropCycles =  practicedCropCycleService.getAllPracticedPerennialCropCycle(practicedSystem);
            practicedPerennialCyclesInput(practicedPerennialCropCycles);
            List<PracticedSeasonalCropCycle> practicedSeasonalCropCycles = practicedCropCycleService.getAllPracticedSeasonalCropCycle(practicedSystem);
            practicedSeasonalCyclesInput(practicedSeasonalCropCycles);
        }

        return INPUT;
    }

    protected void practicedPerennialCyclesInput(List<PracticedPerennialCropCycle> practicedPerennialCropCycles) {
        for (PracticedPerennialCropCycle practicedPerennialCropCycle : practicedPerennialCropCycles) {
            PracticedPerennialCropCycleDto practicedPerennialCropCycleDto = new PracticedPerennialCropCycleDto();

            String cycleId = practicedPerennialCropCycle.getTopiaId();
            practicedPerennialCropCycleDto.setTopiaId(cycleId);
            practicedPerennialCropCycleDto.setPlantingYear(practicedPerennialCropCycle.getPlantingYear());
            practicedPerennialCropCycleDto.setPlantingDensity(practicedPerennialCropCycle.getPlantingDensity());
            practicedPerennialCropCycleDto.setPlantingInterFurrow(practicedPerennialCropCycle.getPlantingInterFurrow());
            practicedPerennialCropCycleDto.setPlantingSpacing(practicedPerennialCropCycle.getPlantingSpacing());
            practicedPerennialCropCycleDto.setPollinator(practicedPerennialCropCycle.isPollinator());
            practicedPerennialCropCycleDto.setPlantingDeathRate(practicedPerennialCropCycle.getPlantingDeathRate());
            practicedPerennialCropCycleDto.setPlantingDeathRateMeasureYear(practicedPerennialCropCycle.getPlantingDeathRateMeasureYear());
            practicedPerennialCropCycleDto.setVineFrutalForm(practicedPerennialCropCycle.getVineFrutalForm());
            practicedPerennialCropCycleDto.setOrchardFrutalForm(practicedPerennialCropCycle.getOrchardFrutalForm());
//            practicedPerennialCropCycleDto.setPollinatorType(practicedPerennialCropCycle.getPollinatorType());
//            practicedPerennialCropCycleDto.setPollinatorSpreadMode(practicedPerennialCropCycle.getPollinatorSpreadMode());
            practicedPerennialCropCycleDto.setWeedType(practicedPerennialCropCycle.getWeedType());
            practicedPerennialCropCycleDto.setPollinatorPercent(practicedPerennialCropCycle.getPollinatorPercent());
            practicedPerennialCropCycleDto.setOtherCharacteristics(practicedPerennialCropCycle.getOtherCharacteristics());

            RefOrientationEDI orientation = practicedPerennialCropCycle.getOrientation();
            if (orientation != null) {
                practicedPerennialCropCycleDto.setOrientationTopiaId(orientation.getTopiaId());
            }

            String croppingPlanEntryCode = practicedPerennialCropCycle.getCroppingPlanEntryCode();
            practicedPerennialCropCycleDto.setCroppingPlanEntryCode(croppingPlanEntryCode);

            List<CropCyclePhaseDto> cropCyclePhaseDtos = convertCropCyclePhases(practicedPerennialCropCycle);
            practicedPerennialCropCycleDto.setCropCyclePhaseDtos(cropCyclePhaseDtos);

            List<CropCyclePerennialSpeciesDto> cropCyclePerennialSpeciesDtos = practicedCropCycleService.getCropCyclePerennialSpecies(croppingPlanEntryCode, cycleId, null, null);
            practicedPerennialCropCycleDto.setCropCyclePerennialSpeciesDto(cropCyclePerennialSpeciesDtos);

            practicedPerennialCropCycleDtos.add(practicedPerennialCropCycleDto);
        }
    }

    /**
     * Convert database seasonal entity to dto.
     */
    protected void practicedSeasonalCyclesInput(List<PracticedSeasonalCropCycle> practicedSeasonalCropCycles) {

        for (PracticedSeasonalCropCycle practicedSeasonalCropCycle : practicedSeasonalCropCycles) {
            PracticedSeasonalCropCycleDto dto = new PracticedSeasonalCropCycleDto();
            dto.setTopiaId(practicedSeasonalCropCycle.getTopiaId());

            // graph data for each cycle
            List<CropCycleNodeDto> nodes = Lists.newArrayList();
            Map<CropCycleNodeDto, List<CroppingPlanSpeciesDto>> nodesToSpecies = practicedCropCycleService.getNodes(practicedSeasonalCropCycle.getTopiaId());
            for (Map.Entry<CropCycleNodeDto, List<CroppingPlanSpeciesDto>> entry : nodesToSpecies.entrySet()) {
                CropCycleNodeDto cycleNodeDto = entry.getKey();
                nodes.add(cycleNodeDto);
            }
            dto.setCropCycleNodeDtos(nodes);
            List<CropCycleConnectionDto> connections = practicedCropCycleService.getConnections(practicedSeasonalCropCycle.getTopiaId());
            dto.setCropCycleConnectionDtos(connections);

            // valuation du label des connexions
            if (connections != null && seasonalGraphMainIntermediateCrops != null) {
                for (CropCycleConnectionDto connection : connections) {
                    final String expectedCode = connection.getIntermediateCroppingPlanEntryCode();
                    if (!Strings.isNullOrEmpty(expectedCode)) {
                        Optional<CropCycleModelDto> optional = Iterables.tryFind(seasonalGraphMainIntermediateCrops, new Predicate<CropCycleModelDto>() {
                            @Override
                            public boolean apply(CropCycleModelDto input) {
                                return expectedCode.equals(input.getCroppingPlanEntryCode());
                            }
                        });
                        if (optional.isPresent()) {
                            connection.setLabel(optional.get().getLabel());
                        }
                    }
                }
            }

            // add dto
            practicedSeasonalCropCycleDtos.add(dto);
        }
    }

    @Override
    protected void initForInput() {
        // warning, practicedSystem's growingSystem must always be part of growingSystems set
        // even not selected be navigation context
        if (getPracticedSystem().getTopiaId() != null) {
            growingSystems = Collections.singletonList(getPracticedSystem().getGrowingSystem());
            growingSystemId = getPracticedSystem().getGrowingSystem().getTopiaId();
            toolsCouplingsDtos = practicedSystemService.getToolsCouplingModel(growingSystemId);
            try {
                croppingPlanEntryDtos = practicedSystemService.getCropCycleModel(
                        getPracticedSystem().getGrowingSystem().getTopiaId(),
                        getPracticedSystem().getCampaigns(),
                        false);
            } catch (NumberFormatException nfe) {
                // May fail if input campaigns are invalid
            }
        } else {
            GrowingSystemFilter growingSystemFilter = new GrowingSystemFilter();
            NavigationContext navigationContext = getNavigationContext();
            growingSystemFilter.setNavigationContext(navigationContext);
            growingSystemFilter.setActive(Boolean.TRUE);
            growingSystemFilter.setPageSize(GrowingSystemFilter.ALL_PAGE_SIZE);
            ResultList<GrowingSystem> growingSystemListResult = growingSystemService.getFilteredGrowingSystems(growingSystemFilter);
            growingSystems = growingSystemListResult.getElements();
        }
        refOrientationEDIs = referentielService.findAllReferentielEDI();
        refActionAgrosystTravailEDIs = referentielService.getRefActionAgrosystTravailEDIs();

        if (!Strings.isNullOrEmpty(growingSystemId) && !Strings.isNullOrEmpty(getPracticedSystem().getCampaigns())) {
            // init wall page data
            growingSystemsCodeToSpecies = Maps.newHashMap();

            // recuperation de toutes les cultures
            Map<CropCycleModelDto, List<CroppingPlanSpeciesDto>> modelToSpecies =
                    practicedSystemService.getCropCycleModelMap(
                            growingSystemId,
                            getPracticedSystem().getCampaigns(),
                            true);
            for (Map.Entry<CropCycleModelDto, List<CroppingPlanSpeciesDto>> entry : modelToSpecies.entrySet()) {
                this.growingSystemsCodeToSpecies.put(entry.getKey().getCroppingPlanEntryCode(), entry.getValue());
            }
            Set<CropCycleModelDto> model = modelToSpecies.keySet();

            // definition de la liste de culture principale
            Iterable<CropCycleModelDto> modelMain = Iterables.filter(model, CroppingPlans.IS_NOT_INTERMEDIATE);
            this.seasonalGraphMainCrops = Lists.newArrayList(modelMain);

            // définition de la liste de culture intermédiaire
            Iterable<CropCycleModelDto> modelIntermediate = Iterables.filter(model, CroppingPlans.IS_INTERMEDIATE);
            this.seasonalGraphMainIntermediateCrops = Lists.newArrayList(modelIntermediate);
        }
    }

    @Override
    public void validate() {
        if (StringUtils.isBlank(getPracticedSystem().getTopiaId())) {
            if (StringUtils.isBlank(growingSystemId)) {
                addFieldError("growingSystemId", "Le système de culture est requis !");
            }
        }

        if (StringUtils.isBlank(practicedSystem.getName())) {
            addFieldError("practicedSystem.name", "Le nom du système pratiqué est requis !");
        }
        String campaigns = practicedSystem.getCampaigns();
        if (StringUtils.isBlank(campaigns)) {
            addFieldError("practicedSystem.campaigns", "La série de campagne est obligatoire");
        }
        if (StringUtils.isNotBlank(campaigns)) {
            if (!practicedSystemService.areCampaignsValid(campaigns)) {
                String format = "La série de campagne doit être composée d'années séparées par des virgules (,), " +
                        "espaces ( ) ou point-virgules (;). Les campagnes doivent être contenues entre %d et %d.";
                Pair<Integer,Integer> bounds = practicedSystemService.getCampaignsBounds();
                addFieldError("practicedSystem.campaigns", String.format(format, bounds.getLeft(), bounds.getRight()));
            }
        }

        if (practicedPerennialCropCycleDtos != null){
            practicedPerennialCycleValidate();
        }
        
        if (hasErrors()) {
            initForInput();
        }
    }
    
    protected void practicedPerennialCycleValidate() {
        // validate perennial cycle
        for (PracticedPerennialCropCycleDto practicedPerennialCropCycleDto : practicedPerennialCropCycleDtos){
            if (Strings.isNullOrEmpty(practicedPerennialCropCycleDto.getCroppingPlanEntryCode())) {
                addFieldError("croppingPlanEntry", "Champ obligatoire");
            }

            List<CropCyclePhaseDto> cropCyclePhaseDtos = practicedPerennialCropCycleDto.getCropCyclePhaseDtos();
            if (cropCyclePhaseDtos == null){
                addActionError("Un cycle pluriannuel doit avoir une phase de pleine production");
            } else {
                int instCount = 0, montCount = 0, pleiCount = 0, declCount = 0;
                for (CropCyclePhaseDto cropCyclePhaseDto : cropCyclePhaseDtos) {
                    if (cropCyclePhaseDto.getCropCyclePhaseType() != null) {
                        switch (cropCyclePhaseDto.getCropCyclePhaseType()) {
                            case INSTALLATION:
                                instCount++;
                                break;
                            case MONTEE_EN_PRODUCTION:
                                montCount++;
                                break;
                            case PLEINE_PRODUCTION:
                                pleiCount++;
                                break;
                            case DECLIN_DE_PRODUCTION:
                                declCount++;
                                break;
                            default:
                                break;
                        }

                    }
                    List<InterventionDto> interventionDtos = cropCyclePhaseDto.getInterventions();
                    if (interventionDtos != null) {
                        for (InterventionDto interventionDto : interventionDtos) {
                            if (Strings.isNullOrEmpty(interventionDto.getName())) {
                                addFieldError("intervention_name", "Le nom est obligatoire");
                            }
                        }
                    }
                }
                if (instCount > 1) {
                    addActionError("Un cycle pluriannuel ne peut avoir qu'une seul phase d'installation");
                }
                if (montCount > 1) {
                    addActionError("Un cycle pluriannuel ne peut avoir qu'une seul phase de montée en production");
                }
                if (pleiCount == 0) {
                    addActionError("Un cycle pluriannuel doit avoir une phase de pleine production");
                }
                else if (pleiCount > 1) {
                    addActionError("Un cycle pluriannuel ne peut avoir qu'une seul phase de pleine production");
                }
                if (declCount > 1) {
                    addActionError("Un cycle pluriannuel ne peut avoir qu'une seul phase de déclin de production");
                }
            }
        }
        // / validate perennial
        
        // validate 
    }

    @Action(results = {@Result(type = "redirectAction", params = {"actionName", "practiced-systems-list"})})
    @Override
    public String execute() throws Exception {
        PracticedSystem practicedSystem = getPracticedSystem();
        // can define domain only during create action
        if (StringUtils.isBlank(practicedSystem.getTopiaId())) {
            GrowingSystem growingSystem = growingSystemService.getGrowingSystem(growingSystemId);
            getPracticedSystem().setGrowingSystem(growingSystem);
        }

        List<PracticedPerennialCropCycle> perennialCropCycles = convertPracticedPerennialCycles();
        practicedSystemService.createOrUpdatePracticedSystem(practicedSystem, perennialCropCycles,
                practicedSeasonalCropCycleDtos, interventions);

        notificationSupport.practicedSystemSaved(practicedSystem);

        return SUCCESS;
    }
    
    protected List<PracticedPerennialCropCycle> convertPracticedPerennialCycles() throws Exception {
        List<PracticedPerennialCropCycle> practicedPerennialCropCycles = Lists.newArrayList();
        if (practicedPerennialCropCycleDtos != null) {
            for (PracticedPerennialCropCycleDto practicedPerennialCropCycleDto : practicedPerennialCropCycleDtos){
                if (!Strings.isNullOrEmpty(practicedPerennialCropCycleDto.getCroppingPlanEntryCode())) {
                    PracticedPerennialCropCycle cycle;
                    String practicedPerennialCropCycleTopiaId = practicedPerennialCropCycleDto.getTopiaId();
                    if (practicedPerennialCropCycleTopiaId != null){
                        cycle = practicedCropCycleService.getPracticedPerennialCropCycle(practicedPerennialCropCycleTopiaId);
                    } else {
                        cycle = practicedCropCycleService.newPracticedPerennialCropCycleInstance();
                    }
                    cycle.setPlantingYear(practicedPerennialCropCycleDto.getPlantingYear());
                    cycle.setPlantingDensity(practicedPerennialCropCycleDto.getPlantingDensity());
                    cycle.setPlantingInterFurrow(practicedPerennialCropCycleDto.getPlantingInterFurrow());
                    cycle.setPlantingSpacing(practicedPerennialCropCycleDto.getPlantingSpacing());
                    cycle.setPollinator(practicedPerennialCropCycleDto.isPollinator());
                    cycle.setPlantingDeathRate(practicedPerennialCropCycleDto.getPlantingDeathRate());
                    cycle.setPlantingDeathRateMeasureYear(practicedPerennialCropCycleDto.getPlantingDeathRateMeasureYear());
                    cycle.setVineFrutalForm(practicedPerennialCropCycleDto.getVineFrutalForm());
                    cycle.setOrchardFrutalForm(practicedPerennialCropCycleDto.getOrchardFrutalForm());
//                    cycle.setPollinatorType(practicedPerennialCropCycleDto.getPollinatorType());
//                    cycle.setPollinatorSpreadMode(practicedPerennialCropCycleDto.getPollinatorSpreadMode());
                    cycle.setWeedType(practicedPerennialCropCycleDto.getWeedType());
                    cycle.setPollinatorPercent(practicedPerennialCropCycleDto.getPollinatorPercent());
                    cycle.setOtherCharacteristics(practicedPerennialCropCycleDto.getOtherCharacteristics());

                    // set cropping plan entry
    //                if (cycle.getCroppingPlanEntry() == null || !cycle.getCroppingPlanEntry().getTopiaId().contentEquals(practicedPerennialCropCycleDto.getCroppingPlanEntryDto().getTopiaId())){
    //                    CroppingPlanEntry croppingPlanEntry = practicedCropCycleService.getCroppingPlanEntry(practicedPerennialCropCycleDto.getCroppingPlanEntryDto().getTopiaId());
    //                    cycle.setCroppingPlanEntry(croppingPlanEntry);
    //                }
                    cycle.setCroppingPlanEntryCode(practicedPerennialCropCycleDto.getCroppingPlanEntryCode());

                    // set orientation edi entity
                    if (cycle.getOrientation() == null || !cycle.getOrientation().getTopiaId().contentEquals(practicedPerennialCropCycleDto.getOrientationTopiaId())) {
                        if (!Strings.isNullOrEmpty(practicedPerennialCropCycleDto.getOrientationTopiaId())) {
                            RefOrientationEDI orientation = referentielService.findOrientation(practicedPerennialCropCycleDto.getOrientationTopiaId());
                            cycle.setOrientation(orientation);
                        } else {
                            cycle.setOrientation(null);
                        }
                    }

                    // set associated entities
                    convertCropCyclePerennialSpeciesDto(practicedPerennialCropCycleDto, cycle);
                    convertCropCyclePhaseDto(practicedPerennialCropCycleDto, cycle);
                    practicedPerennialCropCycles.add(cycle);
                }
            }
        }

        
        return practicedPerennialCropCycles;
    }
    
    protected void convertCropCyclePerennialSpeciesDto(PracticedPerennialCropCycleDto cycleDto,PracticedPerennialCropCycle cycle) {
        // TODO AThimel 05/09/13 refactor
        List<CropCyclePerennialSpecies> currentSpecies = cycle.getCropCyclePerennialSpecies();
        Collection<CropCyclePerennialSpecies> nonDeleted = Lists.newArrayList();
        if (currentSpecies == null) {
            currentSpecies = Lists.newArrayList();
            cycle.setCropCyclePerennialSpecies(currentSpecies);
        }

        // update list with dto
        Map<String, CropCyclePerennialSpecies> currentSpeciesMap = Maps.uniqueIndex(currentSpecies, new Function<CropCyclePerennialSpecies, String>() {
            @Override
            public String apply(CropCyclePerennialSpecies input) {
                return input.getCroppingPlanSpeciesCode();
            }
        });
        List<CropCyclePerennialSpeciesDto> cropCyclePerennialSpeciesDtos = cycleDto.getCropCyclePerennialSpeciesDto();
        if (cropCyclePerennialSpeciesDtos != null) {
            for (CropCyclePerennialSpeciesDto cropCyclePerennialSpeciesDto : cropCyclePerennialSpeciesDtos) {
                String code = cropCyclePerennialSpeciesDto.getCode();
                CropCyclePerennialSpecies cropCyclePerennialSpecies = currentSpeciesMap.get(code);
                if (cropCyclePerennialSpecies == null) {
                    cropCyclePerennialSpecies = practicedCropCycleService.newCropCyclePerennialSpeciesInstance();
                    cropCyclePerennialSpecies.setCroppingPlanSpeciesCode(code);
                    cycle.addCropCyclePerennialSpecies(cropCyclePerennialSpecies);
                }

                cropCyclePerennialSpecies.setPlantsCertified(cropCyclePerennialSpeciesDto.isPlantCertified());
                cropCyclePerennialSpecies.setOverGraftDate(cropCyclePerennialSpeciesDto.getOverGraftDate());

                if (cropCyclePerennialSpeciesDto.getGraftClone() != null) {
                    String graftTopiaId = cropCyclePerennialSpeciesDto.getGraftClone().getTopiaId();
                    if (StringUtils.isNotBlank(graftTopiaId)) {
                        RefClonePlantGrape clonePlantGrape = referentielService.getClonePlantGrape(graftTopiaId);
                        cropCyclePerennialSpecies.setGraftClone(clonePlantGrape);
                    }
                }
                if (cropCyclePerennialSpeciesDto.getGraftSupport() != null) {
                    String graftTopiaId = cropCyclePerennialSpeciesDto.getGraftSupport().getTopiaId();
                    if (StringUtils.isNotBlank(graftTopiaId)) {
                        RefVariete variete = referentielService.getVariete(graftTopiaId);
                        cropCyclePerennialSpecies.setGraftSupport(variete);
                    }
                }

                nonDeleted.add(cropCyclePerennialSpecies);
            }
        }
        currentSpecies.retainAll(nonDeleted);
    }

    protected void convertCropCyclePhaseDto(PracticedPerennialCropCycleDto cycleDto, PracticedPerennialCropCycle cycle) {
        List<CropCyclePhase> currentPhases = cycle.getCropCyclePhases();
        List<CropCyclePhase> nonDeleted = Lists.newArrayList();
        if (currentPhases == null) {
            currentPhases = Lists.newArrayList();
            cycle.setCropCyclePhases(currentPhases);
        }

        // update list with dto
        Map<String, CropCyclePhase> currentPhasesMap = Maps.uniqueIndex(currentPhases, Entities.GET_TOPIA_ID);
        // there is at least one CropCyclePhase
        for (CropCyclePhaseDto cropCyclePhaseDto : cycleDto.getCropCyclePhaseDtos()) {
            String phaseId = cropCyclePhaseDto.getTopiaId();
            CropCyclePhase phase;
            if (StringUtils.isBlank(phaseId)) {
                phase = practicedCropCycleService.newCropCyclePhaseIntance();
            } else {
                phase = currentPhasesMap.get(phaseId);
            }
            
            phase.setCropCyclePhaseType(cropCyclePhaseDto.getCropCyclePhaseType());
            phase.setDuration(cropCyclePhaseDto.getDuration());
            if (StringUtils.isBlank(phaseId)) {
                currentPhases.add(phase);
            }
            convertInterventionDto(cropCyclePhaseDto, phase);
            nonDeleted.add(phase);
        }
        currentPhases.retainAll(nonDeleted);
    }

    protected void convertInterventionDto(CropCyclePhaseDto cropCyclePhaseDto, CropCyclePhase phase) {
        
        List<InterventionDto> interventionDtos = cropCyclePhaseDto.getInterventions();
        List<Intervention> interventions = null;
        if (phase.isPersisted()){
            interventions = practicedCropCycleService.getCropCyclePhaseInterventions(phase);
        } else {
            interventions = Lists.newArrayList();
        }
        if (this.interventions == null) {
            this.interventions = Lists.newArrayList();
        }

        List<Intervention> nonDeleted = Lists.newArrayList();

        if(interventionDtos != null && !interventionDtos.isEmpty()) {

            Map<InterventionDto, ToolsCoupling> interventionsToolsCouplings = Maps.newHashMap();
            for (InterventionDto interventionDto:interventionDtos) {
                ToolsCouplingDto toolsCouplingDto = interventionDto.getToolsCouplingDto();
                if (toolsCouplingDto != null) {
                    String toolsCouplingId = toolsCouplingDto.getTopiaId();
                    ToolsCoupling toolsCoupling = practicedCropCycleService.getToolsCoupling(toolsCouplingId);
                    interventionsToolsCouplings.put(interventionDto, toolsCoupling);
                }
            }

            Map<String, CroppingPlanSpecies> allInterventionsAllSpecies = Maps.newHashMap();
            for (InterventionDto interventionDto:interventionDtos) {
                List<String> interventionSpeciesIds = interventionDto.getInterventionSpeciesIds();
                if (interventionSpeciesIds != null) {
                    for (String interventionSpeciesId:interventionSpeciesIds){
                        CroppingPlanSpecies interventionSpecies = practicedCropCycleService.getCroppingPlanSpecies(interventionSpeciesId);
                        allInterventionsAllSpecies.put(interventionSpeciesId, interventionSpecies);
                    }
                }
            }
          
            Map<String, RefActionAgrosystTravailEDI> refActionAgrosystTravailEdis = Maps.newHashMap();
            for (InterventionDto interventionDto:interventionDtos) {
                List<ActionDto> actionsDtos = interventionDto.getActionDtos();
                if (actionsDtos != null) {
                    for (ActionDto actionsDto:actionsDtos){
                        String refActionAgrosystTravailEdiId = actionsDto.getRefActionAgrosystTravailEdiId();
                        if (!StringUtils.isBlank(refActionAgrosystTravailEdiId)) {
                          RefActionAgrosystTravailEDI refActionAgrosystTravailEDI = referentielService.getRefActionAgrosystTravailEDI(refActionAgrosystTravailEdiId);
                            refActionAgrosystTravailEdis.put(refActionAgrosystTravailEdiId, refActionAgrosystTravailEDI);
                        }
                    }
                }
            }
            
            Map<String, Intervention> currentInterventionsMap = Maps.uniqueIndex(interventions, Entities.GET_TOPIA_ID);

            for (InterventionDto interventionDto:interventionDtos) {

                Intervention intervention;
                if (StringUtils.isBlank(interventionDto.getTopiaId())) {
                    intervention = practicedSystemService.newInterventionInstance();
                    intervention.setCropCyclePhase(phase);
                    interventions.add(intervention);
                } else {
                    intervention = currentInterventionsMap.get(interventionDto.getTopiaId());
                }

                ToolsCouplingDto toolsCouplingDto = interventionDto.getToolsCouplingDto();
                if (toolsCouplingDto != null) {
                    String toolsCouplingId = toolsCouplingDto.getTopiaId();
                    if (intervention.getToolsCoupling() == null || intervention.getToolsCoupling().getTopiaId().contentEquals(toolsCouplingId)) {
                        ToolsCoupling toolsCoupling = interventionsToolsCouplings.get(interventionDto);
                        intervention.setToolsCoupling(toolsCoupling);
                    }
                }

                List<CroppingPlanSpecies> allInterventionSpecies = intervention.getInterventionSpecies();
                if (allInterventionSpecies == null){
                    allInterventionSpecies = Lists.newArrayList();
                    intervention.setInterventionSpecies(allInterventionSpecies);
                }
                List<CroppingPlanSpecies> nonDeletedSpeces = new ArrayList<CroppingPlanSpecies>();
                Map<String, CroppingPlanSpecies> currentAllCroppingPlanSpeciesMap = Maps.uniqueIndex(allInterventionSpecies, Entities.GET_TOPIA_ID);
                List<String> interventionSpeciesIds = interventionDto.getInterventionSpeciesIds();
                if (interventionSpeciesIds != null) {

                    for (String interventionSpeciesId:interventionSpeciesIds){
                        CroppingPlanSpecies interventionSpecies = currentAllCroppingPlanSpeciesMap.get(interventionSpeciesId);
                        if (interventionSpecies == null) {
                            interventionSpecies = allInterventionsAllSpecies.get(interventionSpeciesId);
                            allInterventionSpecies.add(interventionSpecies);
                        }
                        nonDeletedSpeces.add(interventionSpecies);
                    }
                }
                allInterventionSpecies.retainAll(nonDeletedSpeces);

                convertActionDto(interventionDto, intervention, refActionAgrosystTravailEdis);

                intervention.setComment(interventionDto.getComment());
                intervention.setCrop(interventionDto.getCrop());
                intervention.setStartingPeriodDate(interventionDto.getStartingPeriodDate());
                intervention.setEndingPeriodDate(interventionDto.getEndingPeriodDate());
                intervention.setIntermediateCrop(interventionDto.isIntermediateCrop());
                intervention.setInvolvedPeopleNumber(interventionDto.getInvolvedPeopleNumber());
                intervention.setName(interventionDto.getName());
                intervention.setOtherToolSettings(interventionDto.getOtherToolSettings());
                intervention.setProgressionSpeed(interventionDto.getProgressionSpeed());
                intervention.setRate(interventionDto.getRate());
                intervention.setRefStadeEdi(interventionDto.getRefStadeEdi());
                intervention.setTillageDepth(interventionDto.getTillageDepth());
                intervention.setWorkRate(interventionDto.getWorkRate());
                intervention.setAvrerageArea(interventionDto.getAvrerageArea());

                nonDeleted.add(intervention);
                //this.interventions.add(intervention);
            }
        }
        interventions.retainAll(nonDeleted);
        this.interventions.addAll(interventions);
    }
    
    protected void convertActionDto(InterventionDto interventionDto, Intervention intervention, Map<String, RefActionAgrosystTravailEDI> refActionAgrosystTravailEdis) {
        List<ActionDto> actionsDtos = interventionDto.getActionDtos();
        List<fr.inra.agrosyst.api.entities.Action> actions = intervention.getActions();
        List<fr.inra.agrosyst.api.entities.Action> nonDeleted = Lists.newArrayList();
        if (actions == null){
            actions = Lists.newArrayList();
            intervention.setActions(actions);
        }
        
        if (actionsDtos != null) {
            Map<String, fr.inra.agrosyst.api.entities.Action> currentAllActionMap = Maps.uniqueIndex(actions, Entities.GET_TOPIA_ID);
            
            for (ActionDto actionsDto:actionsDtos){
                fr.inra.agrosyst.api.entities.Action action = currentAllActionMap.get(actionsDto);
                if (action == null) {
                    action = practicedCropCycleService.newActionInstance(actionsDto.getActionType());
                    actions.add(action);
                }
                action.setActionType(actionsDto.getActionType());
                String refActionAgrosystTravailEdiId = actionsDto.getRefActionAgrosystTravailEdiId();
                if (!StringUtils.isBlank(refActionAgrosystTravailEdiId)) {
                    RefActionAgrosystTravailEDI refActionAgrosystTravailEDI = refActionAgrosystTravailEdis.get(refActionAgrosystTravailEdiId);
                    action.setRefActionAgrosystTravailEDI(refActionAgrosystTravailEDI);
                }
                action.setIntervention(intervention);
                nonDeleted.add(action);
            }
        }
        actions.retainAll(nonDeleted);
    }

//    protected CroppingPlanEntryDto convertCroppingPlanEntry(PracticedPerennialCropCycle cycle){
//        CroppingPlanEntryDto result = null;
//        CroppingPlanEntry croppingPlanEntry = cycle.getCroppingPlanEntry();
//        if (croppingPlanEntry != null) {
//            result = new CroppingPlanEntryDto();
//            result.setTopiaId(croppingPlanEntry.getTopiaId());
//            String name = croppingPlanEntry.getName();
//            result.setName(name);
//            result.setSellingPrice(croppingPlanEntry.getSellingPrice());
//            result.setType(croppingPlanEntry.getType());
//            result.setSpecies(Lists.transform(croppingPlanEntry.getCroppingPlanSpecies(), CroppingPlans.CROPPING_PLAN_SPECIES_TO_DTO));
//            result.setColor(CroppingPlans.STRING_TO_COLOR.apply(name));
//        }
//
//        return result;
//    }

    protected List<CropCyclePhaseDto> convertCropCyclePhases(PracticedPerennialCropCycle cycle) {
        List<CropCyclePhase> currentPhases = cycle.getCropCyclePhases();
        List<CropCyclePhaseDto> currentPhasesDtos = Lists.newArrayListWithCapacity(currentPhases.size());
        for (CropCyclePhase currentPhase : currentPhases) {
            CropCyclePhaseDto currentPhaseDto = new CropCyclePhaseDto();
            currentPhaseDto.setCropCyclePhaseType(currentPhase.getCropCyclePhaseType());
            currentPhaseDto.setDuration(currentPhase.getDuration());
            currentPhaseDto.setTopiaId(currentPhase.getTopiaId());
            List<Intervention> interventions = practicedCropCycleService.getCropCyclePhaseInterventions(currentPhase);
            if (interventions != null) {
                List<InterventionDto> interventionDtos = Lists.newArrayList(Iterables.transform(interventions, PracticedCropCycleServiceImpl.INTERVENTION_TO_INTERVENTION_DTO));
                currentPhaseDto.setInterventions(interventionDtos);
            }
            currentPhasesDtos.add(currentPhaseDto);

        }
        return currentPhasesDtos;
    }

//    protected List<CropCyclePerennialSpeciesDto> convertCropCyclePerennialSpecies(PracticedPerennialCropCycle cycle) {
//        List<CropCyclePerennialSpeciesDto> result = null;
//        List<CropCyclePerennialSpecies> allCropCyclePerennialSpecies = cycle.getCroppingPlanSpeciesCropCyclePerennialSpecies();
//        if (allCropCyclePerennialSpecies != null) {
//            result = Lists.newArrayListWithCapacity(allCropCyclePerennialSpecies.size());
//            for (CropCyclePerennialSpecies cropCyclePerennialSpecies :allCropCyclePerennialSpecies){
//                CropCyclePerennialSpeciesDto cropCyclePerennialSpeciesDto = new CropCyclePerennialSpeciesDto();
//                cropCyclePerennialSpeciesDto.setTopiaId(cropCyclePerennialSpecies.getTopiaId());
//                cropCyclePerennialSpeciesDto.setCroppingPlanSpeciesId(cropCyclePerennialSpecies.getCroppingPlanSpecies().getTopiaId());
//                cropCyclePerennialSpeciesDto.setCroppingPlanSpeciesEspece(cropCyclePerennialSpecies.getCroppingPlanSpecies().getSpecies().getLibelle_espece_botanique());
//                cropCyclePerennialSpeciesDto.setPlantCertified(cropCyclePerennialSpecies.isPlantsCertified());
//                cropCyclePerennialSpeciesDto.setOverGraftDate(cropCyclePerennialSpecies.getOverGraftDate());
//                cropCyclePerennialSpeciesDto.setSpeciedPercent(cropCyclePerennialSpecies.getSpeciesPercent());
//                RefVariete graftClone = cropCyclePerennialSpecies.getGraftClone();
//                if (graftClone != null) {
//                    CropCycleGraftDto cropCycleGraftDto = new CropCycleGraftDto();
//                    cropCycleGraftDto.setTopiaId(graftClone.getTopiaId());
//                    cropCyclePerennialSpeciesDto.setGraftClone(cropCycleGraftDto);
//                }
//                RefVariete graftSupport = cropCyclePerennialSpecies.getGraftSupport();
//                if(graftSupport != null) {
//                    CropCycleGraftDto cropCycleGraftSupportDto = new CropCycleGraftDto();
//                    cropCycleGraftSupportDto.setTopiaId(graftSupport.getTopiaId());
//                    cropCyclePerennialSpeciesDto.setGraftSupport(cropCycleGraftSupportDto);
//                }
//                result.add(cropCyclePerennialSpeciesDto);
//            }
//        }
//        return result;
//    }

    public List<GrowingSystem> getGrowingSystems() {
        return growingSystems;
    }
    
    public String getGrowingSystemsJson() {
        return getGson().toJson(growingSystems);
    }

    public void setPracticedSystemTopiaId(String practicedSystemTopiaId) {
        this.practicedSystemId = practicedSystemTopiaId;
    }

    public void setGrowingSystemId(String growingSystemId) {
        if ("? undefined:undefined ?".equals(growingSystemId)) {  // TODO AThimel 07/09/13 Find where it comes from
            this.growingSystemId = null;
        } else {
            this.growingSystemId = growingSystemId;
        }
    }

    public Map<PracticedSystemSource, String> getSources() {
        return getEnumAsMap(PracticedSystemSource.values());
    }
    
    public List<RefOrientationEDI> getRefOrientationEDIs() {
        return refOrientationEDIs;
    }

    public String getRefOrientationEDIsJson() {
        return getGson().toJson(refOrientationEDIs);
    }

    public void setPracticedSystemId(String practicedSystemId) {
        this.practicedSystemId = practicedSystemId;
    }

    public String getPracticedSystemId() {
        return practicedSystemId;
    }

    public List<CropCycleModelDto> getCroppingPlanEntryDtos() {
        return croppingPlanEntryDtos;
    }

    public String getCroppingPlanEntryDtosJson() {
        return getGson().toJson(croppingPlanEntryDtos);
    }
    
//    public void setCroppingPlanEntryDtos(List<CroppingPlanEntryDto> croppingPlanEntryDtos) {
//        this.croppingPlanEntryDtos = croppingPlanEntryDtos;
//    }

    public Map<VineFrutalForm, String> getVineFrutalForms() {
        return getEnumAsMap(VineFrutalForm.values());
    }

    public Map<OrchardFrutalForm, String> getOrchardFrutalForms() {
        return getEnumAsMap(OrchardFrutalForm.values());
    }
    
    public Map<PollinatorType, String> getPollinatorTypes() {
        return getEnumAsMap(PollinatorType.values());
    }
    
    public Map<PollinatorSpreadMode, String> getPollinatorSpreadModes() {
        return getEnumAsMap(PollinatorSpreadMode.values());
    }

    public Map<CropCyclePhaseType, String> getCropCyclePhaseTypes() {
        return getEnumAsMap(CropCyclePhaseType.values());
    }
    
    public Map<WeedType, String> getWeedTypes() {
        return getEnumAsMap(WeedType.values());
    }

    public String getCampaigns() {
        String result = "";
        if (practicedSystem != null) {
            result = practicedSystem.getCampaigns();
        }
        return getGson().toJson(result);
    }
    
    public String getGrowingSystemIdJson(){
        return getGson().toJson(growingSystemId);
    }
    
    public Map<ActionType, String> getActionTypes(){
        return getEnumAsMap(ActionType.values());
    }

    public void setPracticedPerennialCropCycleDtosJson(String json) {
        Type type = new TypeToken<List<PracticedPerennialCropCycleDto>>() {}.getType();
        this.practicedPerennialCropCycleDtos = getGson().fromJson(json, type);
    }

    public void setPracticedSeasonalCropCycleDtosJson(String json) {
        Type type = new TypeToken<List<PracticedSeasonalCropCycleDto>>() {}.getType();
        this.practicedSeasonalCropCycleDtos = getGson().fromJson(json, type);
    }

    public List<PracticedPerennialCropCycleDto> getPracticedPerennialCropCycleDtos() {
        return practicedPerennialCropCycleDtos;
    }

    public String getRefActionAgrosystTravailEDIsJson(){
        return getGson().toJson(refActionAgrosystTravailEDIs);
    }

    public Map<String, List<CroppingPlanSpeciesDto>> getGrowingSystemsCodeToSpecies() {
        return growingSystemsCodeToSpecies;
    }

    public List<CropCycleModelDto> getSeasonalGraphMainCrops() {
        return seasonalGraphMainCrops;
    }

    public List<CropCycleModelDto> getSeasonalGraphMainIntermediateCrops() {
        return seasonalGraphMainIntermediateCrops;
    }

    public List<PracticedSeasonalCropCycleDto> getPracticedSeasonalCropCycleDtos() {
        return practicedSeasonalCropCycleDtos;
    }

    public List<ToolsCouplingDto> getToolsCouplings() {
        return toolsCouplingsDtos;
    }
}
