package fr.inra.agrosyst.web.actions.networks;

/*
 * #%L
 * Agrosyst :: Web
 * $Id: NetworksEdit.java 950 2013-08-19 09:57:28Z dcosse $
 * $HeadURL: https://forge.codelutin.com/svn/agrosyst/tags/agrosyst-0.4.1/agrosyst-web/src/main/java/fr/inra/agrosyst/web/actions/networks/NetworksEdit.java $
 * %%
 * Copyright (C) 2013 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.lang.reflect.Type;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.apache.struts2.convention.annotation.Action;
import org.apache.struts2.convention.annotation.Result;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.gson.reflect.TypeToken;
import com.opensymphony.xwork2.Preparable;

import fr.inra.agrosyst.api.entities.Entities;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.Network;
import fr.inra.agrosyst.api.entities.NetworkImpl;
import fr.inra.agrosyst.api.entities.NetworkManager;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemService;
import fr.inra.agrosyst.api.services.network.NetworkManagerDto;
import fr.inra.agrosyst.api.services.network.NetworkService;
import fr.inra.agrosyst.api.services.users.UserDto;
import fr.inra.agrosyst.api.services.users.UserFilter;
import fr.inra.agrosyst.api.services.users.UserService;
import fr.inra.agrosyst.web.actions.AbstractAgrosystAction;

/**
 * @author cosse
 *
 */
public class NetworksEdit extends AbstractAgrosystAction implements Preparable {

    private static final long serialVersionUID = 6310375684340888705L;

    protected NetworkService networkService;
    
    protected static UserService userService;
    
    protected GrowingSystemService growingSystemService;
    
    public void setNetworkService(NetworkService networkService) {
        this.networkService = networkService;
    }
    
    public void setUserService(UserService userService) {
        this.userService = userService;
    }
    
    public void setGrowingSystemService(GrowingSystemService growingSystemService) {
        this.growingSystemService = growingSystemService;
    }
    
    protected String networkTopiaId;
    
    protected Network network;
    
    protected List<NetworkManagerDto> managersDto;
            
    protected ResultList<UserDto> users;
    
    protected List<GrowingSystem> growingSystemNetworkMember;
        
    protected List<String> parentsIds;
    
    protected Iterable<NetworkManagerDto> activeNetworkManagers;
    
    protected List<NetworkManagerDto> networkManagerDtos;
    
    protected static final Function<NetworkManager, NetworkManagerDto> NETWORK_MANAGER_TO_DTO = new Function<NetworkManager, NetworkManagerDto>() {
        @Override
        public NetworkManagerDto apply(NetworkManager networkManager) {
            String userTopiaId = networkManager.getUser().getTopiaId();
            UserDto userDto = userService.getUser(userTopiaId);
            NetworkManagerDto result = new NetworkManagerDto(
                    networkManager.getTopiaId(), 
                    networkManager.getFromDate(), 
                    networkManager.getToDate(), 
                    networkManager.isActive(),
                    userDto);
            return result;
        }
    };
    
    protected static final Predicate<NetworkManagerDto> IS_NETWORK_MANAGER_ACTIVE = new Predicate<NetworkManagerDto>() {
        
        @Override
        public boolean apply(NetworkManagerDto networkManager) {
            return networkManager.isActive();
        }
    };
    
    @Override
    public void prepare() throws Exception {
        if (StringUtils.isBlank(networkTopiaId)){// new network
            network = networkService.newNetwork();
        } else {
            network = networkService.getNetwork(networkTopiaId);
        }
    }

    protected void initForInput() {
        UserFilter userFilter = new UserFilter();
        userFilter.setActive(true);
        users = userService.getFilteredUsers(userFilter);
    }
    
    @Override
    public void validate() {
        if (StringUtils.isBlank(network.getName())){
            addFieldError("name", "Le nom du réseau est obligatoire.");
        }
        if (networkManagerDtos.isEmpty()){
            addFieldError("user", "Un réseau doit avoir au moins un responsable.");
        }
        if (researchParentsCycleProcess()) {
            addFieldError("parent", "Cycle détecté, un réseau ne peut avoir lui-même comme parent ou un réseau parent ayant ce réseau comme parent.");
        }
        
        if (hasErrors()) {
            initForInput();
        }
    }
    
    
    protected Boolean researchParentsCycleProcess() {
        Boolean result = false;
        for (String parentId : parentsIds) {
            Network parent = networkService.getNetwork(parentId);
            researchCycle(parent);
        }
        return result;
    }
    
    protected Boolean researchCycle(Network parent){
        Boolean result = false;

        if (networkTopiaId == parent.getTopiaId()) {
            result = true;
        } else {
            for (Network grandParent : parent.getParents()){
                result = researchCycle(grandParent);
            }
        }
        return result;
    }
    
    @Override
    public String input() {
        if (!StringUtils.isBlank(networkTopiaId)){
            managersDto = Lists.transform(network.getManagers(), NETWORK_MANAGER_TO_DTO);
            activeNetworkManagers = Iterables.filter(managersDto, IS_NETWORK_MANAGER_ACTIVE);
            growingSystemNetworkMember = growingSystemService.findAllByNetwork(network);
        }
        initForInput();
        return INPUT; 
    }
    

    @Override
    @Action(results = {@Result(type = "redirectAction", params = {"actionName", "networks-list"})})
    public String execute() throws Exception {
        
        Map<NetworkManager, String> mapNetworkManagerUser = propagateNetworkManagersChanges();
        networkService.updateNetwork(network, mapNetworkManagerUser, parentsIds);
        return SUCCESS;
    }
    
    
    public Map<NetworkManager, String> propagateNetworkManagersChanges() {
        Map<NetworkManager, String> mapNetworkManagerUser = new HashMap<NetworkManager, String>();
        List<NetworkManager> networkManagers = network.getManagers();
        if (networkManagers == null) {
            networkManagers = Lists.newArrayList();
            network.setManagers(networkManagers);
        }

        Map<String, NetworkManager> indexNetworkManagers = Maps.uniqueIndex(networkManagers, Entities.GET_TOPIA_ID);

        networkManagers.clear(); // On vide la liste pour mieux la remplir

        if (networkManagerDtos != null) {
            for (NetworkManagerDto networkManagerDto : networkManagerDtos) {
                String topiaId = networkManagerDto.getTopiaId();
                NetworkManager networkManager = null;
                if (Strings.isNullOrEmpty(topiaId)) {
                    // nouvelle donnée gps
                    networkManager = networkService.newNetworkManager();
                } else {
                    // donnée gps déjà existante
                    networkManager = indexNetworkManagers.get(topiaId);
                    Preconditions.checkState(networkManager != null, "Invalid topiaId for the given entity: " + topiaId);
                }
                propagateNetworkManagerChange(networkManager, networkManagerDto);
                networkManagers.add(networkManager);
                mapNetworkManagerUser.put(networkManager, networkManagerDto.getUser().getTopiaId());
            }
        }
        return mapNetworkManagerUser;
    }
    
    protected void propagateNetworkManagerChange(NetworkManager entity, NetworkManagerDto dto) {
        // at that time the user is not added because not available.
        entity.setActive(dto.getActive());
        entity.setFromDate(dto.getFromDate());
        entity.setToDate(dto.getToDate());
    }
    

    public String getNetworkTopiaId() {
        return networkTopiaId;
    }

    public void setNetworkTopiaId(String networkTopiaId) {
        this.networkTopiaId = networkTopiaId;
    }

    public List<String> getParentsIds() {
        return parentsIds;
    }

    public void setParents(List<String> parentsIds) {
        this.parentsIds = parentsIds;
    }
    
    public void setParentsJson(String json) {
        Type type = new TypeToken<List<String>>() {}.getType();
        this.parentsIds = getGson().fromJson(json, type);
    }

    public Network getNetwork() {
        if (network == null) {
            // AThimel 26/06/2013 Fais chier de devoir écrire ça, mais c'est la seule option pour ne pas avoir une grosse dose d'exceptions avec du paramsPrepareParams
            return new NetworkImpl();
        }
        return network;
    }
    
    public String getNetworkJson() {
        Network result = getNetwork();
        return getGson().toJson(result);
    }

    public void setNetwork(Network network) {
        this.network = network;
    }

    public List<NetworkManagerDto> getManagersDto() {
        return managersDto;
    }
    
    public String getManagersDtoJson() {
        return getGson().toJson(managersDto);
    }

    public void setManagersDto(List<NetworkManagerDto> managersDto) {
        this.managersDto = managersDto;
    }

    public ResultList<UserDto> getUsers() {
        return users;
    }

    public String getUsersJson() {
        return getGson().toJson(users.getElements());
    }
    
    public void setUsers(ResultList<UserDto> users) {
        this.users = users;
    }
    
    public void setNetworkManagerDtoJson(String json) {
        Type type = new TypeToken<List<NetworkManagerDto>>() {}.getType();
        this.networkManagerDtos = getGson().fromJson(json, type);
    }

    public List<GrowingSystem> getGrowingSystemNetworkMember() {
        return growingSystemNetworkMember;
    }

    public void setGrowingSystemNetworkMember(
            List<GrowingSystem> growingSystemNetworkMember) {
        this.growingSystemNetworkMember = growingSystemNetworkMember;
    }

}
