package fr.inra.agrosyst.web.actions.domains;

/*
 * #%L
 * Agrosyst :: Web
 * $Id: DomainsEdit.java 1109 2013-09-02 16:27:32Z echatellier $
 * $HeadURL: https://forge.codelutin.com/svn/agrosyst/tags/agrosyst-0.4.1/agrosyst-web/src/main/java/fr/inra/agrosyst/web/actions/domains/DomainsEdit.java $
 * %%
 * Copyright (C) 2013 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.apache.struts2.convention.annotation.Action;
import org.apache.struts2.convention.annotation.Result;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.gson.reflect.TypeToken;
import com.opensymphony.xwork2.Preparable;

import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.DomainImpl;
import fr.inra.agrosyst.api.entities.DomainType;
import fr.inra.agrosyst.api.entities.Entities;
import fr.inra.agrosyst.api.entities.GPSData;
import fr.inra.agrosyst.api.entities.InterventionType;
import fr.inra.agrosyst.api.entities.Materiel;
import fr.inra.agrosyst.api.entities.Sol;
import fr.inra.agrosyst.api.entities.ToolsCoupling;
import fr.inra.agrosyst.api.entities.Zoning;
import fr.inra.agrosyst.api.entities.referentiels.RefLegalStatus;
import fr.inra.agrosyst.api.entities.referentiels.RefLocation;
import fr.inra.agrosyst.api.entities.referentiels.RefMateriel;
import fr.inra.agrosyst.api.entities.referentiels.RefSolArvalis;
import fr.inra.agrosyst.api.services.domain.CroppingPlanEntryDto;
import fr.inra.agrosyst.api.services.domain.DomainService;
import fr.inra.agrosyst.api.services.domain.ToolsCouplingDto;
import fr.inra.agrosyst.api.services.referentiels.MaterielType;
import fr.inra.agrosyst.api.services.referentiels.ReferentielService;
import fr.inra.agrosyst.web.actions.AbstractAgrosystAction;

/**
 * Action d'édition d'un domaine.
 *
 * @author Eric Chatellier
 */
public class DomainsEdit extends AbstractAgrosystAction implements Preparable {

    private static final String REQUIRED_FIELD = "Champ obligatoire";

    private static final long serialVersionUID = 390873886885798716L;

    protected static final Function<Sol, SolDto> SOL_TO_DTO = new Function<Sol, SolDto>() {
        @Override
        public SolDto apply(Sol input) {
            RefSolArvalis refSolArvalis = input.getRefSolArvalis();
            SolDto result = new SolDto(
                    input.getTopiaId(),
                    input.getName(),
                    input.getComment(),
                    input.getImportance()
            );
            result.setSolArvalisTopiaId(refSolArvalis.getTopiaId());
            result.setSolNom(refSolArvalis.getSol_nom());
            result.setSolTexture(refSolArvalis.getSol_texture());
            result.setSolCalcaire(refSolArvalis.getSol_calcaire());
            result.setSolProfondeur(refSolArvalis.getSol_profondeur());
            result.setSolHydromorphie(refSolArvalis.getSol_hydromorphie());
            result.setSolPierrosite(refSolArvalis.getSol_pierrosite());
            result.setSolRegion(refSolArvalis.getSol_region());
            result.setSolRegionCode(refSolArvalis.getSol_region_code());
            return result;
        }
    };

    protected ReferentielService referentielService;

    protected DomainService domainService;

    protected Domain domain;

    protected List<Domain> relatedDomains;

    protected List<GPSDataDto> gpsDataDtos;

    protected String domainTopiaId;

    protected String commune;

    protected String legalStatusId;

    protected List<RefLegalStatus> allRefLegalStatus;

    protected RefLocation location;

    protected String departement;
    protected String petiteRegionAgricole;
    protected String petiteRegionAgricoleName;

    protected String locationTopiaId;

    protected List<MaterielDto> domainMaterielDtos;
    
    protected List<ToolsCouplingDto> toolsCouplingDtos;

    protected Map<MaterielType, List<String>> materielType1List;

    protected Map<Integer, String> solArvalisRegions;

    protected List<SolDto> solDtos;

    protected List<CroppingPlanEntryDto> croppingPlan;
    
    protected Map<String, Materiel> materiels;
    
    protected InterventionType selectedInterventionType;
    
    protected Integer otex18;
    
    protected Integer otex70;
    
    protected Map<Integer, String> otex18s;

    public void setReferentielService(ReferentielService referentielService) {
        this.referentielService = referentielService;
    }

    public void setDomainService(DomainService domainService) {
        this.domainService = domainService;
    }

    @Override
    public void prepare() throws Exception {

        if (StringUtils.isEmpty(domainTopiaId)) {
            // Cas de création d'un domain
            domain = domainService.newDomain();
            domain.setCampaign(getNavigationContext().getCampaigns().iterator().next());
        } else {

            // Cas d'une mise à jour de domain
            domain = domainService.getDomain(domainTopiaId);

            if (StringUtils.isBlank(locationTopiaId)) {
                // Cas ou la commune n'a pas étée modifiée
                locationTopiaId = domain.getLocation().getTopiaId();

                // Dans le cas ou il y a eu modification du choix de la commune
                location = referentielService.getRefLocation(locationTopiaId);

                departement = location.getDepartement();

                petiteRegionAgricole = location.getPetiteRegionAgricoleCode();
                petiteRegionAgricoleName = location.getPetiteRegionAgricoleNom();
            }
        }
    }

    public Domain getDomain() {
        if (domain == null) {
            // AThimel 26/06/2013 Fais chier de devoir écrire ça, mais c'est la seule option pour ne pas avoir une grosse dose d'exceptions avec du paramsPrepareParams
            return new DomainImpl();
        }
        return domain;
    }

    @Override
    protected void initForInput() {
        allRefLegalStatus = domainService.getAllRefLegalStatus();

        materielType1List = referentielService.getTypeMateriel1List();

        solArvalisRegions = referentielService.getSolArvalisRegions();
        
        otex18s = referentielService.getOtex18s();

        if (StringUtils.isNotEmpty(domainTopiaId)) {
            relatedDomains = domainService.getRelatedDomains(domain);
        }
    }

    @Override
    public String input() throws Exception {
        initForInput();

        if (StringUtils.isNotEmpty(domainTopiaId)) {

            gpsDataDtos = Lists.transform(domain.getGpsDatas(), GpsDatas.GPS_DATA_TO_DTO);
            domainMaterielDtos = Lists.transform(domain.getMateriels(), Materiels.MATERIEL_TO_DTO);
            solDtos = Lists.transform(domain.getSols(), SOL_TO_DTO);
            toolsCouplingDtos = Lists.transform(domain.getToolsCoupling(), Materiels.TOOLS_COUPLING_TO_DTO);
            croppingPlan = domainService.getCroppingPlan(domainTopiaId);
        } else {
            gpsDataDtos = Collections.emptyList();
            domainMaterielDtos = Collections.emptyList();
            solDtos = Collections.emptyList();
            croppingPlan = Collections.emptyList();
        }

        return INPUT;
    }

    @Override
    public void validate() {
        // campagne
        if (domain.getCampaign() == 0) {
            addFieldError("domain.campaign", REQUIRED_FIELD);
        }

        // nom du domain
        if (StringUtils.isBlank(domain.getName())) {
            addFieldError("domain.name", REQUIRED_FIELD);
        }

        // nom de l'interlocuteur principal
        if (StringUtils.isBlank(domain.getMainContact())) {
            addFieldError("domain.mainContact", REQUIRED_FIELD);
        }

        // commune
        if (StringUtils.isBlank(locationTopiaId)) {
            addFieldError("commune", REQUIRED_FIELD);
        }

        // validation des materiels
        // FIXME echatellier 20130724 TODO

        // validation des sols
        // FIXME echatellier 20130724 TODO

        // TODO AThimel 01/08/13 validation de l'assolement

        if (hasErrors()) {
            initForInput();
        }
    }

    @Override
    @Action(results = {@Result(type = "redirectAction", params = {"actionName", "domains-list"})})
    public String execute() throws Exception {

        propagateGpsDatasChanges();
        propagateSolChanges();
        propagateMaterielChanges();
        propagateToolsCouplingChanges();

        if (StringUtils.isBlank(domain.getTopiaId())) {
            domain = domainService.updateDomain(domain, locationTopiaId, legalStatusId, croppingPlan, otex18, otex70);
            notificationSupport.domainCreated(domain);
        } else {
            domain = domainService.updateDomain(domain, locationTopiaId, legalStatusId, croppingPlan, otex18, otex70);
            notificationSupport.domainUpdated(domain);
        }
        return SUCCESS;
    }

    /**
     * Propagate changes on domains's GpsDatas.
     */
    protected void propagateGpsDatasChanges() {
        List<GPSData> gpsDatas = getDomain().getGpsDatas();
        if (gpsDatas == null) {
            gpsDatas = Lists.newArrayList();
            domain.setGpsDatas(gpsDatas);
        }

        Iterables.removeIf(gpsDataDtos, GpsDatas.IS_INVALID_GPS_DATA);

        Map<String, GPSData> indexGpsDatas = Maps.uniqueIndex(gpsDatas, Entities.GET_TOPIA_ID);

        gpsDatas.clear(); // On vide la liste pour mieux la remplir

        if (gpsDataDtos != null) {
            for (GPSDataDto gpsDataDto : gpsDataDtos) {
                String topiaId = gpsDataDto.getTopiaId();
                GPSData gpsData;
                if (Strings.isNullOrEmpty(topiaId)) {
                    // nouvelle donnée gps
                    gpsData = domainService.newGpsData();
                } else {
                    // donnée gps déjà existante
                    gpsData = indexGpsDatas.get(topiaId);
                    Preconditions.checkState(gpsData != null, "Invalid topiaId for the given entity: " + topiaId);
                }
                propagateGpsDataChanges(gpsData, gpsDataDto);
                gpsDatas.add(gpsData);
            }
        }
    }

    /**
     * Propagate the GPSData's changes according the gpsDataDto.
     *
     * @param entity the gpsData
     * @param dto    the gpsDataDto
     */
    protected void propagateGpsDataChanges(GPSData entity, GPSDataDto dto) {
        entity.setName(dto.getName());
        entity.setLatitude(dto.getLatitude());
        entity.setLongitude(dto.getLongitude());
        entity.setDescription(dto.getDescription());
    }

    /**
     * Convert submited DTO list to up to date entity list.
     *
     * @return current entities list
     * @throws Exception
     */
    protected void propagateMaterielChanges() throws Exception {

        List<Materiel> domainMateriels = getDomain().getMateriels();
        List<Materiel> nonDeleted = Lists.newArrayList();
        if (domainMateriels == null) {
            domainMateriels = Lists.newArrayList();
            getDomain().setMateriels(domainMateriels);
        }

        // update list with dto
        Map<String, Materiel> domainMaterielMap = Maps.uniqueIndex(domainMateriels, Entities.GET_TOPIA_ID);

        if (domainMaterielDtos != null) {
            materiels = new HashMap<String, Materiel>();
            for (MaterielDto domainMaterielDto : domainMaterielDtos) {
                // find  entity instance
                Materiel domainMateriel = null;
                String topiaId = domainMaterielDto.getTopiaId();
                if (StringUtils.isNotBlank(topiaId)) {
                    domainMateriel = domainMaterielMap.get(topiaId);
                } else {
                    domainMateriel = domainService.newMateriel();
                }
                materiels.put(domainMaterielDto.getHashKey(), domainMateriel);
                // overwrite entity instance
                domainMateriel.setName(domainMaterielDto.getName());
                domainMateriel.setDescription(domainMaterielDto.getDescription());
                domainMateriel.setMaterielETA(domainMaterielDto.isMaterielETA());

                // manual materiel management
                String materielTopiaId = domainMaterielDto.getMaterielTopiaId();
                if (StringUtils.isNotBlank(materielTopiaId)) {
                    RefMateriel refMateriel = referentielService.findMateriel(domainMaterielDto.getMaterielTopiaId());
                    domainMateriel.setRefMateriel(refMateriel);
                } else {
                    domainMateriel.setRefMateriel(null);
                }

                if (StringUtils.isBlank(topiaId)) {
                    domainMateriels.add(domainMateriel);
                }

                nonDeleted.add(domainMateriel);
            }
        }

        domainMateriels.retainAll(nonDeleted);
    }
    
    protected void propagateToolsCouplingChanges() {

        List<ToolsCoupling> toolsCouplings = getDomain().getToolsCoupling();
        List<ToolsCoupling> nonDeleted = Lists.newArrayList();
        if (toolsCouplings == null) {
            toolsCouplings = Lists.newArrayList();
            getDomain().setToolsCoupling(toolsCouplings);
        }

        // update list with dto
        Map<String, ToolsCoupling> toolsCouplingsMap = Maps.uniqueIndex(toolsCouplings, Entities.GET_TOPIA_ID);

        if (toolsCouplingDtos != null) {
            for (ToolsCouplingDto toolsCouplingDto : toolsCouplingDtos) {
                // find  entity instance
                ToolsCoupling toolsCoupling = null;
                String topiaId = toolsCouplingDto.getTopiaId();
                if (StringUtils.isNotBlank(topiaId)) {
                    toolsCoupling = toolsCouplingsMap.get(topiaId);
                } else {
                    toolsCoupling = domainService.newToolsCoupling();
                }
                // overwrite entity instance
                Materiel tractor = materiels.get(toolsCouplingDto.getTractorKey());
                List<Materiel> equipements = new ArrayList<Materiel>();
                if (toolsCouplingDto.getEquipementsKeys() != null) {
                    for (String equipementKey : toolsCouplingDto.getEquipementsKeys()) {
                        equipements.add(materiels.get(equipementKey));
                    }
                }
                toolsCoupling.setTractor(tractor);
                toolsCoupling.setCouplingEquipements(equipements);
                toolsCoupling.setComment(toolsCouplingDto.getComment());
                toolsCoupling.setFlow(toolsCouplingDto.getFlow());
                toolsCoupling.setInterventionName(toolsCouplingDto.getInterventionName());
                toolsCoupling.setInterventionType(toolsCouplingDto.getInterventionType());
                toolsCoupling.setWorkforce(toolsCouplingDto.getWorkforce());

                if (StringUtils.isBlank(topiaId)) {
                    toolsCouplings.add(toolsCoupling);
                }

                nonDeleted.add(toolsCoupling);
            }
        }

        toolsCouplings.retainAll(nonDeleted);
    }

    protected void propagateSolChanges() {
        List<Sol> domainSols = getDomain().getSols();
        List<Sol> nonDeleted = Lists.newArrayList();
        if (domainSols == null) {
            domainSols = Lists.newArrayList();
            getDomain().setSols(domainSols);
        }

        // update list with dto
        Map<String, Sol> domainSolMap = Maps.uniqueIndex(domainSols, Entities.GET_TOPIA_ID);

        if (solDtos != null) {
            for (SolDto solDto : solDtos) {
                String topiaId = solDto.getTopiaId();
                Sol sol;
                if (Strings.isNullOrEmpty(topiaId)) {
                    // nouvelle période d'activité
                    sol = domainService.newSol();
                } else {
                    // période d'activité déjà existante
                    sol = domainSolMap.get(topiaId);
                }

                // propagate changes
                sol.setName(solDto.getName());
                sol.setComment(solDto.getComment());
                RefSolArvalis refSolArvalis = referentielService.findSolArvalis(solDto.getSolArvalisTopiaId());
                sol.setRefSolArvalis(refSolArvalis);

                // can't do this before findSolArvalis, otherwize hibernate complains !!!
                if (Strings.isNullOrEmpty(topiaId)) {
                    domainSols.add(sol);
                }
                nonDeleted.add(sol);
            }
        }

        domainSols.retainAll(nonDeleted);
    }

    /**
     * Get all the type that a a domain could be.
     *
     * @return all the type
     */
    public Map<DomainType, String> getTypes() {
        return getEnumAsMap(DomainType.values());
    }
    
    
    public Map<InterventionType, String> getDomainInterventionTypes() {
        return getEnumAsMap(InterventionType.values());
    }
    /**
     * Get all the interventionType
     * @return all the interventionType
     */
    public String getInterventionTypes() {
        Map<InterventionType, String> interventionTypes = getEnumAsMap(InterventionType.values());
        String interventionTypesJson = getGson().toJson(interventionTypes);
        return interventionTypesJson;
    }

    /**
     * Get all the VulnerableArea that a a domain could be.
     *
     * @return all the VulnerableArea
     */
    public Map<Zoning, String> getZoningValues() {
        return getEnumAsMap(Zoning.values());
    }

    public void setLocationTopiaId(String locationTopiaId) {
        this.locationTopiaId = locationTopiaId;
    }

    public String getDepartement() {
        return departement;
    }

    public String getFormatedDepartement() {
        if (StringUtils.isBlank(departement)) {
            return "";
        }
        String depNumber = Strings.padStart(departement, 2, '0');
        String key = "departement." + depNumber;
        String result = getText(key);
        result = result + " (" + depNumber + ")";
        return result;
    }

    public String getFormatedDepartementJson() {
        return getGson().toJson(getFormatedDepartement());
    }

    public String getPetiteRegionAgricole() {
        return petiteRegionAgricole;
    }

    public String getPetiteRegionAgricoleName() {
        return petiteRegionAgricoleName;
    }

    public String getFormatedPetiteRegionAgricoleName() {
        if (StringUtils.isBlank(petiteRegionAgricoleName)) {
            return "";
        }
        String result = petiteRegionAgricoleName + " (" + petiteRegionAgricole + ")";
        return result;
    }

    public String getFormatedPetiteRegionAgricoleNameJson() {
        return getGson().toJson(getFormatedPetiteRegionAgricoleName());
    }

    public String getGpsDataDtosJson() {
        return getGson().toJson(gpsDataDtos);
    }

    public void setGpsDataDtosJson(String gpsDataDtos) {
        Type type = new TypeToken<List<GPSDataDto>>() {
        }.getType();
        this.gpsDataDtos = getGson().fromJson(gpsDataDtos, type);
    }

    public void setDomainTopiaId(String domainTopiaId) {
        this.domainTopiaId = domainTopiaId;
    }

    public String getCommune() {
        return commune;
    }

    public void setCommune(String commune) {
        this.commune = commune;
    }

    public String getLegalStatusId() {
        return legalStatusId;
    }

    public void setLegalStatusId(String legalStatusId) {
        this.legalStatusId = legalStatusId;
    }

    public List<RefLegalStatus> getAllRefLegalStatus() {
        return allRefLegalStatus;
    }
    
    public String getAllRefLegalStatusJson() {
        return getGson().toJson(allRefLegalStatus);
    }

    public String getLocationTopiaId() {
        return locationTopiaId;
    }

    public String getDomainMaterielDtosJson() {
        return getGson().toJson(domainMaterielDtos);
    }

    public void setDomainMaterielDtosJson(String json) {
        Type type = new TypeToken<List<MaterielDto>>() {
        }.getType();
        this.domainMaterielDtos = getGson().fromJson(json, type);
    }

    public String getToolsCouplingDtosJson() {
        return getGson().toJson(toolsCouplingDtos);
    }

    public void setToolsCouplingDtosJson(String json) {
        Type type = new TypeToken<List<ToolsCouplingDto>>() {
        }.getType();
        this.toolsCouplingDtos = getGson().fromJson(json, type);
    }

    public String getMaterielType1ListJson() {
        return getGson().toJson(materielType1List);
    }

    public String getSolDtosJson() {
        return getGson().toJson(solDtos);
    }

    public void setSolDtosJson(String json) {
        Type type = new TypeToken<List<SolDto>>() {
        }.getType();
        this.solDtos = getGson().fromJson(json, type);
    }

    public Map<Integer, String> getSolArvalisRegions() {
        return solArvalisRegions;
    }

    public List<Domain> getRelatedDomains() {
        return relatedDomains;
    }

    public List<CroppingPlanEntryDto> getCroppingPlan() {
        return croppingPlan;
    }

    public void setCroppingPlanJson(String json) {
        Type type = new TypeToken<List<CroppingPlanEntryDto>>() {
        }.getType();
        this.croppingPlan = getGson().fromJson(json, type);
    }

    public void setSelectedInterventionType(
            InterventionType selectedInterventionType) {
        this.selectedInterventionType = selectedInterventionType;
    }

    public Integer getOtex18() {
        return otex18;
    }

    public void setOtex18(Integer otex18) {
        this.otex18 = otex18;
    }

    public Map<Integer, String> getOtex18s() {
        return otex18s;
    }

    public void setOtex70(Integer otex70) {
        this.otex70 = otex70;
    }
}
