package fr.inra.agrosyst.web.actions;

/*
 * #%L
 * Agrosyst :: Web
 * $Id: AbstractAgrosystAction.java 1186 2013-09-06 14:46:40Z athimel $
 * $HeadURL: https://forge.codelutin.com/svn/agrosyst/tags/agrosyst-0.4.1/agrosyst-web/src/main/java/fr/inra/agrosyst/web/actions/AbstractAgrosystAction.java $
 * %%
 * Copyright (C) 2013 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.io.IOException;
import java.util.Calendar;
import java.util.Map;
import java.util.Set;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.struts2.interceptor.ParameterAware;
import org.apache.struts2.interceptor.ServletRequestAware;
import org.apache.struts2.interceptor.ServletResponseAware;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import com.google.gson.ExclusionStrategy;
import com.google.gson.FieldAttributes;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonSyntaxException;
import com.google.gson.TypeAdapter;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonToken;
import com.google.gson.stream.JsonWriter;
import com.opensymphony.xwork2.ActionSupport;

import fr.inra.agrosyst.api.NavigationContext;
import fr.inra.agrosyst.api.entities.CropCyclePerennialSpecies;
import fr.inra.agrosyst.api.entities.CropCyclePerennialSpeciesAbstract;
import fr.inra.agrosyst.api.entities.CroppingPlanSpecies;
import fr.inra.agrosyst.api.entities.CroppingPlanSpeciesAbstract;
import fr.inra.agrosyst.api.entities.PracticedSeasonalCropCycle;
import fr.inra.agrosyst.api.entities.PracticedSeasonalCropCycleAbstract;
import fr.inra.agrosyst.api.services.context.NavigationContextService;
import fr.inra.agrosyst.web.AgrosystWebConfig;
import fr.inra.agrosyst.web.AgrosystWebNotification;
import fr.inra.agrosyst.web.AgrosystWebNotificationSupport;
import fr.inra.agrosyst.web.AgrosystWebSession;

/**
 * Toutes les actions Struts doivent hériter de cette classe.
 *
 * @author Arnaud Thimel <thimel@codelutin.com>
 */
public abstract class AbstractAgrosystAction extends ActionSupport implements ServletRequestAware, ServletResponseAware,
        ParameterAware {

    private static final long serialVersionUID = 4829352350362628085L;

    protected static final Multimap<Class<?>, String> GSON_EXCLUSIONS = HashMultimap.create();
    public static final GsonBuilder GSON_BUILDER = new GsonBuilder();

    protected static final TypeAdapter<Integer> INTEGER_ADAPTER = new TypeAdapter<Integer>() {
        @Override
        public void write(JsonWriter out, Integer value) throws IOException {
            out.value(value);
        }

        @Override
        public Integer read(JsonReader in) throws IOException {
            if (in.peek() == JsonToken.NULL) {
                in.nextNull();
                return null;
            }
            try {
                String result = in.nextString();
                if ("".equals(result)) {
                    return null;
                }
                return Integer.valueOf(result);
            } catch (NumberFormatException e) {
                throw new JsonSyntaxException(e);
            }
        }
    };

    /**
     * Add class attributes exclusion to prevent stackoverflow exception during json serialisation.
     */
    static {
        GSON_EXCLUSIONS.put(CroppingPlanSpeciesAbstract.class, CroppingPlanSpecies.PROPERTY_CROPPING_PLAN_ENTRY);
        GSON_EXCLUSIONS.put(PracticedSeasonalCropCycleAbstract.class, PracticedSeasonalCropCycle.PROPERTY_CROP_CYCLE_NODES);
        GSON_EXCLUSIONS.put(CropCyclePerennialSpeciesAbstract.class, CropCyclePerennialSpecies.PROPERTY_CYCLE);

        GSON_BUILDER.registerTypeAdapterFactory(HibernateProxyTypeAdapter.FACTORY);
        GSON_BUILDER.registerTypeAdapter(Integer.class, INTEGER_ADAPTER);
        GSON_BUILDER.setDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'");
        GSON_BUILDER.addSerializationExclusionStrategy(new ExclusionStrategy() {
            @Override
            public boolean shouldSkipField(FieldAttributes f) {
                // TODO AThimel 06/08/13 Maybe another Multimap implementation will do the job ?
                Class<?> declaringClass = f.getDeclaringClass();
                String attributeName = f.getName();
                boolean result = GSON_EXCLUSIONS.containsEntry(declaringClass, attributeName);
                return result;
            }

            @Override
            public boolean shouldSkipClass(Class<?> clazz) {
                return false;
            }
        });
    }

    // Injectés par fr.inra.agrosyst.web.AgrosystWebInterceptor
    protected AgrosystWebConfig config;
    protected AgrosystWebSession session;
    protected AgrosystWebNotificationSupport notificationSupport;
    protected NavigationContextService navigationContextService;

    protected HttpServletRequest servletRequest;
    protected HttpServletResponse servletResponse;

    protected Map<String, String[]> parameters;

    protected RichNavigationContext richNavigationContext;

    /**
     * gson (can be used into non json action to initialize json data).
     */
    protected Gson gson;

    public Gson getGson() {
        if (gson == null) {
            gson = GSON_BUILDER.create();
        }
        return gson;
    }

    public String toJson(Object element) {
        return getGson().toJson(element);
    }

    @Override
    public void setServletRequest(HttpServletRequest servletRequest) {
        this.servletRequest = servletRequest;
    }

    @Override
    public void setServletResponse(HttpServletResponse servletResponse) {
        this.servletResponse = servletResponse;
    }

    @Override
    public void setParameters(Map<String, String[]> parameters) {
        this.parameters = parameters;
    }

    public void setConfig(AgrosystWebConfig config) {
        this.config = config;
    }

    public void setSession(AgrosystWebSession session) {
        this.session = session;
    }

    public void setNavigationContextService(NavigationContextService navigationContextService) {
        this.navigationContextService = navigationContextService;
    }

    public void setNotificationSupport(AgrosystWebNotificationSupport notificationSupport) {
        this.notificationSupport = notificationSupport;
    }

    // Ne pas supprimer, méthode utilisée depuis les JSP
    public AgrosystWebConfig getConfig() {
        return config;
    }

    // Ne pas supprimer, méthode utilisée depuis les JSP
    public AgrosystWebSession getSession() {
        return session;
    }

    protected NavigationContext getNavigationContext() {
        NavigationContext navigationContext = session.getNavigationContext();

        // 'dirty' permet de savoir si les données du contexte doivent être validées
        boolean dirty = false;
        if (navigationContext == null) {

            // Lecture du cookie
            navigationContext = readNavigationContextCookie();
            dirty = true;

            // Pas de cookie trouvé, création du contexte de navigation par défaut
            if (navigationContext == null) {
                navigationContext = new NavigationContext();
                dirty = false;
            }

            if (navigationContext.getCampaigns().isEmpty()) {
                int campaign = Calendar.getInstance().get(Calendar.YEAR);
                navigationContext.setCampaigns(Sets.newHashSet(campaign));
            }
        }

        if (dirty) {
            navigationContext = verifyAndSaveNavigationContext(navigationContext);
        }
        return navigationContext;
    }

    protected NavigationContext verifyAndSaveNavigationContext(NavigationContext navigationContext) {
        NavigationContext newNavigationContext = navigationContextService.verify(navigationContext);
        session.setNavigationContext(newNavigationContext);
        writeNavigationContextCookie(newNavigationContext);
        return newNavigationContext;
    }

    protected NavigationContext readNavigationContextCookie() {
        NavigationContext result = null;
        if (servletRequest != null) {
            Cookie[] cookies = servletRequest.getCookies();
            if (cookies != null) {
                for (Cookie cookie : cookies) {
                    if (AgrosystWebConfig.NAVIGATION_CONTEXT_COOKIE_NAME.equals(cookie.getName())) {
                        result = new Gson().fromJson(cookie.getValue(), NavigationContext.class);
                        break;
                    }
                }
            }
        }
        return result;
    }

    protected void writeNavigationContextCookie(NavigationContext navigationContext) {
        Cookie cookie = new Cookie(AgrosystWebConfig.NAVIGATION_CONTEXT_COOKIE_NAME, getGson().toJson(navigationContext));
        cookie.setPath(servletRequest.getContextPath());
        servletResponse.addCookie(cookie);
    }

    public RichNavigationContext getRichNavigationContext() {
        if (richNavigationContext == null) {
            NavigationContext navigationContext = getNavigationContext();
            richNavigationContext = new RichNavigationContext(navigationContext, navigationContextService);
        }
        return richNavigationContext;
    }

    /**
     * Transform enumeration values into map with i18n value for each enum value.
     * <p/>
     * i18n key is fqn.NAME
     *
     * @param values values to transform
     * @return map (enum value > i18n text)
     */
    protected <T> Map<T, String> getEnumAsMap(T[] values) {
        Map<T, String> valuesMap = Maps.newLinkedHashMap();
        for (T value : values) {
            String i18n = value.getClass().getName() + "." + value.toString();
            String trans = getText(i18n);
            valuesMap.put(value, trans);
        }
        return valuesMap;
    }

    protected void initForInput() {

    }

    /**
     * Return session info notification.
     * 
     * Warning : this get method clear resulting collection
     * 
     * @return session info notifications
     */
    public Set<AgrosystWebNotification> getInfoNotifications() {
        Set<AgrosystWebNotification> messages = Sets.newHashSet(session.getInfoNotifications());
        session.getInfoNotifications().clear();
        return messages;
    }

    public boolean isInfoNotificationsEmpty() {
        return session.getInfoNotifications().isEmpty();
    }
    
    /**
     * Return session warning notifications.
     * 
     * Warning : this get method clear resulting collection
     * 
     * @return session warning notifications
     */
    public Set<AgrosystWebNotification> getWarningNotifications() {
        Set<AgrosystWebNotification> messages = Sets.newHashSet(session.getWarningNotifications());
        session.getInfoNotifications().clear();
        return messages;
    }

    public boolean isWarningNotificationsEmpty() {
        return session.getWarningNotifications().isEmpty();
    }
    
    /**
     * Return session error notifications.
     * 
     * Warning : this get method clear resulting collection
     * 
     * @return session error notifications
     */
    public Set<AgrosystWebNotification> getErrorNotifications() {
        Set<AgrosystWebNotification> messages = Sets.newHashSet(session.getErrorNotifications());
        session.getErrorNotifications().clear();
        return messages;
    }

    public boolean isErrorNotificationsEmpty() {
        return session.getErrorNotifications().isEmpty();
    }
}
