package fr.inra.agrosyst.web;

/*
 * #%L
 * Agrosyst :: Web
 * $Id: AgrosystWebInterceptor.java 1109 2013-09-02 16:27:32Z echatellier $
 * $HeadURL: https://forge.codelutin.com/svn/agrosyst/tags/agrosyst-0.4.1/agrosyst-web/src/main/java/fr/inra/agrosyst/web/AgrosystWebInterceptor.java $
 * %%
 * Copyright (C) 2013 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.beans.PropertyDescriptor;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts2.StrutsStatics;
import org.nuiton.topia.TopiaContext;
import org.nuiton.util.beans.BeanUtil;
import org.nuiton.web.filter.TopiaTransactionFilter;

import com.google.common.base.Preconditions;
import com.google.common.base.Supplier;
import com.opensymphony.xwork2.ActionInvocation;
import com.opensymphony.xwork2.interceptor.Interceptor;

import fr.inra.agrosyst.api.services.AgrosystService;
import fr.inra.agrosyst.api.services.ServiceFactory;
import fr.inra.agrosyst.services.AgrosystServiceConfig;
import fr.inra.agrosyst.services.DefaultServiceContext;
import fr.inra.agrosyst.web.actions.AbstractAgrosystAction;

/**
 * Interceptor struts qui inject les parametres de type connu (Session, Service, Config) dans
 * l'action struts avant son execution.
 * 
 * @author Arnaud Thimel <thimel@codelutin.com>
 */
public class AgrosystWebInterceptor implements Interceptor {

    private static final long serialVersionUID = 2035088485913506328L;

    private static final Log log = LogFactory.getLog(AgrosystWebInterceptor.class);

    @Override
    public void init() {
        if (log.isInfoEnabled()) {
            log.info("Initializing Agrosyst Web Interceptor");
        }
    }

    @Override
    public String intercept(ActionInvocation invocation) throws Exception {

        Object action = invocation.getAction();

        if (action instanceof AbstractAgrosystAction) {

            ServiceFactory serviceFactory = newServiceFactory(invocation);

            AgrosystWebSession agrosystSession = getAgrosystSession(invocation);

            Set<PropertyDescriptor> descriptors =
                    BeanUtil.getDescriptors(
                            action.getClass(),
                            BeanUtil.IS_WRITE_DESCRIPTOR);

            
            for (PropertyDescriptor propertyDescriptor : descriptors) {

                Class<?> propertyType = propertyDescriptor.getPropertyType();

                Object toInject = null;

                if (AgrosystService.class.isAssignableFrom(propertyType)) {

                    Class<? extends AgrosystService> serviceClass =
                            (Class<? extends AgrosystService>) propertyType;

                    toInject = serviceFactory.newService(serviceClass);

                } else if (AgrosystWebSession.class.isAssignableFrom(propertyType)) {

                    toInject = agrosystSession;

                } else if (AgrosystWebConfig.class.isAssignableFrom(propertyType)) {

                    toInject = getAgrosystApplicationContext(invocation).getWebConfig();

                } else if (AgrosystServiceConfig.class.isAssignableFrom(propertyType)) {

                    toInject = getAgrosystApplicationContext(invocation).getServiceConfig();

                } else if (AgrosystWebNotificationSupport.class.isAssignableFrom(propertyType)) {
                    toInject = new AgrosystWebNotificationSupport(agrosystSession);
                }

                if (toInject != null) {

                    if (log.isTraceEnabled()) {
                        log.trace("injecting " + toInject + " in action " + action);
                    }

                    propertyDescriptor.getWriteMethod().invoke(action, toInject);

                }
            }

            return invocation.invoke();

        } else {

            // not an action, just process
            return invocation.invoke();

        }

    }

    protected AgrosystWebSession getAgrosystSession(ActionInvocation invocation) {

        AgrosystWebSession result = (AgrosystWebSession) invocation.getInvocationContext().getSession().get(AgrosystWebSession.SESSION_PARAMETER);

        if (result == null) {
            result = new AgrosystWebSession();
            invocation.getInvocationContext().getSession().put(AgrosystWebSession.SESSION_PARAMETER, result);
        }

        return result;

    }

    protected AgrosystWebApplicationContext getAgrosystApplicationContext(ActionInvocation invocation) {

        AgrosystWebApplicationContext applicationContext =
                (AgrosystWebApplicationContext) invocation
                        .getInvocationContext()
                        .getApplication()
                        .get(AgrosystWebApplicationContext.APPLICATION_CONTEXT_PARAMETER);

        Preconditions.checkState(applicationContext != null, AgrosystWebTransactionFilter.MISSING_APPLICATION_CONTEXT);

        return applicationContext;

    }

    protected ServiceFactory newServiceFactory(ActionInvocation invocation) {

        final TopiaContext topiaContext = (TopiaContext)
                ((HttpServletRequest) invocation
                        .getInvocationContext()
                        .get(StrutsStatics.HTTP_REQUEST))
                        .getAttribute(TopiaTransactionFilter.TOPIA_TRANSACTION_REQUEST_ATTRIBUTE);

        final AgrosystWebApplicationContext applicationContext = getAgrosystApplicationContext(invocation);

        @Deprecated // TODO AThimel 13/06/13 Dans l'optique d'un mode distant, ça n'a rien à faire ici
        DefaultServiceContext serviceContext = new DefaultServiceContext(applicationContext.getServiceConfig(), new Supplier<TopiaContext>() {
            @Override
            public TopiaContext get() {
                return topiaContext;
            }
        });

        ServiceFactory result = serviceContext.getServiceFactory();
        return result;

    }

    @Override
    public void destroy() {
        if (log.isInfoEnabled()) {
            log.info("Destroy Interceptor");
        }
    }

}
