package fr.inra.agrosyst.web.actions.growingsystems;

import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.apache.struts2.convention.annotation.Action;
import org.apache.struts2.convention.annotation.Result;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.opensymphony.xwork2.Preparable;

import fr.inra.agrosyst.api.NavigationContext;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.Entities;
import fr.inra.agrosyst.api.entities.GrowingPlan;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.GrowingSystemImpl;
import fr.inra.agrosyst.api.entities.Plot;
import fr.inra.agrosyst.api.exceptions.AgrosystTechnicalException;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.domain.DomainFilter;
import fr.inra.agrosyst.api.services.growingPlan.GrowingPlanFilter;
import fr.inra.agrosyst.api.services.growingPlan.GrowingPlanService;
import fr.inra.agrosyst.api.services.growingSystem.GrowingSystemService;
import fr.inra.agrosyst.api.services.plot.PlotFilter;
import fr.inra.agrosyst.api.services.plot.PlotService;
import fr.inra.agrosyst.web.actions.AbstractAgrosystAction;

/*
 * #%L
 * Agrosyst :: Web
 * $Id: GrowingSystemsEdit.java 685 2013-07-23 16:27:57Z dcosse $
 * $HeadURL: https://forge.codelutin.com/svn/agrosyst/tags/agrosyst-0.2/agrosyst-web/src/main/java/fr/inra/agrosyst/web/actions/growingsystems/GrowingSystemsEdit.java $
 * %%
 * Copyright (C) 2013 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

public class GrowingSystemsEdit extends AbstractAgrosystAction implements Preparable {

    private static final long serialVersionUID = -7030514291314044857L;

    protected String growingSystemTopiaId;

    protected String domainId;

    protected GrowingSystem growingSystem;

    protected GrowingSystemService growingSystemService;

    public void setGrowingSystemService(GrowingSystemService growingSystemService) {
        this.growingSystemService = growingSystemService;
    }
    
    protected PlotService plotService;
    
    protected Set<Plot> plots;
    
    public void setPlotService(PlotService plotService) {
        this.plotService = plotService;
    }
    
    protected GrowingPlanService growingPlanService;

    public void setGrowingPlanService(GrowingPlanService growingPlanService) {
        this.growingPlanService = growingPlanService;
    }

    protected Set<GrowingPlan> growingPlans;

    protected String growingPlanTopiaId;
    
    protected List<String> selectedPlotsIds;

    protected static final String REQUIRED_FIELD = "Champ obligatoire";
    protected static final String PERCENT_FIELD = "La valeur doit être comprise entre 0 et 100";
    protected static final String INCOHERENT_STARTING_AND_ENDING_DATES = "Les dates de début et de fin sont incohérentes";

    @Override
    public void prepare() throws Exception {


        if (StringUtils.isEmpty(growingSystemTopiaId)) {
            // Cas de création d'un growingSystem
            growingSystem = growingSystemService.newGrowingSystem();
        } else {
            // Cas d'une mise à jour de growingSystem
            growingSystem = growingSystemService.getGrowingSystem(growingSystemTopiaId);
            growingPlanTopiaId = growingSystem.getGrowingPlan().getTopiaId();
            Domain domain = growingSystem.getGrowingPlan().getDomain();
            
            NavigationContext navigationContext = getNavigationContext();
            PlotFilter filter = new PlotFilter();
            //filter.setActive(true);//TODO valider
            filter.setDomainTopiaId(domain.getTopiaId());
            filter.setNavigationContext(navigationContext);
            filter.setPageSize(DomainFilter.ALL_PAGE_SIZE);
            plots =  Sets.newLinkedHashSet(plotService.getFilteredPlots(filter));
        }
    }

    public GrowingSystem getGrowingSystem() {
        if (growingSystem == null) {
            // AThimel 26/06/2013 Fais chier de devoir écrire ça, mais c'est la seule option pour ne pas avoir une grosse dose d'exceptions avec du paramsPrepareParams
            GrowingSystem growingSystem = new GrowingSystemImpl();
            return growingSystem;
        }
        return growingSystem;
    }

    @Override
    public String input() throws Exception {
        // warning, growingSystem's must always be part of growingPlan set
        // even not selected be navigation context
        GrowingSystem growingSystem = getGrowingSystem();
        if (growingSystem.getTopiaId() != null) {
            growingPlans = Collections.singleton(growingSystem.getGrowingPlan());
        } else {
            GrowingPlanFilter growingPlanFilter = new GrowingPlanFilter();
            NavigationContext navigationContext = getNavigationContext();
            growingPlanFilter.setNavigationContext(navigationContext);
            growingPlanFilter.setPageSize(GrowingPlanFilter.ALL_PAGE_SIZE);
            ResultList<GrowingPlan> growingPlanListResult = growingPlanService.getFilteredGrowingPlans(growingPlanFilter);
            growingPlans = Sets.newLinkedHashSet(growingPlanListResult.getElements());
            
            // plots != null dans le cas ou un domain a été selectionné
            if (plots != null) {
                Map<String, Plot> indexedPlot = Maps.uniqueIndex(plots, Entities.GET_TOPIA_ID);
                Set<String> selectedPlotsIdsSet = indexedPlot.keySet();
                selectedPlotsIds = Lists.newArrayList(selectedPlotsIdsSet);
            }
        }
        return INPUT;
    }

    @Override
    public void validate() {
        // Valid if the growing system :
        //  is attached to a growingPlan
        if (StringUtils.isBlank(growingPlanTopiaId)) {
            addFieldError("growingPlanTopiaId", REQUIRED_FIELD);
        }
        //  has a name
        if (StringUtils.isBlank(growingSystem.getName())) {
            addFieldError("growingSystem.name", REQUIRED_FIELD);
        }
        //  has a DEPHY number
        if (StringUtils.isBlank(growingSystem.getDephyNumber())) {
            addFieldError("growingSystem.dephyNumber", REQUIRED_FIELD);
        }
        //  has startingDate
        Date startingDate = growingSystem.getStartingDate();
        if (startingDate == null) {
            addFieldError("growingSystem.startingDate", REQUIRED_FIELD);
        }
        //  has endingDate
        Date endingDate = growingSystem.getEndingDate();
        if (endingDate != null) {
            if (endingDate.before(startingDate)) {
                addFieldError("growingSystem.startingDate", INCOHERENT_STARTING_AND_ENDING_DATES);
                addFieldError("growingSystem.endingDate", INCOHERENT_STARTING_AND_ENDING_DATES);
            }
        }
        // affectedAreaRate
        Integer affectedAreaRate = growingSystem.getAffectedAreaRate();
        if (affectedAreaRate != null) {
            if (affectedAreaRate < 0 || affectedAreaRate > 100) {
                addFieldError("growingSystem.affectedAreaRate", PERCENT_FIELD);

            }
        }
        // affectedWorkForceRate
        Integer affectedWorkForceRate = growingSystem.getAffectedWorkForceRate();
        if (affectedWorkForceRate != null) {
            if (affectedWorkForceRate < 0 || affectedWorkForceRate > 100) {
                addFieldError("growingSystem.affectedWorkForceRate", PERCENT_FIELD);

            }
        }
        // domainsToolsUsageRate
        Integer domainsToolsUsageRate = growingSystem.getDomainsToolsUsageRate();
        if (domainsToolsUsageRate != null) {
            if (domainsToolsUsageRate < 0 || domainsToolsUsageRate > 100) {
                addFieldError("growingSystem.domainsToolsUsageRate", PERCENT_FIELD);

            }
        }
        try {
            input();
        } catch (Exception eee) {
            throw new AgrosystTechnicalException("Unable to prepare for input", eee);
        }
    }

    @Override
    @Action(results = {@Result(type = "redirectAction", params = {"actionName", "growing-systems-list"})})
    public String execute() throws Exception {
        if (StringUtils.isBlank(getGrowingSystem().getTopiaId())) {
            GrowingPlan growingPlan = growingPlanService.getGrowingPlan(growingPlanTopiaId);
            growingSystem.setGrowingPlan(growingPlan);
        }
        growingSystem = growingSystemService.updateGrowingSystem(growingSystem);
        
        plotService.updatePlotsGrowingSystemRelationship(growingSystem, selectedPlotsIds);

        return SUCCESS;
    }

    public String getStartingDateFormatted() {
        Date startingDate = growingSystem.getStartingDate();
        if (startingDate == null) {
            return "";
        }
        return new SimpleDateFormat("dd/MM/yyyy").format(startingDate); // TODO AThimel 27/06/13 Better way ?
    }

    public String getEndingDateFormatted() {
        String result = null;
        Date endingDate = growingSystem.getEndingDate();
        if (endingDate != null) {
            result = new SimpleDateFormat("dd/MM/yyyy").format(endingDate); // TODO AThimel 27/06/13 Better way ?
        }
        return result;
    }

    public void setGrowingSystem(GrowingSystem growingSystem) {
        this.growingSystem = growingSystem;
    }
    
    public String getSelectedPlotsIdsJson() {
        String result = "{}";
        if (plots != null && !plots.isEmpty()) {
            Map<String, Boolean> plotsIds = new HashMap<String, Boolean>();
            for (Plot plot : plots) {
                String plotTopiaId = plot.getTopiaId();
                boolean alreadyInUse = plot.getGrowingSystem() != null;
                plotsIds.put(plotTopiaId, alreadyInUse);
            }
            result = getGson().toJson(plotsIds);
        }
        return result;
    }
    
    public String getGrowingSystemJson() {
        return getGson().toJson(getGrowingSystem());
    }
    
    public String getGrowingPlansJson() {
        String result = "{}";
        if (growingPlans != null) {
            result= getGson().toJson(growingPlans);
        }
        return result;
    }
    
    public String getActivePlotsJson() {
        String result = "{}";
        if (plots != null && !plots.isEmpty()) {
            Map<String, Boolean> plotsIds = new HashMap<String, Boolean>();
            for (Plot plot : plots) {
                String plotTopiaId = plot.getTopiaId();
                plotsIds.put(plotTopiaId, plot.isActive());
            }
            result = getGson().toJson(plotsIds);
        }
        return result;
    }
    
    public String getPlotsJson() {
        return getGson().toJson(plots);
    }

    public String getGrowingSystemTopiaId() {
        return growingSystemTopiaId;
    }

    public void setGrowingSystemTopiaId(String growingSystemTopiaId) {
        this.growingSystemTopiaId = growingSystemTopiaId;
    }

    public void setDomainId(String domainId) {
        this.domainId = domainId;
    }

    public String getGrowingPlanTopiaId() {
        return growingPlanTopiaId;
    }

    public void setGrowingPlanTopiaId(String growingPlanTopiaId) {
        this.growingPlanTopiaId = growingPlanTopiaId;
    }

    public Collection<GrowingPlan> getGrowingPlans() {
        return growingPlans;
    }

    public void setSelectedPlotsIds(List<String> selectedPlotsIds) {
        this.selectedPlotsIds = selectedPlotsIds;
    }

}
