package fr.inra.agrosyst.web.actions.domains;

/*
 * #%L
 * Agrosyst :: Web
 * $Id: DomainsEdit.java 565 2013-07-18 11:01:06Z athimel $
 * $HeadURL: https://forge.codelutin.com/svn/agrosyst/tags/agrosyst-0.2/agrosyst-web/src/main/java/fr/inra/agrosyst/web/actions/domains/DomainsEdit.java $
 * %%
 * Copyright (C) 2013 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.apache.struts2.convention.annotation.Action;
import org.apache.struts2.convention.annotation.Result;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Strings;
import com.google.common.collect.Collections2;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.gson.reflect.TypeToken;
import com.opensymphony.xwork2.Preparable;

import fr.inra.agrosyst.api.entities.ActivityPeriod;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.DomainImpl;
import fr.inra.agrosyst.api.entities.DomainType;
import fr.inra.agrosyst.api.entities.Entities;
import fr.inra.agrosyst.api.entities.GPSData;
import fr.inra.agrosyst.api.entities.Materiel;
import fr.inra.agrosyst.api.entities.Sol;
import fr.inra.agrosyst.api.entities.Zoning;
import fr.inra.agrosyst.api.entities.referentiels.RefLegalStatus;
import fr.inra.agrosyst.api.entities.referentiels.RefLocation;
import fr.inra.agrosyst.api.entities.referentiels.RefMateriel;
import fr.inra.agrosyst.api.entities.referentiels.RefMaterielAutomoteur;
import fr.inra.agrosyst.api.entities.referentiels.RefMaterielIrrigation;
import fr.inra.agrosyst.api.entities.referentiels.RefMaterielOutil;
import fr.inra.agrosyst.api.entities.referentiels.RefMaterielTraction;
import fr.inra.agrosyst.api.entities.referentiels.RefSolArvalis;
import fr.inra.agrosyst.api.services.domain.DomainService;
import fr.inra.agrosyst.api.services.referentiels.MaterielType;
import fr.inra.agrosyst.api.services.referentiels.ReferentielService;
import fr.inra.agrosyst.web.actions.AbstractAgrosystAction;

/**
 * Action d'édition d'un domaine.
 * 
 * @author Eric Chatellier
 */
public class DomainsEdit extends AbstractAgrosystAction implements Preparable {

    protected static final Comparator<ActivityPeriodDto> ACTIVITY_PERIOD_STARTING_DATE_ORDERING = new Comparator<ActivityPeriodDto>() {
        public int compare(ActivityPeriodDto e1, ActivityPeriodDto e2) {
            return e1.getStartingDate().compareTo(e2.getStartingDate());
        }
    };

    protected static final Predicate<ActivityPeriodDto> IS_EMPTY_PERIOD = new Predicate<ActivityPeriodDto>() {
        
        @Override
        public boolean apply(ActivityPeriodDto activityPeriod) {
            return activityPeriod.getStartingDate()== null;
        }
    };
    
    protected static final Predicate<GPSDataDto> IS_INVALID_GPS_DATA = new Predicate<GPSDataDto>() {
        
        @Override
        public boolean apply(GPSDataDto gpsDataDto) {
            return !gpsDataDto.isValid();
        }
    };

    private static final String REQUIRED_FIELD = "Champ obligatoire";

    private static final long serialVersionUID = 1L;

    protected static final Function<GPSData, GPSDataDto> GPS_DATA_TO_DTO = new Function<GPSData, GPSDataDto>() {
        @Override
        public GPSDataDto apply(GPSData input) {
            GPSDataDto result = new GPSDataDto(
                    input.getTopiaId(),
                    input.getName(),
                    input.getLongitude(),
                    input.getLatitude(),
                    input.getDescription());
            return result;
        }
    };

    protected static final Function<ActivityPeriod, ActivityPeriodDto> ACTIVITY_PERIOD_TO_DTO = new Function<ActivityPeriod, ActivityPeriodDto>() {
        @Override
        public ActivityPeriodDto apply(ActivityPeriod input) {
            ActivityPeriodDto result = new ActivityPeriodDto(
                    input.getTopiaId(),
                    input.getStartingDate(),
                    input.getEndingDate(),
                    input.getEndActivityComment()
            );
            return result;
        }
    };

    protected static final Function<Materiel, MaterielDto> MATERIEL_TO_DTO = new Function<Materiel, MaterielDto>() {
        @Override
        public MaterielDto apply(Materiel input) {
            MaterielDto dto = new MaterielDto();
            dto.setTopiaId(input.getTopiaId());
            dto.setName(input.getName());
            dto.setDescription(input.getDescription());
            dto.setMaterielETA(input.isMaterielETA());
            RefMateriel refMateriel = input.getRefMateriel();
            if (refMateriel != null) {
                if (refMateriel instanceof RefMaterielAutomoteur) {
                    dto.setMaterielType(MaterielType.AUTOMOTEUR);
                } else if (refMateriel instanceof RefMaterielIrrigation) {
                    dto.setMaterielType(MaterielType.IRRIGATION);
                } else if (refMateriel instanceof RefMaterielTraction) {
                    dto.setMaterielType(MaterielType.TRACTEUR);
                } else if (refMateriel instanceof RefMaterielOutil) {
                    dto.setMaterielType(MaterielType.OUTIL);
                }
                dto.setMaterielTopiaId(refMateriel.getTopiaId());
                dto.setTypeMateriel1(refMateriel.getTypeMateriel1());
                dto.setTypeMateriel2(refMateriel.getTypeMateriel2());
                dto.setTypeMateriel3(refMateriel.getTypeMateriel3());
                dto.setTypeMateriel4(refMateriel.getTypeMateriel4());
                dto.setUnite(refMateriel.getUnite());
                dto.setUniteParAn(refMateriel.getUniteParAn());
            } else {
                dto.setMaterielType(MaterielType.AUTRE);
            }
            return dto;
        }
    };

    protected static final Function<Sol, SolDto> SOL_TO_DTO = new Function<Sol, SolDto>() {
        @Override
        public SolDto apply(Sol input) {
            RefSolArvalis refSolArvalis = input.getRefSolArvalis();
            SolDto result = new SolDto(
                    input.getTopiaId(),
                    input.getName(),
                    input.getComment(),
                    input.getImportance()
            );
            result.setSolArvalisTopiaId(refSolArvalis.getTopiaId());
            result.setSol_nom(refSolArvalis.getSol_nom());
            result.setSol_texture(refSolArvalis.getSol_texture());
            result.setSol_calcaire(refSolArvalis.getSol_calcaire());
            result.setSol_profondeur(refSolArvalis.getSol_profondeur());
            result.setSol_hydromorphie(refSolArvalis.getSol_hydromorphie());
            result.setSol_pierrosite(refSolArvalis.getSol_pierrosite());
            result.setSol_region(refSolArvalis.getSol_region());
            result.setSol_region_code(refSolArvalis.getSol_region_code());
            return result;
        }
    };

    protected ReferentielService referentielService;

    protected DomainService domainService;

    protected Domain domain;

    protected List<Domain> relatedDomains;

    protected List<GPSDataDto> gpsDataDtos;

    protected String domainTopiaId;

    protected String commune;

    protected String legalStatusId;

    protected List<RefLegalStatus> allRefLegalStatus;

    protected RefLocation location;

    protected String departement;
    protected String petiteRegionAgricole;
    protected String petiteRegionAgricoleName;

    protected String locationTopiaId;

    protected List<ActivityPeriodDto> activityPeriodsDtos;

    protected List<MaterielDto> domainMaterielDtos;

    protected Map<MaterielType, List<String>> materielType1List;

    protected Map<Integer, String> solArvalisRegions;
    
    protected Collection<SolDto> solDtos;

    public void setReferentielService(ReferentielService referentielService) {
        this.referentielService = referentielService;
    }

    public void setDomainService(DomainService domainService) {
        this.domainService = domainService;
    }

    @Override
    public void prepare() throws Exception {

        if (StringUtils.isEmpty(domainTopiaId)) {
            // Cas de création d'un domain
            domain = domainService.newDomain();

        } else {

            // Cas d'une mise à jour de domain
            domain = domainService.getDomain(domainTopiaId);

            if (StringUtils.isBlank(locationTopiaId)) {
                // Cas ou la commune n'a pas étée modifiée
                locationTopiaId = domain.getLocation().getTopiaId();

                // Dans le cas ou il y a eu modification du choix de la commune
                location = domainService.getRefLocation(locationTopiaId);

                departement = location.getDepartement();

                petiteRegionAgricole = location.getPetiteRegionAgricoleCode();
                petiteRegionAgricoleName = location.getPetiteRegionAgricoleNom();
            }
        }
    }

    public Domain getDomain() {
        if (domain == null) {
            // AThimel 26/06/2013 Fais chier de devoir écrire ça, mais c'est la seule option pour ne pas avoir une grosse dose d'exceptions avec du paramsPrepareParams
            return new DomainImpl();
        }
        return domain;
    }

    @Override
    public void initForInput() {
        allRefLegalStatus = domainService.getAllRefLegalStatus();

        materielType1List = referentielService.getTypeMateriel1List();

        solArvalisRegions = referentielService.getSolArvalisRegions();

        if (StringUtils.isNotEmpty(domainTopiaId)) {
            relatedDomains = domainService.getRelatedDomains(domain);
        }
    }

    @Override
    public String input() throws Exception {
        initForInput();
        
        if (StringUtils.isNotEmpty(domainTopiaId)) {

            activityPeriodsDtos = Lists.transform(domain.getActivityPeriods(), ACTIVITY_PERIOD_TO_DTO);

            gpsDataDtos = Lists.transform(domain.getGpsDatas(), GPS_DATA_TO_DTO);

            domainMaterielDtos = Lists.transform(domain.getMateriels(), MATERIEL_TO_DTO);

            // convert current domain sol to dto
            solDtos = Collections2.transform(domain.getSols(), SOL_TO_DTO);
        } else {
            solDtos = Collections.EMPTY_LIST;
        }

        return INPUT;
    }

    @Override
    public void validate() {
        // campagne
        if (domain.getCampaign() == 0) {
            addFieldError("domain_campaign", REQUIRED_FIELD);
        }
        
        // nom du domain
        if (StringUtils.isBlank(domain.getName())) {
            addFieldError("domain_name", REQUIRED_FIELD);
        }

        // nom de l'interlocuteur principal
        if (StringUtils.isBlank(domain.getMainContact())) {
            addFieldError("domain_mainContact", REQUIRED_FIELD);
        }

        // commune
        if (StringUtils.isBlank(locationTopiaId)) {
            addFieldError("commune", REQUIRED_FIELD);
        }

        // validation des périodes d'activitées
        if (activityPeriodsDtos != null) {
            // retire les périodes vides
            Iterables.removeIf(activityPeriodsDtos, IS_EMPTY_PERIOD);
            // trie par ordre de début de période des périodes
            Collections.sort(activityPeriodsDtos, ACTIVITY_PERIOD_STARTING_DATE_ORDERING);
            // calcul de la validation
            //  un début de période ne doit pas se situer avant la fin de période précédente.
            Date previousEndingDate = null;
            int previousEndingDateIndex = 0;
            int lastperiodIndex = (activityPeriodsDtos.size() - 1);
            List<Integer> openPeriods = new ArrayList<Integer>();
            for (ActivityPeriodDto activityPeriodDto : activityPeriodsDtos) {
                Date actualEndingingDate = activityPeriodDto.getEndingDate();
                int startingDateIndex = activityPeriodsDtos.indexOf(activityPeriodDto);
                if (actualEndingingDate == null) {
                    openPeriods.add(startingDateIndex);
                }
                if (previousEndingDate != null) {
                    Date actualStartingDate = activityPeriodDto.getStartingDate();
                    boolean res = actualStartingDate.before(previousEndingDate);
                    if (res) {
                        addFieldError("activityPeriodsDto[" + previousEndingDateIndex + "].endingDate", "Les périodes se chevauchent !");
                        addFieldError("activityPeriodsDto[" + startingDateIndex + "].startingDate", "Les périodes se chevauchent !");
                    }
                } else {
                    if (startingDateIndex != 0 && startingDateIndex<lastperiodIndex){
                        addFieldError("activityPeriodsDto[" + previousEndingDateIndex + "].endingDate", "La période ouverte n'est pas valide !");
                    }
                }
                previousEndingDate = actualEndingingDate;
                previousEndingDateIndex = startingDateIndex;
            }
            if (openPeriods.size()>1) {
                for (Integer index : openPeriods) {
                    addFieldError("activityPeriodsDto[" + index + "].endingDate", "Les périodes se chevauchent !");
                }
            }
        }

        if (hasErrors()) {
            initForInput();
        }
    }

    @Override
    @Action(results = { @Result(type = "redirectAction", params = { "actionName", "domains-list" }) })
    public String execute() throws Exception {

        if (domain.getType().equals(DomainType.EXPE)) {
            domain.setLegalStatus(null);
        }

        propagateGpsDatasChanges();
        propagateActivitiesPeriodsChanges();
        propagateSolChanges();
        propagateMaterielChanges();

        domain = domainService.updateDomain(domain, locationTopiaId);
        return SUCCESS;
    }

    /**
     * Propagate changes on domains's GpsDatas.
     */
    protected void propagateGpsDatasChanges() {
        List<GPSData> gpsDatas = getDomain().getGpsDatas();
        if (gpsDatas == null) {
            gpsDatas = Lists.newArrayList();
            domain.setGpsDatas(gpsDatas);
        }

        Iterables.removeIf(gpsDataDtos, IS_INVALID_GPS_DATA);
        
        Map<String, GPSData> indexGpsDatas = Maps.uniqueIndex(gpsDatas, Entities.GET_TOPIA_ID);

        gpsDatas.clear(); // On vide la liste pour mieux la remplir

        if (gpsDataDtos != null) {
            for (GPSDataDto gpsDataDto : gpsDataDtos) {
                String topiaId = gpsDataDto.getTopiaId();
                GPSData gpsData;
                if (Strings.isNullOrEmpty(topiaId)) {
                    // nouvelle donnée gps
                    gpsData = domainService.newGpsData();
                } else {
                    // donnée gps déjà existante
                    gpsData = indexGpsDatas.get(topiaId);
                    Preconditions.checkState(gpsData != null, "Invalid topiaId for the given entity: " + topiaId);
                }
                propagateGpsDataChanges(gpsData, gpsDataDto);
                gpsDatas.add(gpsData);
            }
        }
    }

    /**
     * Propagate the GPSData's changes according the gpsDataDto.
     * @param entity the gpsData
     * @param dto the gpsDataDto
     */
    protected void propagateGpsDataChanges(GPSData entity, GPSDataDto dto) {
        entity.setName(dto.getName());
        entity.setLatitude(dto.getLatitude());
        entity.setLongitude(dto.getLongitude());
        entity.setDescription(dto.getDescription());
    }

    /**
     * Propagate changes on domains's ActivitiesPeriods.
     */
    protected void propagateActivitiesPeriodsChanges() {
        List<ActivityPeriod> activitiesPeriods = getDomain().getActivityPeriods();
        if (activitiesPeriods == null) {
            activitiesPeriods = Lists.newArrayList();
            domain.setActivityPeriods(activitiesPeriods);
        }

        Map<String, ActivityPeriod> indexActivitesPeriods = Maps.uniqueIndex(activitiesPeriods, Entities.GET_TOPIA_ID);

        activitiesPeriods.clear(); // On vide la liste pour mieux la remplir

        if (activityPeriodsDtos != null) {
            for (ActivityPeriodDto activityPeriodDto : activityPeriodsDtos) {
                String topiaId = activityPeriodDto.getTopiaId();
                ActivityPeriod activityPeriod;
                if (Strings.isNullOrEmpty(topiaId)) {
                    // nouvelle période d'activité
                    activityPeriod = domainService.newActivityPeriod();
                } else {
                    // période d'activité déjà existante
                    activityPeriod = indexActivitesPeriods.get(topiaId);
                }
                propagateActivityPeriodsChanges(activityPeriod, activityPeriodDto);
                activitiesPeriods.add(activityPeriod);
            }
        }
    }

    /**
     * Propagate the ActivityPeriod's changes according the ActivityPeriodDto.
     * @param entity the ActivityPeriod
     * @param dto the ActivityPeriodDto
     * @return the modified ActivityPeriod
     */
    protected ActivityPeriod propagateActivityPeriodsChanges(ActivityPeriod entity, ActivityPeriodDto dto) {
        entity.setStartingDate(dto.getStartingDate());
        entity.setEndingDate(dto.getEndingDate());
        entity.setEndActivityComment(dto.getEndActivityComment());
        return entity;
    }

    /**
     * Convert submited DTO list to up to date entity list.
     * 
     * @return current entities list
     * @throws Exception 
     */
    protected void propagateMaterielChanges() throws Exception {

        List<Materiel> domainMateriels = getDomain().getMateriels();
        List<Materiel> nonDeleted = Lists.newArrayList();
        if (domainMateriels == null) {
            domainMateriels = Lists.newArrayList();
            getDomain().setMateriels(domainMateriels);
        }

        // update list with dto
        Map<String, Materiel> domainMaterielMap = Maps.uniqueIndex(domainMateriels, Entities.GET_TOPIA_ID);

        if (domainMaterielDtos != null) {
            for (MaterielDto domainMaterielDto : domainMaterielDtos) {
                // find  entity instance
                Materiel domainMateriel = null;
                String topiaId = domainMaterielDto.getTopiaId();
                if (StringUtils.isNotBlank(topiaId)) {
                    domainMateriel = domainMaterielMap.get(topiaId);
                } else {
                    domainMateriel = domainService.newMateriel();
                }

                // overwrite entity instance
                domainMateriel.setName(domainMaterielDto.getName());
                domainMateriel.setDescription(domainMaterielDto.getDescription());
                domainMateriel.setMaterielETA(domainMaterielDto.isMaterielETA());

                // manual materiel management
                String materielTopiaId = domainMaterielDto.getMaterielTopiaId();
                if (StringUtils.isNotBlank(materielTopiaId)) {
                    RefMateriel refMateriel = referentielService.findMateriel(domainMaterielDto.getMaterielTopiaId());
                    domainMateriel.setRefMateriel(refMateriel);
                } else {
                    domainMateriel.setRefMateriel(null);
                }

                if (StringUtils.isBlank(topiaId)) {
                    domainMateriels.add(domainMateriel);
                }

                nonDeleted.add(domainMateriel);
            }
        }

        domainMateriels.retainAll(nonDeleted);
    }

    protected void propagateSolChanges() {
        List<Sol> domainSols = getDomain().getSols();
        List<Sol> nonDeleted = Lists.newArrayList();
        if (domainSols == null) {
            domainSols = Lists.newArrayList();
            getDomain().setSols(domainSols);
        }

        // update list with dto
        Map<String, Sol> domainSolMap = Maps.uniqueIndex(domainSols, Entities.GET_TOPIA_ID);

        if (solDtos != null) {
            for (SolDto solDto : solDtos) {
                String topiaId = solDto.getTopiaId();
                Sol sol;
                if (Strings.isNullOrEmpty(topiaId)) {
                    // nouvelle période d'activité
                    sol = domainService.newSol();
                } else {
                    // période d'activité déjà existante
                    sol = domainSolMap.get(topiaId);
                }

                // propagate changes
                sol.setName(solDto.getName());
                sol.setComment(solDto.getComment());
                RefSolArvalis refSolArvalis = referentielService.findSolArvalis(solDto.getSolArvalisTopiaId());
                sol.setRefSolArvalis(refSolArvalis);

                // can't do this before findSolArvalis, otherwize hibernate complains !!!
                if (Strings.isNullOrEmpty(topiaId)) {
                    domainSols.add(sol);
                }
                nonDeleted.add(sol);
            }
        }

        domainSols.retainAll(nonDeleted);
    }

    /**
     * Get all the type that a a domain could be.
     * 
     * @return all the type
     */
    public DomainType[] getTypes() {
        return DomainType.values();
    }

    /**
     * Get all the VulnerableArea that a a domain could be.
     * @return all the VulnerableArea
     */
    public Zoning[] getZoningValues() {
        return Zoning.values();
    }

    protected RefLocation getLocation() {
        return location;
    }

    public void setLocationTopiaId(String locationTopiaId) {
        this.locationTopiaId = locationTopiaId;
    }

    public String getDepartement() {
        return departement;
    }
    
    public String getFormatedDepartement() {
        if (StringUtils.isBlank(departement)) {
            return "";
        }
        String depNumber = Strings.padStart(departement, 2, '0');
        String key = "departement." + depNumber;
        String result = getText(key);
        result = result + " (" + depNumber +")";
        return result;
    }
    
    public String getFormatedDepartementJson() {
        return getGson().toJson(getFormatedDepartement());
    }
    
    public String getPetiteRegionAgricole() {
        return petiteRegionAgricole;
    }

    public String getPetiteRegionAgricoleName() {
        return petiteRegionAgricoleName;
    }
    
    public String getFormatedPetiteRegionAgricoleName() {
        if (StringUtils.isBlank(petiteRegionAgricoleName)) {
            return "";
        }
        String result = petiteRegionAgricoleName + " (" + petiteRegionAgricole + ")";
        return result;
    }
    
    public String getFormatedPetiteRegionAgricoleNameJson() {
        return getGson().toJson(getFormatedPetiteRegionAgricoleName());
    }

    public String getGpsDataDtosJson() {
        return getGson().toJson(gpsDataDtos);
    }

    public void setGpsDataDtosJson(String gpsDataDtos) {
        Type type = new TypeToken<List<GPSDataDto>>() {}.getType();
        this.gpsDataDtos = getGson().fromJson(gpsDataDtos, type);
    }

    public void setDomainTopiaId(String domainTopiaId) {
        this.domainTopiaId = domainTopiaId;
    }

    public String getCommune() {
        return commune;
    }

    public void setCommune(String commune) {
        this.commune = commune;
    }

    public String getLegalStatusId() {
        return legalStatusId;
    }

    public void setLegalStatusId(String legalStatusId) {
        this.legalStatusId = legalStatusId;
    }

    public List<RefLegalStatus> getAllRefLegalStatus() {
        return allRefLegalStatus;
    }

    public String getActivityPeriodsDtosJson() {
        String result = getGson().toJson(activityPeriodsDtos);
        return result;
    }

    public void setActivityPeriodsDtosJson(String activityPeriodsDtos) {
        Type type = new TypeToken<List<ActivityPeriodDto>>() {}.getType();
        this.activityPeriodsDtos = getGson().fromJson(activityPeriodsDtos, type);
    }

    public String getLocationTopiaId() {
        return locationTopiaId;
    }

    public String getDomainMaterielDtosJson() {
        return getGson().toJson(domainMaterielDtos);
    }

    public void setDomainMaterielDtosJson(String json) {
        Type type = new TypeToken<List<MaterielDto>>() {}.getType();
        this.domainMaterielDtos = getGson().fromJson(json, type);
    }

    public String getMaterielType1ListJson() {
        return getGson().toJson(materielType1List);
    }

    public String getSolDtosJson() {
        return getGson().toJson(solDtos);
    }
    
    public void setSolDtosJson(String json) {
        Type type = new TypeToken<List<SolDto>>() {}.getType();
        this.solDtos = getGson().fromJson(json, type);
    }

    public Map<Integer, String> getSolArvalisRegions() {
        return solArvalisRegions;
    }

    public List<Domain> getRelatedDomains() {
        return relatedDomains;
    }
}
