package fr.inra.agrosyst.web.actions.performances;

/*
 * #%L
 * Agrosyst :: Web
 * $Id: PerformancesEdit.java 3427 2014-02-24 10:15:49Z echatellier $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-0.11/agrosyst-web/src/main/java/fr/inra/agrosyst/web/actions/performances/PerformancesEdit.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.List;
import java.util.Set;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.struts2.convention.annotation.Action;
import org.apache.struts2.convention.annotation.Result;
import org.nuiton.topia.persistence.TopiaEntities;

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.opensymphony.xwork2.Preparable;

import fr.inra.agrosyst.api.NavigationContext;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.Plot;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.entities.performance.Performance;
import fr.inra.agrosyst.api.entities.performance.PerformanceImpl;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.domain.DomainFilter;
import fr.inra.agrosyst.api.services.domain.DomainService;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemFilter;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemService;
import fr.inra.agrosyst.api.services.performance.PerformanceService;
import fr.inra.agrosyst.web.actions.AbstractAgrosystAction;

/**
 * Edition d'une performance.
 * 
 * @author Eric Chatellier
 */
public class PerformancesEdit extends AbstractAgrosystAction implements Preparable {

    /** serialVersionUID. */
    private static final long serialVersionUID = 525633506544808582L;

    protected transient PerformanceService performanceService;

    protected transient GrowingSystemService growingSystemService;

    protected transient DomainService domainService;

    /** Performance en cours d'edition. */
    protected Performance performance;

    /** Id de la performance a editer. */
    protected String performanceTopiaId;

    /** Liste des domaines sélectionnable. */
    protected Set<Domain> domains;

    /** Liste des systèmes de culture sélectionnables. */
    protected Set<GrowingSystem> growingSystems;

    /** Liste des parcelles sélectionnable. */
    protected Set<Plot> plots;

    /** Liste des zones sélectionnable. */
    protected Set<Zone> zones;

    /** Liste des id de domaines sélectionés. */
    protected List<String> domainIds;

    /** Liste des id de système de culture sélectionnés. */
    protected List<String> growingSystemIds;

    /** Liste des id de parcelle sélectionnées. */
    protected List<String> plotIds;

    /** Liste des id de zones sélectionnées. */
    protected List<String> zoneIds;

    /** Flag used to use interface for practiced or not. */
    protected boolean practiced;

    public void setPerformanceService(PerformanceService performanceService) {
        this.performanceService = performanceService;
    }

    public void setGrowingSystemService(GrowingSystemService growingSystemService) {
        this.growingSystemService = growingSystemService;
    }

    public void setDomainService(DomainService domainService) {
        this.domainService = domainService;
    }

    public void setDomainIds(List<String> domainIds) {
        this.domainIds = domainIds;
    }

    public List<String> getDomainIds() {
        return domainIds;
    }

    public void setPerformanceTopiaId(String performanceTopiaId) {
        this.performanceTopiaId = performanceTopiaId;
    }

    public String getPerformanceTopiaId() {
        return performanceTopiaId;
    }

    public List<String> getGrowingSystemIds() {
        return growingSystemIds;
    }

    public void setGrowingSystemIds(List<String> growingSystemIds) {
        this.growingSystemIds = growingSystemIds;
    }

    public List<String> getPlotIds() {
        return plotIds;
    }

    public void setPlotIds(List<String> plotIds) {
        this.plotIds = plotIds;
    }

    public List<String> getZoneIds() {
        return zoneIds;
    }

    public void setZoneIds(List<String> zoneIds) {
        this.zoneIds = zoneIds;
    }

    public boolean isPracticed() {
        return practiced;
    }

    public void setPracticed(boolean practiced) {
        this.practiced = practiced;
    }

    public Performance getPerformance() {
        if (performance == null) {
            // EChatellier 27/06/2013 Fais chier de devoir écrire ça, mais c'est la seule option pour ne pas avoir une grosse dose d'exceptions avec du paramsPrepareParams
            return new PerformanceImpl();
        }
        return performance;
    }

    @Override
    public void prepare() throws Exception {
        performance = performanceService.getPerformance(performanceTopiaId);

        if (performance.isPersisted()) {
            // Cas d'une mise à jour de performance
            readOnly = !authorizationService.isPerformanceWritable(performanceTopiaId);
            if (readOnly) {
                notificationSupport.performanceNotWritable();
            }
        }
    }

    @Override
    @Action("performances-edit-input")
    public String input() throws Exception {

        // convert selected instance to id collections
        domainIds = Lists.newArrayList(Iterables.transform(CollectionUtils.emptyIfNull(performance.getDomains()), TopiaEntities.getTopiaIdFunction()));
        growingSystemIds = Lists.newArrayList(Iterables.transform(CollectionUtils.emptyIfNull(performance.getGrowingSystems()), TopiaEntities.getTopiaIdFunction()));
        plotIds = Lists.newArrayList(Iterables.transform(CollectionUtils.emptyIfNull(performance.getPlots()), TopiaEntities.getTopiaIdFunction()));
        zoneIds = Lists.newArrayList(Iterables.transform(CollectionUtils.emptyIfNull(performance.getZones()), TopiaEntities.getTopiaIdFunction()));

        initForInput();

        return INPUT;
    }

    @Override
    protected void initForInput() {

        // get domain list from navigation context
        DomainFilter domainFilter = new DomainFilter();
        NavigationContext navigationContext = getNavigationContext();
        domainFilter.setNavigationContext(navigationContext);
        domainFilter.setActive(true);
        domainFilter.setAllPageSize();
        ResultList<Domain> domainListResult = domainService.getFilteredDomains(domainFilter);
        domains = Sets.newLinkedHashSet(domainListResult.getElements());
        // make sur selected domains are present for edition
        domains.addAll(CollectionUtils.emptyIfNull(performance.getDomains()));

        if (CollectionUtils.isNotEmpty(domainIds)) {
            // create new navigation context to filter others lists
            GrowingSystemFilter growingSystemFilter = new GrowingSystemFilter();
            NavigationContext navigationContext2 = new NavigationContext(navigationContext);
            navigationContext2.setDomains(Sets.newHashSet(domainIds));
            growingSystemFilter.setNavigationContext(navigationContext2);
            growingSystems = Sets.newLinkedHashSet(growingSystemService.getFilteredGrowingSystems(growingSystemFilter));
            // make sur selected growing systems are present for edition
            growingSystems.addAll(CollectionUtils.emptyIfNull(performance.getGrowingSystems()));
        }
        if (CollectionUtils.isNotEmpty(domainIds)) {
            plots = Sets.newLinkedHashSet(performanceService.getPlots(domainIds));
            // make sur selected plots are present for edition
            plots.addAll(CollectionUtils.emptyIfNull(performance.getPlots()));
        }
        if (CollectionUtils.isNotEmpty(plotIds)) {
            zones = Sets.newLinkedHashSet(performanceService.getZones(plotIds));
            // make sur selected zones are present for edition
            zones.addAll(CollectionUtils.emptyIfNull(performance.getZones()));
        }
    }

    @Override
    public void validate() {
        if (StringUtils.isBlank(performance.getName())) {
            addFieldError("performance.name", "Le nom est obligatoire !");
        }
        if (CollectionUtils.isEmpty(domainIds)) {
            addFieldError("performance.domainIds", "Un moins un domaine est obligatoire !");
        }

        if (hasErrors()) {
            initForInput();
        }
    }

    @Override
    @Action(results = {@Result(type = "redirectAction", params = {"actionName", "performances-edit-input", "performanceTopiaId", "${performance.topiaId}", "practiced", "${practiced}"})})
    public String execute() {

        // special management for 'practiced' flag
        if (!performance.isPersisted()) {
            performance.setPracticed(practiced);
        }
        performance = performanceService.createOrUpdatePerformance(performance, domainIds, growingSystemIds, plotIds, zoneIds);
        notificationSupport.performanceSaved(performance);

        return SUCCESS;
    }

    public Set<Domain> getDomains() {
        return domains;
    }

    public Set<GrowingSystem> getGrowingSystems() {
        return growingSystems;
    }

    public Set<Plot> getPlots() {
        return plots;
    }

    public Set<Zone> getZones() {
        return zones;
    }
}
