package fr.inra.agrosyst.web.actions.effective;

/*
 * #%L
 * Agrosyst :: Web
 * $Id: EffectiveCropCyclesEdit.java 3214 2014-02-12 22:56:47Z athimel $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-0.11/agrosyst-web/src/main/java/fr/inra/agrosyst/web/actions/effective/EffectiveCropCyclesEdit.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.gson.reflect.TypeToken;
import com.opensymphony.xwork2.Preparable;

import fr.inra.agrosyst.api.entities.AgrosystInterventionType;
import fr.inra.agrosyst.api.entities.CropCyclePhaseType;
import fr.inra.agrosyst.api.entities.CroppingPlanEntry;
import fr.inra.agrosyst.api.entities.OrchardFrutalForm;
import fr.inra.agrosyst.api.entities.PollinatorSpreadMode;
import fr.inra.agrosyst.api.entities.Price;
import fr.inra.agrosyst.api.entities.ToolsCoupling;
import fr.inra.agrosyst.api.entities.VineFrutalForm;
import fr.inra.agrosyst.api.entities.WeedType;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.entities.referential.RefOrientationEDI;
import fr.inra.agrosyst.api.services.domain.DomainService;
import fr.inra.agrosyst.api.services.effective.EffectiveCropCycleConnectionDto;
import fr.inra.agrosyst.api.services.effective.EffectiveCropCycleModelDto;
import fr.inra.agrosyst.api.services.effective.EffectiveCropCycleNodeDto;
import fr.inra.agrosyst.api.services.effective.EffectiveCropCyclePhaseDto;
import fr.inra.agrosyst.api.services.effective.EffectiveCropCycleService;
import fr.inra.agrosyst.api.services.effective.EffectiveCropCycles;
import fr.inra.agrosyst.api.services.effective.EffectiveInterventionDto;
import fr.inra.agrosyst.api.services.effective.EffectivePerennialCropCycleDto;
import fr.inra.agrosyst.api.services.effective.EffectiveSeasonalCropCycleDto;
import fr.inra.agrosyst.api.services.plot.PlotService;
import fr.inra.agrosyst.web.actions.itk.AbstractItkAction;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts2.convention.annotation.Action;
import org.apache.struts2.convention.annotation.Result;

import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class EffectiveCropCyclesEdit extends AbstractItkAction implements Preparable {

    private static final Log log = LogFactory.getLog(EffectiveCropCyclesEdit.class);

    /**
     * serialVersionUID.
     */
    private static final long serialVersionUID = -1526208592874558224L;

    protected transient EffectiveCropCycleService effectiveCropCycleService;

    protected transient PlotService plotService;

    protected transient DomainService domainService;

    /**
     * Topia id de la zone à laquelle sont liés les cycles.
     */
    protected String zoneTopiaId;

    /**
     * Zone à laquelle sont liés les cycles.
     */
    protected Zone zone;

    /**
     * Cycles assolés liés à la parcelle.
     */
    protected List<EffectiveSeasonalCropCycleDto> effectiveSeasonalCropCycles;

    /**
     * Cycles pérennes lié à la parcelle.
     */
    protected List<EffectivePerennialCropCycleDto> effectivePerennialCropCycles;

    /**
     * Cultures associées à la zone.
     */
    protected List<CroppingPlanEntry> croppingPlanEntries;

    /**
     * Main cropping plan model entries.
     */
    protected List<EffectiveCropCycleModelDto> effectiveCropCyclesMainModels;

    /**
     * Intermediate cropping plan model entries.
     */
    protected List<EffectiveCropCycleModelDto> effectiveCropCyclesIntermediateModels;

    /**
     * Orientation EDI.
     */
    protected List<RefOrientationEDI> refOrientationEDIs;

    protected List<ToolsCoupling> toolsCouplings;

    /**
     * Liste des prix liés à ces interventions culturales
     */
    protected List<Price> prices;

    protected String domainId;

    /** Zone related to current zone for navigation. */
    protected LinkedHashMap<Integer, String> relatedZones;

    public void setDomainService(DomainService domainService) {
        this.domainService = domainService;
    }

    public void setPlotService(PlotService plotService) {
        this.plotService = plotService;
    }

    public void setEffectiveCropCycleService(EffectiveCropCycleService effectiveCropCycleService) {
        this.effectiveCropCycleService = effectiveCropCycleService;
    }

    @Override
    public void prepare() throws Exception {
        zone = effectiveCropCycleService.getZone(zoneTopiaId);
    }

    @Override
    @Action("effective-crop-cycles-edit-input")
    public String input() throws Exception {

        domainId = zone.getPlot().getDomain().getTopiaId();

        authorizationService.checkEffectiveCropCyclesReadable(zoneTopiaId);
        readOnly = !authorizationService.areEffectiveCropCyclesWritable(zoneTopiaId);
        if (readOnly) {
            notificationSupport.effectiveCropCyclesNotWritable();
        }

        effectivePerennialCropCycles = effectiveCropCycleService.getAllEffectivePerennialCropCycles(zoneTopiaId);

        effectiveSeasonalCropCycles = effectiveCropCycleService.getAllEffectiveSeasonalCropCycles(zoneTopiaId);

        // add before node for all seasonnal crop cycles
        CroppingPlanEntry beforeNodeCroppingPlanEntry = effectiveCropCycleService.getPreviousCampaignCroppingPlanEntry(zoneTopiaId);
        if (beforeNodeCroppingPlanEntry != null) {
            EffectiveCropCycleNodeDto previousNodeCroppingPlanEntry = new EffectiveCropCycleNodeDto();
            previousNodeCroppingPlanEntry.setLabel(beforeNodeCroppingPlanEntry.getName());
            previousNodeCroppingPlanEntry.setNodeId(EffectiveCropCycleNodeDto.NODE_BEFORE_ID);
            previousNodeCroppingPlanEntry.setType(EffectiveCropCycleNodeDto.NODE_BEFORE_ID);
            
            // fix no yet cycle defined
            if (CollectionUtils.isEmpty(effectiveSeasonalCropCycles)) {
                EffectiveSeasonalCropCycleDto cropCycle = new EffectiveSeasonalCropCycleDto();
                cropCycle.setNodeDtos(new ArrayList<EffectiveCropCycleNodeDto>());
                cropCycle.getNodeDtos().add(previousNodeCroppingPlanEntry);
                cropCycle.setConnectionDtos(new ArrayList<EffectiveCropCycleConnectionDto>());
                effectiveSeasonalCropCycles = Lists.newArrayList(cropCycle);
            } else {
                // prepend node into cycles
                for (EffectiveSeasonalCropCycleDto nodedto : effectiveSeasonalCropCycles) {
                    nodedto.getNodeDtos().add(0, previousNodeCroppingPlanEntry);
                    // link "null" source connection to BEFORE NODE (automatic)
                    for (EffectiveCropCycleConnectionDto conndto : nodedto.getConnectionDtos()) {
                        if (conndto.getSourceId() == null) {
                            conndto.setSourceId(previousNodeCroppingPlanEntry.getNodeId());
                        }
                    }
                }
            }
        }

        prices = effectiveCropCycleService.getEffectivePrices(zoneTopiaId);

        initForInput();
        return INPUT;
    }

    @Override
    protected void initForInput() {

        croppingPlanEntries = effectiveCropCycleService.getZoneCroppingPlanEntries(zone);

        // build main and intermediate cropping plan entry map
        effectiveCropCyclesMainModels = Lists.newArrayList();
        effectiveCropCyclesIntermediateModels = Lists.newArrayList();
        for (CroppingPlanEntry croppingPlanEntry : croppingPlanEntries) {
            EffectiveCropCycleModelDto effectiveCropCycleModelDto = EffectiveCropCycles.CROPPING_PLAN_ENTRY_TO_DTO.apply(croppingPlanEntry);
            if (effectiveCropCycleModelDto.isIntermediate()) {
                effectiveCropCyclesIntermediateModels.add(effectiveCropCycleModelDto);
            } else {
                effectiveCropCyclesMainModels.add(effectiveCropCycleModelDto);
            }
        }

        refOrientationEDIs = referentialService.findAllReferentielEDI();
        toolsCouplings = domainService.getToolsCouplings(zone.getPlot().getDomain().getTopiaId());
        relatedZones = plotService.getRelatedZones(zone.getCode());

        super.initForInput();
    }

    @Override
    public void validate() {
        if (zone == null) {
            addFieldError("zoneTopiaId", "Zone can't be null");
        }
        if (effectiveSeasonalCropCycles == null) {
            addActionError("Seasonal crop cycles list can't be null");
        } else {
            boolean error = false;
            Set<Integer> xAlreadyDefined = Sets.newHashSet();
            for (EffectiveSeasonalCropCycleDto effectiveSeasonalCropCycleDto : effectiveSeasonalCropCycles) {
                List<EffectiveCropCycleNodeDto> nodeDtos = effectiveSeasonalCropCycleDto.getNodeDtos();
                for (EffectiveCropCycleNodeDto nodeDto : nodeDtos) {
                    
                    // ignore NODE_BEFORE
                    if (EffectiveCropCycleNodeDto.NODE_BEFORE_ID.equals(nodeDto.getNodeId())) {
                        continue;
                    }

                    if (xAlreadyDefined.contains(nodeDto.getX())) {
                        addActionError("Un cycle de culture(s) assolée(s) du réalisé doit être linéaire, soit une culture par rang.");
                        error = true;
                        break;
                    }
                    xAlreadyDefined.add(nodeDto.getX());

                    for (EffectiveInterventionDto intervention : nodeDto.getInterventions()) {
                        if (Strings.isNullOrEmpty(intervention.getName())) {
                            addActionError("Le nom d'une intervention est obligatoire");
                        }
                    }
                }
                List<EffectiveCropCycleConnectionDto> effectiveCropCycleConnectionDtos = effectiveSeasonalCropCycleDto.getConnectionDtos();
                Set<String> sources = Sets.newHashSet();
                Set<String> targets = Sets.newHashSet();
                for (EffectiveCropCycleConnectionDto connection : effectiveCropCycleConnectionDtos) {
                    String sourceId = connection.getSourceId();
                    if (StringUtils.isNotBlank(sourceId)) {
                        if (sources.contains(sourceId)) {
                            addActionError("Une culture assolée du réalisé ne peut avoir qu'une seule culture précédente");
                            break;
                        } else {
                            sources.add(sourceId);
                        }
                    }
                    String targetId = connection.getTargetId();
                    if (StringUtils.isNotBlank(targetId)) {
                        if (targets.contains(targetId)) {
                            addActionError("Une culture assolée du réalisé ne peut avoir qu'une seule culture suivante");
                            break;
                        } else {
                            targets.add(targetId);
                        }
                    }
                }
                if (error) {
                    break;
                }
            }
        }
        if (effectivePerennialCropCycles == null) {
            addFieldError("effectivePerennialCropCycles", "Perennial crop cycles list can't be null");
        } else {
            for (EffectivePerennialCropCycleDto perennial : effectivePerennialCropCycles) {
                for (EffectiveCropCyclePhaseDto phase : perennial.getPhaseDtos()) {
                    for (EffectiveInterventionDto intervention : phase.getInterventions()) {
                        if (Strings.isNullOrEmpty(intervention.getName())) {
                            addActionError("Le nom d'une intervention est obligatoire");
                        }
                    }
                }
            }
        }
        if (hasErrors()) {
            initForInput();
        }
    }

    @Action(results = {@Result(type = "redirectAction", params = {"actionName", "effective-crop-cycles-edit-input", "zoneTopiaId", "${zoneTopiaId}"})})
    @Override
    public String execute() throws Exception {
        effectiveCropCycleService.udpateEffectiveCropCycles(
                zoneTopiaId,
                effectiveSeasonalCropCycles,
                effectivePerennialCropCycles,
                prices);
        notificationSupport.culturalInterventionCreated();
        return SUCCESS;
    }

    public String getZoneTopiaId() {
        return zoneTopiaId;
    }

    public void setZoneTopiaId(String plotTopiaId) {
        this.zoneTopiaId = plotTopiaId;
    }

    public Zone getZone() {
        return zone;
    }

    public void setEffectiveSeasonalCropCyclesJson(String json) {
        Type type = new TypeToken<List<EffectiveSeasonalCropCycleDto>>() {
        }.getType();
        this.effectiveSeasonalCropCycles = getGson().fromJson(json, type);
    }

    public void setEffectivePerennialCropCyclesJson(String json) {
        Type type = new TypeToken<List<EffectivePerennialCropCycleDto>>() {
        }.getType();
        this.effectivePerennialCropCycles = getGson().fromJson(json, type);
    }

    public List<EffectiveSeasonalCropCycleDto> getEffectiveSeasonalCropCycles() {
        return effectiveSeasonalCropCycles;
    }

    public List<EffectivePerennialCropCycleDto> getEffectivePerennialCropCycles() {
        return effectivePerennialCropCycles;
    }

    public List<EffectiveCropCycleModelDto> getEffectiveCropCyclesMainModels() {
        return effectiveCropCyclesMainModels;
    }

    public List<EffectiveCropCycleModelDto> getEffectiveCropCyclesIntermediateModels() {
        return effectiveCropCyclesIntermediateModels;
    }

    public List<CroppingPlanEntry> getCroppingPlanEntries() {
        return croppingPlanEntries;
    }

    public List<RefOrientationEDI> getRefOrientationEDIs() {
        return refOrientationEDIs;
    }

    public List<ToolsCoupling> getToolsCouplings() {
        return toolsCouplings;
    }

    public Map<VineFrutalForm, String> getVineFrutalForms() {
        return getEnumAsMap(VineFrutalForm.values());
    }

    public Map<OrchardFrutalForm, String> getOrchardFrutalForms() {
        return getEnumAsMap(OrchardFrutalForm.values());
    }

    public Map<PollinatorSpreadMode, String> getPollinatorSpreadModes() {
        return getEnumAsMap(PollinatorSpreadMode.values());
    }

    public Map<CropCyclePhaseType, String> getPerennialPhaseTypes() {
        return getEnumAsMap(CropCyclePhaseType.values());
    }

    public Map<WeedType, String> getWeedTypes() {
        return getEnumAsMap(WeedType.values());
    }

    public Map<AgrosystInterventionType, String> getAgrosystInterventionTypes(){
        return getEnumAsMap(AgrosystInterventionType.values());
    }

    public List<Price> getPrices() {
        return prices;
    }

    public void setPricesJson(String json) {
        Type type = new TypeToken<List<Price>>() {
        }.getType();
        this.prices = getGson().fromJson(json, type);
    }

    public String getDomainId() {
        return domainId;
    }
    
    public LinkedHashMap<Integer, String> getRelatedZones() {
        return relatedZones;
    }
}
