package fr.inra.agrosyst.web;

/*
 * #%L
 * Agrosyst :: Web
 * $Id: AgrosystWebApplicationContext.java 3325 2014-02-19 09:44:38Z echatellier $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-0.11/agrosyst-web/src/main/java/fr/inra/agrosyst/web/AgrosystWebApplicationContext.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Method;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import javax.servlet.ServletContext;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.persistence.TopiaApplicationContextCache;

import com.google.common.collect.Maps;

import fr.inra.agrosyst.api.entities.AgrosystTopiaApplicationContext;
import fr.inra.agrosyst.api.exceptions.AgrosystTechnicalException;
import fr.inra.agrosyst.services.AgrosystServiceConfig;
import fr.inra.agrosyst.services.DefaultConfigurationSupplier;

/**
 * @author Arnaud Thimel : thimel@codelutin.com
 */
public class AgrosystWebApplicationContext {

    private static final Log log = LogFactory.getLog(AgrosystWebApplicationContext.class);

    public static final String APPLICATION_CONTEXT_PARAMETER = AgrosystWebApplicationContext.class.getName();

    public static final String MISSING_APPLICATION_CONTEXT =
            AgrosystWebApplicationContext.class.getSimpleName() + " not found. You probably forgot to" +
                    " register " + AgrosystWebApplicationListener.class.getName() + " in your web.xml";

    protected AgrosystWebConfig webConfig;

    protected Object serviceConfig;

    protected AgrosystTopiaApplicationContext applicationContext;

    /** Banner id => {banner path, banner copyright}. */
    protected Map<String, String[]> bannersMap = Maps.newHashMap();

    public void init() {

        webConfig = new AgrosystWebConfig();
        webConfig.parse();

        initApplicationContext();
    }

    protected void initApplicationContext() {
        if (!webConfig.isServicesRemoteEnabled()) {
            try {
//                Class<?> defaultConfigSupplierClass = Class.forName("fr.inra.agrosyst.services.DefaultConfigurationSupplier");
//                serviceConfig = defaultConfigSupplierClass.getMethod("getDefaultConfiguration").invoke(null);
//
//                Class<?> applicationContextClass = Class.forName("fr.inra.agrosyst.api.entities.AgrosystTopiaApplicationContext");
//                serviceConfig = defaultConfigSupplierClass.getConstructor(Properties.class);
                AgrosystServiceConfig config = DefaultConfigurationSupplier.getDefaultConfiguration();
                serviceConfig = config;

                Properties contextProperties = DefaultConfigurationSupplier.getRootContextProperties(config);
                applicationContext = TopiaApplicationContextCache.getContext(contextProperties, DefaultConfigurationSupplier.CREATE_CONTEXT_FUNCTION);
                applicationContext.init();
            } catch (Exception eee) {
                throw new AgrosystTechnicalException("Un exception occurred", eee);
            }
        }
    }

    protected void initApplicationBanners(ServletContext servletContext) {
        Set<String> resources = servletContext.getResourcePaths("/img/Header-Backgrounds/");
        for (String resource : resources) {
            String extension = FilenameUtils.getExtension(resource);
            String basename = FilenameUtils.getBaseName(resource);
            if ("jpg".equals(extension)) {
                String metapath = resource.replace("." + extension, ".txt");
                InputStream metastream = null;
                try {
                    metastream = servletContext.getResourceAsStream(metapath);
                    String metacontent = IOUtils.toString(metastream);
                    
                    if (log.isDebugEnabled()) {
                        log.debug("Adding banner " + basename + "(" + metacontent + ")");
                    }

                    // make banner path context relative
                    String resourceContextPath = StringUtils.removeStart(resource, "/");
                    bannersMap.put(basename, new String[]{resourceContextPath, metacontent});
                } catch (IOException ex) {
                    throw new AgrosystTechnicalException("Can't read banner metadata", ex);
                } finally {
                    IOUtils.closeQuietly(metastream);
                }
            }
        }
        if (log.isInfoEnabled()) {
            log.info("Detecting " + bannersMap.size() + " banners");
        }
    }

    protected void checkForDemoData() {
        if (!getWebConfig().isServicesRemoteEnabled()) {
            try {
                Class<?> demoDataClass = Class.forName("fr.inra.agrosyst.services.demo.DemoDatas");
                Method method = demoDataClass.getMethod("onInit");
                method.invoke(null);
            } catch (Exception eee) {
                throw new AgrosystTechnicalException("Unable to initialize data", eee);
            }
        }
    }

    public AgrosystWebConfig getWebConfig() {
        return webConfig;
    }

    public Object getServiceConfig() {
        return serviceConfig;
    }

    public AgrosystTopiaApplicationContext getApplicationContext() {
        return applicationContext;
    }

    public Map<String, String[]> getBannersMap() {
        return bannersMap;
    }

    public void close() {
        if (applicationContext != null && !applicationContext.isClosed()) {
            applicationContext.closeContext();
        }
    }

}
